const PI: f32 = 3.141_592_653_589_793_238f32;
const PI_2: f32 = 1.570_796_326_794_896_619f32;
const PI_NEG_2: f32 = -1.570_796_326_794_896_619f32;
const CONST: f32 = 0.28087f32; // Trial and error

/// Average error of 0.00231 radians.
/// Largest error of 0.00488 radians.
/// Speedup of 20.67x over f32.atan2(y);
#[inline]
pub fn atan2(y: f32, x: f32) -> f32 {
    if x == 0f32 {
        if y > 0f32 {
            return PI_2;
        }
        if y == 0f32 {
            return 0f32;
        }
        return PI_NEG_2;
    }
    let z: f32 = y / x;
    let atan: f32;
    if z.abs() < 1f32 {
        atan = z / (z * z * CONST + 1f32);
        if x < 0f32 {
            if y < 0f32 {
                return atan - PI;
            }
            return atan + PI;
        }
        return atan;
    }
    atan = PI_2 - z / (z * z + CONST);
    if y < 0f32 {
        return atan - PI;
    }
    atan
}

const SIN_BITS: i32 = 11i32;
const SIN_MASK: i32 = !(-1i32 << SIN_BITS);
const SIN_COUNT: i32 = SIN_MASK + 1i32;
const RAD_FULL: f32 = PI * 2f32;
const DEG_FULL: f32 = 360f32;
const RAD_INDEX: f32 = (SIN_COUNT as f32) / RAD_FULL;
const DEG_INDEX: f32 = (SIN_COUNT as f32) / DEG_FULL;
const SIZE: usize = SIN_COUNT as usize;

/// Average error of 0.00060 radians.
/// Largest error of 0.00229 radians.
/// Speedup of 30x over f32.sin();
/// Input in radians, output in radians.
#[inline]
pub fn sin_rad(rad: f32) -> f32 {
    unsafe { *SIN.get_unchecked((((rad * RAD_INDEX) as i32) & SIN_MASK) as usize) }
}

/// Average error of 0.00060 radians.
/// Largest error of 0.00229 radians.
/// Speedup of 30x over f32.cos();
/// Input in radians, output in radians.
#[inline]
pub fn cos_rad(rad: f32) -> f32 {
    unsafe { *COS.get_unchecked((((rad * RAD_INDEX) as i32) & SIN_MASK) as usize) }
}

/// Average error of 0.00060 radians.
/// Largest error of 0.00229 radians.
/// Speedup of 30x over f32.sin();
/// Input in degrees, output in radians.
#[inline]
pub fn sin_deg(deg: f32) -> f32 {
    unsafe { *SIN.get_unchecked((((deg * DEG_INDEX) as i32) & SIN_MASK) as usize) }
}

/// Average error of 0.00060 radians.
/// Largest error of 0.00229 radians.
/// Speedup of 30x over f32.cos();
/// Input in degrees, output in radians.
#[inline]
pub fn cos_deg(deg: f32) -> f32 {
    unsafe { *COS.get_unchecked((((deg * DEG_INDEX) as i32) & SIN_MASK) as usize) }
}

// Tokenization is skipped for long lines
#[rustfmt::skip] #[allow(clippy::all)] static SIN: [f32; SIZE] = [0f32, 0.004601926f32, 0.0076698293f32, 0.010737659f32, 0.0138053885f32, 0.016872989f32, 0.019940428f32, 0.023007682f32, 0.026074719f32, 0.02914151f32, 0.03220803f32, 0.035274237f32, 0.03834012f32, 0.041405644f32, 0.044470772f32, 0.047535487f32, 0.05059975f32, 0.05366354f32, 0.05672682f32, 0.059789572f32, 0.06285176f32, 0.06591336f32, 0.06897433f32, 0.07203466f32, 0.075094305f32, 0.078153245f32, 0.08121145f32, 0.08426889f32, 0.087325536f32, 0.09038136f32, 0.09343634f32, 0.096490435f32, 0.099543616f32, 0.10259587f32, 0.105647154f32, 0.108697444f32, 0.11174671f32, 0.11479493f32, 0.11784206f32, 0.12088809f32, 0.12393298f32, 0.1269767f32, 0.13001922f32, 0.13306053f32, 0.13610058f32, 0.13913935f32, 0.1421768f32, 0.14521293f32, 0.14824769f32, 0.15128104f32, 0.15431298f32, 0.15734346f32, 0.16037247f32, 0.16339995f32, 0.1664259f32, 0.1694503f32, 0.17247309f32, 0.17549427f32, 0.17851378f32, 0.18153162f32, 0.18454774f32, 0.18756212f32, 0.19057477f32, 0.19358559f32, 0.19659461f32, 0.19960175f32, 0.20260705f32, 0.20561042f32, 0.20861185f32, 0.21161133f32, 0.21460882f32, 0.21760428f32, 0.2205977f32, 0.22358905f32, 0.22657827f32, 0.22956537f32, 0.23255032f32, 0.23553306f32, 0.2385136f32, 0.24149188f32, 0.2444679f32, 0.24744162f32, 0.250413f32, 0.25338206f32, 0.2563487f32, 0.25931293f32, 0.2622747f32, 0.26523405f32, 0.26819086f32, 0.27114516f32, 0.27409694f32, 0.2770461f32, 0.27999264f32, 0.28293657f32, 0.28587785f32, 0.2888164f32, 0.29175228f32, 0.2946854f32, 0.29761574f32, 0.30054325f32, 0.30346796f32, 0.3063898f32, 0.30930877f32, 0.3122248f32, 0.31513795f32, 0.3180481f32, 0.32095525f32, 0.32385936f32, 0.32676047f32, 0.32965845f32, 0.3325534f32, 0.33544517f32, 0.33833376f32, 0.34121922f32, 0.34410143f32, 0.34698042f32, 0.34985614f32, 0.35272858f32, 0.35559767f32, 0.3584634f32, 0.3613258f32, 0.3641848f32, 0.36704037f32, 0.36989245f32, 0.37274107f32, 0.37558618f32, 0.37842774f32, 0.3812658f32, 0.3841002f32, 0.38693103f32, 0.38975817f32, 0.39258167f32, 0.3954015f32, 0.39821756f32, 0.4010299f32, 0.4038385f32, 0.40664324f32, 0.40944415f32, 0.41224125f32, 0.41503444f32, 0.4178237f32, 0.4206091f32, 0.42339048f32, 0.4261679f32, 0.4289413f32, 0.43171066f32, 0.434476f32, 0.43723717f32, 0.43999428f32, 0.44274724f32, 0.44549602f32, 0.4482406f32, 0.450981f32, 0.45371714f32, 0.45644897f32, 0.45917657f32, 0.46189982f32, 0.46461868f32, 0.46733323f32, 0.47004336f32, 0.47274905f32, 0.47545028f32, 0.47814706f32, 0.4808393f32, 0.4835271f32, 0.4862103f32, 0.48888892f32, 0.49156293f32, 0.49423233f32, 0.4968971f32, 0.4995571f32, 0.50221246f32, 0.50486314f32, 0.507509f32, 0.51015013f32, 0.51278645f32, 0.5154179f32, 0.5180445f32, 0.52066624f32, 0.5232831f32, 0.52589506f32, 0.52850205f32, 0.531104f32, 0.533701f32, 0.53629297f32, 0.53887993f32, 0.54146177f32, 0.54403853f32, 0.5466102f32, 0.5491767f32, 0.55173796f32, 0.5542941f32, 0.55684507f32, 0.5593907f32, 0.56193113f32, 0.5644663f32, 0.5669961f32, 0.56952053f32, 0.5720396f32, 0.5745534f32, 0.5770617f32, 0.5795646f32, 0.582062f32, 0.58455396f32, 0.58704036f32, 0.5895213f32, 0.5919967f32, 0.5944665f32, 0.59693074f32, 0.5993893f32, 0.6018423f32, 0.60428953f32, 0.6067311f32, 0.60916704f32, 0.6115972f32, 0.6140216f32, 0.6164402f32, 0.61885303f32, 0.62126f32, 0.6236611f32, 0.6260564f32, 0.6284458f32, 0.6308293f32, 0.6332068f32, 0.63557833f32, 0.6379439f32, 0.6403035f32, 0.64265704f32, 0.6450046f32, 0.647346f32, 0.6496813f32, 0.65201056f32, 0.6543336f32, 0.65665054f32, 0.6589613f32, 0.66126585f32, 0.6635642f32, 0.66585624f32, 0.6681421f32, 0.67042154f32, 0.6726948f32, 0.6749617f32, 0.6772222f32, 0.6794763f32, 0.68172413f32, 0.68396544f32, 0.6862003f32, 0.68842876f32, 0.6906507f32, 0.6928662f32, 0.69507515f32, 0.69727755f32, 0.6994734f32, 0.7016626f32, 0.70384526f32, 0.70602125f32, 0.7081907f32, 0.7103534f32, 0.7125094f32, 0.7146587f32, 0.7168013f32, 0.71893716f32, 0.72106624f32, 0.7231885f32, 0.725304f32, 0.72741264f32, 0.7295144f32, 0.7316094f32, 0.7336975f32, 0.7357786f32, 0.7378528f32, 0.73992014f32, 0.74198043f32, 0.74403375f32, 0.7460801f32, 0.7481194f32, 0.75015163f32, 0.7521769f32, 0.754195f32, 0.75620604f32, 0.7582099f32, 0.7602067f32, 0.7621963f32, 0.76417875f32, 0.766154f32, 0.7681221f32, 0.7700829f32, 0.7720364f32, 0.7739827f32, 0.7759217f32, 0.7778534f32, 0.7797778f32, 0.7816948f32, 0.78360456f32, 0.78550684f32, 0.78740174f32, 0.78928924f32, 0.79116935f32, 0.793042f32, 0.79490715f32, 0.79676485f32, 0.798615f32, 0.80045766f32, 0.8022928f32, 0.8041204f32, 0.8059404f32, 0.80775285f32, 0.8095577f32, 0.8113549f32, 0.81314445f32, 0.8149263f32, 0.8167006f32, 0.81846714f32, 0.820226f32, 0.82197714f32, 0.8237205f32, 0.82545614f32, 0.827184f32, 0.82890415f32, 0.8306164f32, 0.83232087f32, 0.8340175f32, 0.8357063f32, 0.8373872f32, 0.83906025f32, 0.84072536f32, 0.8423826f32, 0.8440319f32, 0.84567326f32, 0.84730667f32, 0.8489321f32, 0.8505495f32, 0.8521589f32, 0.8537603f32, 0.8553537f32, 0.856939f32, 0.8585163f32, 0.8600854f32, 0.86164653f32, 0.8631994f32, 0.86474425f32, 0.866281f32, 0.8678095f32, 0.86932987f32, 0.8708421f32, 0.87234604f32, 0.8738419f32, 0.87532943f32, 0.87680876f32, 0.8782798f32, 0.8797426f32, 0.88119715f32, 0.8826434f32, 0.88408124f32, 0.88551086f32, 0.88693213f32, 0.88834506f32, 0.8897496f32, 0.89114577f32, 0.89253354f32, 0.893913f32, 0.89528394f32, 0.8966465f32, 0.8980006f32, 0.8993463f32, 0.90068346f32, 0.90201217f32, 0.90333235f32, 0.9046441f32, 0.90594727f32, 0.907242f32, 0.90852815f32, 0.9098057f32, 0.91107476f32, 0.9123352f32, 0.9135871f32, 0.9148303f32, 0.916065f32, 0.91729105f32, 0.9185084f32, 0.9197172f32, 0.9209172f32, 0.92210865f32, 0.92329144f32, 0.9244655f32, 0.9256308f32, 0.9267875f32, 0.9279354f32, 0.9290746f32, 0.93020505f32, 0.93132675f32, 0.9324396f32, 0.9335438f32, 0.93463916f32, 0.9357257f32, 0.93680346f32, 0.93787235f32, 0.9389325f32, 0.9399838f32, 0.9410262f32, 0.94205976f32, 0.9430845f32, 0.94410026f32, 0.9451072f32, 0.94610524f32, 0.9470944f32, 0.94807464f32, 0.9490459f32, 0.9500082f32, 0.95096165f32, 0.95190614f32, 0.95284164f32, 0.9537682f32, 0.95468575f32, 0.95559436f32, 0.9564939f32, 0.9573845f32, 0.9582661f32, 0.95913863f32, 0.9600022f32, 0.9608567f32, 0.9617021f32, 0.9625385f32, 0.9633658f32, 0.96418405f32, 0.96499324f32, 0.9657934f32, 0.9665844f32, 0.9673663f32, 0.9681391f32, 0.9689028f32, 0.9696574f32, 0.97040284f32, 0.9711392f32, 0.97186637f32, 0.97258437f32, 0.97329324f32, 0.97399294f32, 0.9746835f32, 0.97536486f32, 0.9760371f32, 0.97670007f32, 0.97735393f32, 0.9779985f32, 0.97863394f32, 0.97926015f32, 0.9798771f32, 0.9804849f32, 0.9810834f32, 0.9816727f32, 0.9822527f32, 0.98282355f32, 0.9833851f32, 0.98393744f32, 0.98448044f32, 0.98501426f32, 0.9855387f32, 0.98605394f32, 0.9865599f32, 0.98705655f32, 0.98754394f32, 0.988022f32, 0.9884908f32, 0.98895025f32, 0.98940045f32, 0.9898413f32, 0.9902728f32, 0.990695f32, 0.99110794f32, 0.99151146f32, 0.9919057f32, 0.9922906f32, 0.9926661f32, 0.99303234f32, 0.99338925f32, 0.99373674f32, 0.9940749f32, 0.99440366f32, 0.99472314f32, 0.9950332f32, 0.9953339f32, 0.99562526f32, 0.99590725f32, 0.9961798f32, 0.99644303f32, 0.9966969f32, 0.9969414f32, 0.99717647f32, 0.99740213f32, 0.99761844f32, 0.9978254f32, 0.99802285f32, 0.998211f32, 0.9983897f32, 0.99855906f32, 0.99871904f32, 0.99886954f32, 0.9990107f32, 0.9991424f32, 0.9992648f32, 0.99937767f32, 0.9994812f32, 0.9995753f32, 0.99966f32, 0.9997353f32, 0.99980116f32, 0.99985766f32, 0.9999047f32, 0.99994236f32, 0.9999706f32, 0.9999894f32, 0.9999988f32, 1f32, 0.9999894f32, 0.9999706f32, 0.99994236f32, 0.9999047f32, 0.99985766f32, 0.99980116f32, 0.9997353f32, 0.99966f32, 0.9995753f32, 0.9994812f32, 0.99937767f32, 0.9992647f32, 0.9991424f32, 0.9990107f32, 0.99886954f32, 0.99871904f32, 0.99855906f32, 0.9983897f32, 0.998211f32, 0.99802285f32, 0.9978253f32, 0.99761844f32, 0.99740213f32, 0.9971764f32, 0.9969413f32, 0.9966969f32, 0.99644303f32, 0.9961798f32, 0.99590725f32, 0.99562526f32, 0.9953339f32, 0.9950332f32, 0.9947231f32, 0.99440366f32, 0.9940749f32, 0.99373674f32, 0.9933892f32, 0.99303234f32, 0.9926661f32, 0.9922906f32, 0.9919057f32, 0.99151146f32, 0.9911079f32, 0.990695f32, 0.9902728f32, 0.9898413f32, 0.9894004f32, 0.98895025f32, 0.9884908f32, 0.988022f32, 0.98754394f32, 0.98705655f32, 0.9865599f32, 0.98605394f32, 0.9855387f32, 0.9850142f32, 0.98448044f32, 0.9839374f32, 0.9833851f32, 0.98282355f32, 0.9822527f32, 0.9816727f32, 0.9810834f32, 0.98048484f32, 0.9798771f32, 0.97926015f32, 0.97863394f32, 0.9779985f32, 0.9773539f32, 0.97670007f32, 0.9760371f32, 0.97536486f32, 0.97468346f32, 0.97399294f32, 0.97329324f32, 0.97258437f32, 0.9718663f32, 0.97113913f32, 0.97040284f32, 0.96965736f32, 0.96890277f32, 0.9681391f32, 0.9673663f32, 0.9665844f32, 0.9657933f32, 0.96499324f32, 0.96418405f32, 0.9633658f32, 0.9625384f32, 0.96170205f32, 0.9608566f32, 0.9600021f32, 0.95913863f32, 0.9582661f32, 0.95738447f32, 0.9564939f32, 0.9555943f32, 0.95468575f32, 0.9537682f32, 0.95284164f32, 0.9519061f32, 0.95096165f32, 0.9500082f32, 0.9490459f32, 0.9480746f32, 0.9470944f32, 0.94610524f32, 0.94510716f32, 0.94410026f32, 0.9430844f32, 0.9420597f32, 0.94102615f32, 0.9399837f32, 0.9389325f32, 0.93787235f32, 0.9368034f32, 0.9357256f32, 0.93463916f32, 0.93354374f32, 0.9324396f32, 0.9313267f32, 0.930205f32, 0.9290746f32, 0.92793536f32, 0.92678744f32, 0.9256308f32, 0.9244654f32, 0.9232914f32, 0.92210865f32, 0.9209172f32, 0.91971713f32, 0.9185084f32, 0.917291f32, 0.916065f32, 0.9148303f32, 0.91358703f32, 0.91233516f32, 0.9110747f32, 0.90980566f32, 0.90852815f32, 0.907242f32, 0.9059473f32, 0.9046441f32, 0.90333235f32, 0.90201217f32, 0.9006834f32, 0.89934623f32, 0.8980006f32, 0.89664644f32, 0.8952839f32, 0.8939129f32, 0.89253354f32, 0.8911457f32, 0.8897495f32, 0.888345f32, 0.8869321f32, 0.8855108f32, 0.8840812f32, 0.8826433f32, 0.88119704f32, 0.8797425f32, 0.87827975f32, 0.87680864f32, 0.8753293f32, 0.87384176f32, 0.8723461f32, 0.8708421f32, 0.86932987f32, 0.86780953f32, 0.866281f32, 0.86474425f32, 0.8631994f32, 0.8616465f32, 0.86008537f32, 0.8585162f32, 0.85693896f32, 0.85535365f32, 0.8537603f32, 0.85215884f32, 0.85054946f32, 0.848932f32, 0.8473066f32, 0.8456732f32, 0.8440319f32, 0.84238255f32, 0.8407253f32, 0.8390602f32, 0.83738714f32, 0.83570623f32, 0.8340174f32, 0.83232075f32, 0.8306163f32, 0.82890403f32, 0.8271841f32, 0.8254562f32, 0.8237205f32, 0.82197714f32, 0.820226f32, 0.81846714f32, 0.8167006f32, 0.8149263f32, 0.8131444f32, 0.8113548f32, 0.8095576f32, 0.8077528f32, 0.80594033f32, 0.80412036f32, 0.80229276f32, 0.8004576f32, 0.7986149f32, 0.79676473f32, 0.79490703f32, 0.7930419f32, 0.7911692f32, 0.7892892f32, 0.7874017f32, 0.7855067f32, 0.78360444f32, 0.7816947f32, 0.7797777f32, 0.7778534f32, 0.7759217f32, 0.7739827f32, 0.77203643f32, 0.77008283f32, 0.768122f32, 0.766154f32, 0.76417875f32, 0.7621963f32, 0.76020664f32, 0.7582099f32, 0.756206f32, 0.7541949f32, 0.7521768f32, 0.7501516f32, 0.74811935f32, 0.74608004f32, 0.7440337f32, 0.7419804f32, 0.73992f32, 0.73785275f32, 0.7357785f32, 0.73369735f32, 0.7316093f32, 0.72951436f32, 0.7274125f32, 0.7253038f32, 0.72318834f32, 0.72106624f32, 0.71893716f32, 0.7168013f32, 0.7146587f32, 0.7125094f32, 0.7103534f32, 0.7081906f32, 0.70602125f32, 0.7038452f32, 0.7016626f32, 0.6994733f32, 0.6972775f32, 0.6950751f32, 0.69286615f32, 0.69065064f32, 0.6884287f32, 0.68620026f32, 0.6839653f32, 0.681724f32, 0.67947626f32, 0.6772221f32, 0.67496157f32, 0.6726947f32, 0.6704214f32, 0.6681419f32, 0.6658561f32, 0.663564f32, 0.6612657f32, 0.65896136f32, 0.6566506f32, 0.65433365f32, 0.65201056f32, 0.6496813f32, 0.64734596f32, 0.6450045f32, 0.64265704f32, 0.6403035f32, 0.63794386f32, 0.6355783f32, 0.6332067f32, 0.63082916f32, 0.6284457f32, 0.6260563f32, 0.62366104f32, 0.6212599f32, 0.6188529f32, 0.61644006f32, 0.6140215f32, 0.61159706f32, 0.6091669f32, 0.606731f32, 0.6042894f32, 0.6018421f32, 0.59938914f32, 0.59693056f32, 0.59446657f32, 0.5919967f32, 0.58952135f32, 0.5870404f32, 0.58455396f32, 0.582062f32, 0.5795646f32, 0.57706165f32, 0.5745533f32, 0.5720396f32, 0.5695205f32, 0.56699604f32, 0.5644662f32, 0.5619311f32, 0.55939066f32, 0.55684495f32, 0.55429405f32, 0.5517379f32, 0.5491766f32, 0.54661006f32, 0.5440384f32, 0.54146165f32, 0.5388798f32, 0.53629285f32, 0.5337009f32, 0.53110385f32, 0.52850187f32, 0.5258949f32, 0.5232831f32, 0.5206663f32, 0.51804453f32, 0.5154179f32, 0.5127864f32, 0.5101501f32, 0.507509f32, 0.5048631f32, 0.50221246f32, 0.49955708f32, 0.496897f32, 0.49423227f32, 0.49156287f32, 0.48888883f32, 0.4862102f32, 0.483527f32, 0.48083925f32, 0.47814697f32, 0.4754502f32, 0.47274894f32, 0.4700432f32, 0.46733308f32, 0.46461856f32, 0.46189964f32, 0.4591764f32, 0.45644882f32, 0.45371696f32, 0.4509808f32, 0.44824064f32, 0.44549605f32, 0.44274724f32, 0.43999428f32, 0.43723717f32, 0.43447596f32, 0.43171066f32, 0.42894128f32, 0.42616788f32, 0.42339045f32, 0.42060903f32, 0.41782367f32, 0.41503435f32, 0.41224116f32, 0.40944406f32, 0.40664312f32, 0.40383837f32, 0.4010298f32, 0.39821744f32, 0.39540136f32, 0.39258155f32, 0.38975805f32, 0.38693085f32, 0.38410005f32, 0.3812656f32, 0.3784276f32, 0.375586f32, 0.3727411f32, 0.36989248f32, 0.36704037f32, 0.3641848f32, 0.3613258f32, 0.3584634f32, 0.35559765f32, 0.35272855f32, 0.3498561f32, 0.34698036f32, 0.34410137f32, 0.34121916f32, 0.3383337f32, 0.33544508f32, 0.3325533f32, 0.3296584f32, 0.32676035f32, 0.32385927f32, 0.32095513f32, 0.31804797f32, 0.3151378f32, 0.3122247f32, 0.30930862f32, 0.30638966f32, 0.30346778f32, 0.30054307f32, 0.29761553f32, 0.29468518f32, 0.2917523f32, 0.28881642f32, 0.28587785f32, 0.28293657f32, 0.27999264f32, 0.27704608f32, 0.2740969f32, 0.27114514f32, 0.26819083f32, 0.265234f32, 0.26227465f32, 0.25931287f32, 0.2563486f32, 0.25338197f32, 0.2504129f32, 0.24744153f32, 0.24446781f32, 0.24149178f32, 0.23851348f32, 0.23553294f32, 0.23255017f32, 0.22956522f32, 0.22657812f32, 0.22358887f32, 0.22059752f32, 0.2176041f32, 0.21460862f32, 0.21161114f32, 0.20861188f32, 0.20561044f32, 0.20260705f32, 0.19960177f32, 0.1965946f32, 0.19358557f32, 0.19057474f32, 0.1875621f32, 0.18454769f32, 0.18153156f32, 0.17851372f32, 0.1754942f32, 0.17247301f32, 0.16945021f32, 0.16642581f32, 0.16339985f32, 0.16037235f32, 0.15734334f32, 0.15431285f32, 0.15128091f32, 0.14824754f32, 0.14521277f32, 0.14217664f32, 0.13913918f32, 0.1361004f32, 0.13306034f32, 0.13001902f32, 0.12697673f32, 0.123933f32, 0.12088811f32, 0.11784207f32, 0.114794925f32, 0.111746706f32, 0.10869743f32, 0.10564713f32, 0.102595836f32, 0.09954358f32, 0.09649038f32, 0.09343628f32, 0.090381294f32, 0.08732546f32, 0.0842688f32, 0.08121135f32, 0.07815314f32, 0.075094186f32, 0.07203453f32, 0.0689742f32, 0.065913215f32, 0.06285161f32, 0.059789415f32, 0.056726657f32, 0.053663366f32, 0.050599568f32, 0.047535293f32, 0.04447057f32, 0.04140567f32, 0.03834014f32, 0.035274252f32, 0.03220803f32, 0.029141504f32, 0.026074704f32, 0.02300766f32, 0.019940397f32, 0.016872948f32, 0.01380534f32, 0.010737603f32, 0.007669763f32, 0.0046018516f32, 0.0015338971f32, -0.00000008742278f32, -0.0046020267f32, -0.007669938f32, -0.010737777f32, -0.013805515f32, -0.016873123f32, -0.019940572f32, -0.023007834f32, -0.026074879f32, -0.02914168f32, -0.032208204f32, -0.035274427f32, -0.038340315f32, -0.041405845f32, -0.044470746f32, -0.047535468f32, -0.050599743f32, -0.05366354f32, -0.056726832f32, -0.05978959f32, -0.06285179f32, -0.06591339f32, -0.068974376f32, -0.07203471f32, -0.075094365f32, -0.07815331f32, -0.08121153f32, -0.08426897f32, -0.08732563f32, -0.090381466f32, -0.09343645f32, -0.096490555f32, -0.09954375f32, -0.10259601f32, -0.1056473f32, -0.1086976f32, -0.11174688f32, -0.1147951f32, -0.11784224f32, -0.12088828f32, -0.12393317f32, -0.1269769f32, -0.1300192f32, -0.13306051f32, -0.13610058f32, -0.13913935f32, -0.14217682f32, -0.14521295f32, -0.1482477f32, -0.15128107f32, -0.15431303f32, -0.1573435f32, -0.16037253f32, -0.16340002f32, -0.16642599f32, -0.16945039f32, -0.17247318f32, -0.17549436f32, -0.17851388f32, -0.18153173f32, -0.18454787f32, -0.18756227f32, -0.1905749f32, -0.19358575f32, -0.19659476f32, -0.19960193f32, -0.20260723f32, -0.2056106f32, -0.20861205f32, -0.2116113f32, -0.2146088f32, -0.21760426f32, -0.2205977f32, -0.22358905f32, -0.22657828f32, -0.2295654f32, -0.23255034f32, -0.2355331f32, -0.23851365f32, -0.24149194f32, -0.24446797f32, -0.2474417f32, -0.2504131f32, -0.25338215f32, -0.2563488f32, -0.25931302f32, -0.26227483f32, -0.26523417f32, -0.26819098f32, -0.2711453f32, -0.27409706f32, -0.27704623f32, -0.27999282f32, -0.28293675f32, -0.28587803f32, -0.2888166f32, -0.29175246f32, -0.29468536f32, -0.2976157f32, -0.30054325f32, -0.30346796f32, -0.3063898f32, -0.3093088f32, -0.31222484f32, -0.31513798f32, -0.31804812f32, -0.32095528f32, -0.32385942f32, -0.32676053f32, -0.32965854f32, -0.33255345f32, -0.33544526f32, -0.33833387f32, -0.3412193f32, -0.34410155f32, -0.34698054f32, -0.34985626f32, -0.3527287f32, -0.35559782f32, -0.3584636f32, -0.36132598f32, -0.36418498f32, -0.36704054f32, -0.36989263f32, -0.37274128f32, -0.37558615f32, -0.37842774f32, -0.38126576f32, -0.3841002f32, -0.38693103f32, -0.3897582f32, -0.3925817f32, -0.39540154f32, -0.39821762f32, -0.40102994f32, -0.40383852f32, -0.4066433f32, -0.40944424f32, -0.4122413f32, -0.41503453f32, -0.41782382f32, -0.42060918f32, -0.4233906f32, -0.42616802f32, -0.42894143f32, -0.4317108f32, -0.4344761f32, -0.43723732f32, -0.43999445f32, -0.4427474f32, -0.4454962f32, -0.44824082f32, -0.45098096f32, -0.4537171f32, -0.45644897f32, -0.45917657f32, -0.46189982f32, -0.4646187f32, -0.46733323f32, -0.47004336f32, -0.47274908f32, -0.47545034f32, -0.47814712f32, -0.4808394f32, -0.48352715f32, -0.48621035f32, -0.48888898f32, -0.49156302f32, -0.49423242f32, -0.49689716f32, -0.49955723f32, -0.5022126f32, -0.50486326f32, -0.5075091f32, -0.51015025f32, -0.51278657f32, -0.51541805f32, -0.51804465f32, -0.5206664f32, -0.5232833f32, -0.525895f32, -0.528502f32, -0.531104f32, -0.533701f32, -0.53629297f32, -0.53887993f32, -0.5414618f32, -0.5440386f32, -0.54661024f32, -0.5491767f32, -0.551738f32, -0.55429417f32, -0.5568451f32, -0.5593908f32, -0.5619312f32, -0.56446636f32, -0.56699616f32, -0.56952065f32, -0.5720397f32, -0.5745535f32, -0.57706183f32, -0.5795647f32, -0.5820621f32, -0.5845541f32, -0.58704054f32, -0.58952147f32, -0.59199685f32, -0.5944667f32, -0.5969307f32, -0.5993893f32, -0.6018423f32, -0.60428953f32, -0.6067312f32, -0.60916704f32, -0.6115972f32, -0.6140216f32, -0.61644024f32, -0.61885303f32, -0.62126005f32, -0.62366116f32, -0.6260565f32, -0.62844586f32, -0.63082933f32, -0.63320684f32, -0.6355784f32, -0.63794404f32, -0.6403036f32, -0.64265716f32, -0.6450047f32, -0.6473461f32, -0.64968145f32, -0.6520107f32, -0.6543338f32, -0.6566507f32, -0.6589615f32, -0.66126585f32, -0.66356415f32, -0.66585624f32, -0.6681421f32, -0.6704216f32, -0.6726948f32, -0.6749617f32, -0.6772222f32, -0.6794764f32, -0.68172413f32, -0.68396544f32, -0.6862004f32, -0.6884288f32, -0.69065076f32, -0.69286627f32, -0.6950752f32, -0.6972776f32, -0.69947344f32, -0.7016627f32, -0.7038453f32, -0.70602137f32, -0.70819074f32, -0.7103535f32, -0.7125095f32, -0.71465886f32, -0.7168014f32, -0.7189373f32, -0.72106636f32, -0.72318846f32, -0.72530395f32, -0.72741264f32, -0.7295145f32, -0.7316094f32, -0.7336975f32, -0.7357786f32, -0.7378529f32, -0.73992014f32, -0.74198043f32, -0.7440338f32, -0.74608016f32, -0.7481195f32, -0.7501517f32, -0.75217694f32, -0.75419503f32, -0.7562061f32, -0.7582098f32, -0.76020664f32, -0.76219624f32, -0.7641787f32, -0.76615393f32, -0.768122f32, -0.77008283f32, -0.7720364f32, -0.7739827f32, -0.7759217f32, -0.77785337f32, -0.77977777f32, -0.7816948f32, -0.7836045f32, -0.78550684f32, -0.7874018f32, -0.7892893f32, -0.79116935f32, -0.793042f32, -0.79490715f32, -0.79676485f32, -0.79861504f32, -0.8004577f32, -0.8022929f32, -0.8041204f32, -0.80594045f32, -0.8077529f32, -0.80955774f32, -0.81135494f32, -0.8131445f32, -0.81492645f32, -0.8167007f32, -0.81846726f32, -0.8202261f32, -0.82197726f32, -0.82372063f32, -0.82545626f32, -0.82718414f32, -0.8289043f32, -0.83061653f32, -0.832321f32, -0.83401763f32, -0.8357064f32, -0.8373874f32, -0.83906037f32, -0.84072554f32, -0.8423828f32, -0.84403205f32, -0.84567344f32, -0.8473068f32, -0.8489322f32, -0.85054964f32, -0.85215884f32, -0.85376024f32, -0.8553536f32, -0.85693896f32, -0.85851616f32, -0.86008537f32, -0.8616464f32, -0.8631994f32, -0.86474425f32, -0.8662809f32, -0.8678095f32, -0.86932987f32, -0.87084204f32, -0.87234604f32, -0.8738418f32, -0.87532943f32, -0.87680876f32, -0.8782798f32, -0.8797426f32, -0.88119715f32, -0.88264334f32, -0.8840813f32, -0.88551086f32, -0.88693213f32, -0.88834506f32, -0.88974965f32, -0.8911458f32, -0.8925336f32, -0.89391303f32, -0.895284f32, -0.89664656f32, -0.89800066f32, -0.8993463f32, -0.9006835f32, -0.9020122f32, -0.9033325f32, -0.9046442f32, -0.9059474f32, -0.90724206f32, -0.9085282f32, -0.9098058f32, -0.9110748f32, -0.9123353f32, -0.91358715f32, -0.91483045f32, -0.9160651f32, -0.9172911f32, -0.9185085f32, -0.91971725f32, -0.92091733f32, -0.92210877f32, -0.92329156f32, -0.9244656f32, -0.925631f32, -0.9267876f32, -0.92793536f32, -0.9290745f32, -0.930205f32, -0.9313267f32, -0.9324396f32, -0.93354374f32, -0.9346391f32, -0.9357257f32, -0.9368034f32, -0.93787235f32, -0.9389325f32, -0.9399837f32, -0.94102615f32, -0.94205976f32, -0.9430844f32, -0.94410026f32, -0.9451072f32, -0.94610524f32, -0.9470944f32, -0.9480746f32, -0.9490459f32, -0.9500083f32, -0.9509617f32, -0.95190614f32, -0.9528417f32, -0.9537682f32, -0.9546858f32, -0.95559436f32, -0.956494f32, -0.9573845f32, -0.95826614f32, -0.9591387f32, -0.9600022f32, -0.9608567f32, -0.9617021f32, -0.96253854f32, -0.96336585f32, -0.9641841f32, -0.9649933f32, -0.96579343f32, -0.96658444f32, -0.96736634f32, -0.9681392f32, -0.9689029f32, -0.9696574f32, -0.9704029f32, -0.97113925f32, -0.9718664f32, -0.9725844f32, -0.9732933f32, -0.97399306f32, -0.9746836f32, -0.975365f32, -0.97603714f32, -0.9767002f32, -0.977354f32, -0.9779985f32, -0.9786339f32, -0.9792601f32, -0.9798771f32, -0.98048484f32, -0.9810834f32, -0.9816727f32, -0.9822527f32, -0.98282355f32, -0.9833851f32, -0.9839374f32, -0.98448044f32, -0.98501426f32, -0.9855387f32, -0.98605394f32, -0.9865599f32, -0.98705655f32, -0.98754394f32, -0.988022f32, -0.9884908f32, -0.98895025f32, -0.98940045f32, -0.9898413f32, -0.9902728f32, -0.99069506f32, -0.99110794f32, -0.99151146f32, -0.9919057f32, -0.9922906f32, -0.9926662f32, -0.9930324f32, -0.99338925f32, -0.99373674f32, -0.9940749f32, -0.9944037f32, -0.99472314f32, -0.9950332f32, -0.9953339f32, -0.99562526f32, -0.99590725f32, -0.9961799f32, -0.9964431f32, -0.9966969f32, -0.9969414f32, -0.99717647f32, -0.99740213f32, -0.99761844f32, -0.9978254f32, -0.9980229f32, -0.998211f32, -0.9983898f32, -0.9985591f32, -0.99871904f32, -0.99886954f32, -0.9990107f32, -0.9991424f32, -0.9992647f32, -0.99937767f32, -0.9994812f32, -0.9995753f32, -0.99966f32, -0.9997353f32, -0.99980116f32, -0.99985766f32, -0.9999047f32, -0.99994236f32, -0.9999706f32, -0.9999894f32, -0.9999988f32, -1f32, -0.9999894f32, -0.9999706f32, -0.99994236f32, -0.9999047f32, -0.99985766f32, -0.99980116f32, -0.9997353f32, -0.99966f32, -0.9995753f32, -0.9994812f32, -0.99937767f32, -0.9992647f32, -0.9991424f32, -0.9990107f32, -0.99886954f32, -0.998719f32, -0.99855906f32, -0.9983897f32, -0.99821097f32, -0.99802285f32, -0.9978253f32, -0.99761844f32, -0.99740213f32, -0.9971764f32, -0.9969413f32, -0.9966969f32, -0.99644303f32, -0.9961798f32, -0.9959072f32, -0.99562526f32, -0.9953339f32, -0.99503314f32, -0.9947231f32, -0.99440366f32, -0.9940748f32, -0.9937367f32, -0.9933892f32, -0.99303234f32, -0.9926661f32, -0.99229056f32, -0.9919056f32, -0.99151146f32, -0.99110794f32, -0.99069506f32, -0.9902728f32, -0.9898413f32, -0.98940045f32, -0.98895025f32, -0.9884908f32, -0.988022f32, -0.98754394f32, -0.98705655f32, -0.9865599f32, -0.98605394f32, -0.9855387f32, -0.9850142f32, -0.98448044f32, -0.9839374f32, -0.9833851f32, -0.98282355f32, -0.9822527f32, -0.98167264f32, -0.9810834f32, -0.98048484f32, -0.97987705f32, -0.9792601f32, -0.9786339f32, -0.9779985f32, -0.9773539f32, -0.97670007f32, -0.976037f32, -0.97536486f32, -0.97468346f32, -0.97399294f32, -0.9732932f32, -0.9725843f32, -0.9718663f32, -0.97113913f32, -0.9704028f32, -0.9696573f32, -0.96890277f32, -0.96813905f32, -0.9673662f32, -0.9665843f32, -0.9657933f32, -0.9649932f32, -0.964184f32, -0.96336573f32, -0.96253836f32, -0.961702f32, -0.96085656f32, -0.96000206f32, -0.9591385f32, -0.95826596f32, -0.9573844f32, -0.9564938f32, -0.95559436f32, -0.9546858f32, -0.9537682f32, -0.95284164f32, -0.95190614f32, -0.9509617f32, -0.9500083f32, -0.9490459f32, -0.9480746f32, -0.9470944f32, -0.94610524f32, -0.9451072f32, -0.94410026f32, -0.9430844f32, -0.94205976f32, -0.94102615f32, -0.9399837f32, -0.9389325f32, -0.93787235f32, -0.9368034f32, -0.9357257f32, -0.9346391f32, -0.93354374f32, -0.93243957f32, -0.9313267f32, -0.930205f32, -0.9290745f32, -0.92793536f32, -0.92678744f32, -0.92563075f32, -0.9244654f32, -0.9232913f32, -0.9221086f32, -0.92091715f32, -0.9197171f32, -0.9185083f32, -0.9172909f32, -0.91606486f32, -0.9148302f32, -0.913587f32, -0.9123351f32, -0.91107464f32, -0.9098056f32, -0.90852803f32, -0.9072419f32, -0.90594715f32, -0.90464395f32, -0.90333223f32, -0.902012f32, -0.9006833f32, -0.8993461f32, -0.8980004f32, -0.8966463f32, -0.89528376f32, -0.8939128f32, -0.8925336f32, -0.8911458f32, -0.88974965f32, -0.88834506f32, -0.88693213f32, -0.88551086f32, -0.8840813f32, -0.88264334f32, -0.8811971f32, -0.8797426f32, -0.8782798f32, -0.8768087f32, -0.8753294f32, -0.8738418f32, -0.87234604f32, -0.87084204f32, -0.86932987f32, -0.8678095f32, -0.8662809f32, -0.86474425f32, -0.86319935f32, -0.8616464f32, -0.86008537f32, -0.85851616f32, -0.8569389f32, -0.8553536f32, -0.85376024f32, -0.85215884f32, -0.8505494f32, -0.84893197f32, -0.84730655f32, -0.84567314f32, -0.8440318f32, -0.8423825f32, -0.84072524f32, -0.8390601f32, -0.8373871f32, -0.8357062f32, -0.8340174f32, -0.83232075f32, -0.83061624f32, -0.828904f32, -0.8271839f32, -0.825456f32, -0.82372034f32, -0.82197696f32, -0.82022583f32, -0.81846696f32, -0.8167004f32, -0.81492615f32, -0.8131442f32, -0.81135464f32, -0.80955744f32, -0.8077526f32, -0.80594015f32, -0.8041201f32, -0.8022928f32, -0.8004577f32, -0.79861504f32, -0.79676485f32, -0.79490715f32, -0.793042f32, -0.79116935f32, -0.78928924f32, -0.78740174f32, -0.78550684f32, -0.7836045f32, -0.7816948f32, -0.77977777f32, -0.77785337f32, -0.7759217f32, -0.77398264f32, -0.7720364f32, -0.7700828f32, -0.76812196f32, -0.76615393f32, -0.7641787f32, -0.76219624f32, -0.76020664f32, -0.7582098f32, -0.7562059f32, -0.7541949f32, -0.75217676f32, -0.7501515f32, -0.7481193f32, -0.74608f32, -0.74403363f32, -0.7419803f32, -0.73991996f32, -0.7378527f32, -0.73577845f32, -0.7336973f32, -0.7316092f32, -0.7295143f32, -0.72741246f32, -0.72530377f32, -0.7231883f32, -0.721066f32, -0.7189369f32, -0.71680105f32, -0.7146585f32, -0.71250916f32, -0.71035314f32, -0.7081904f32, -0.706021f32, -0.703845f32, -0.70166236f32, -0.6994731f32, -0.69727725f32, -0.69507486f32, -0.6928659f32, -0.69065076f32, -0.6884288f32, -0.6862004f32, -0.68396544f32, -0.68172413f32, -0.6794763f32, -0.6772222f32, -0.6749617f32, -0.6726948f32, -0.67042154f32, -0.668142f32, -0.66585624f32, -0.66356415f32, -0.6612658f32, -0.65896124f32, -0.6566505f32, -0.6543336f32, -0.6520105f32, -0.64968127f32, -0.6473459f32, -0.64500445f32, -0.642657f32, -0.6403034f32, -0.6379438f32, -0.6355782f32, -0.63320667f32, -0.6308291f32, -0.6284456f32, -0.62605625f32, -0.623661f32, -0.6212598f32, -0.61885285f32, -0.61644f32, -0.6140214f32, -0.611597f32, -0.60916686f32, -0.60673094f32, -0.60428935f32, -0.60184205f32, -0.5993891f32, -0.5969305f32, -0.59446627f32, -0.5919965f32, -0.5895211f32, -0.5870401f32, -0.5845537f32, -0.5820617f32, -0.5795643f32, -0.5770614f32, -0.5745531f32, -0.57203937f32, -0.56952024f32, -0.56699574f32, -0.56446594f32, -0.5619308f32, -0.5593908f32, -0.55684507f32, -0.55429417f32, -0.551738f32, -0.5491767f32, -0.5466102f32, -0.54403853f32, -0.54146177f32, -0.53887993f32, -0.53629297f32, -0.533701f32, -0.53110397f32, -0.528502f32, -0.525895f32, -0.52328306f32, -0.5206662f32, -0.5180445f32, -0.5154178f32, -0.5127863f32, -0.51015f32, -0.50750893f32, -0.504863f32, -0.5022124f32, -0.49955702f32, -0.49689692f32, -0.49423218f32, -0.49156278f32, -0.48888877f32, -0.48621014f32, -0.48352692f32, -0.48083916f32, -0.47814688f32, -0.4754501f32, -0.47274885f32, -0.47004315f32, -0.46733302f32, -0.46461847f32, -0.46189958f32, -0.45917633f32, -0.45644876f32, -0.45371687f32, -0.45098075f32, -0.44824037f32, -0.44549575f32, -0.44274697f32, -0.43999398f32, -0.43723688f32, -0.43447566f32, -0.43171036f32, -0.42894098f32, -0.42616758f32, -0.42339015f32, -0.42060873f32, -0.41782337f32, -0.41503406f32, -0.41224086f32, -0.4094442f32, -0.40664327f32, -0.40383852f32, -0.40102994f32, -0.3982176f32, -0.3954015f32, -0.3925817f32, -0.38975817f32, -0.386931f32, -0.3841002f32, -0.38126576f32, -0.3784277f32, -0.37558615f32, -0.372741f32, -0.3698924f32, -0.36704028f32, -0.36418474f32, -0.36132574f32, -0.35846335f32, -0.35559756f32, -0.35272846f32, -0.34985602f32, -0.3469803f32, -0.3441013f32, -0.34121907f32, -0.33833364f32, -0.335445f32, -0.3325532f32, -0.3296583f32, -0.3267603f32, -0.32385918f32, -0.32095504f32, -0.31804788f32, -0.3151377f32, -0.3122246f32, -0.30930853f32, -0.30638957f32, -0.3034677f32, -0.30054298f32, -0.29761544f32, -0.2946851f32, -0.29175198f32, -0.28881612f32, -0.28587756f32, -0.28293628f32, -0.27999234f32, -0.27704576f32, -0.27409658f32, -0.27114484f32, -0.2681905f32, -0.26523367f32, -0.26227435f32, -0.25931254f32, -0.2563483f32, -0.25338164f32, -0.25041306f32, -0.24744168f32, -0.24446794f32, -0.24149193f32, -0.23851362f32, -0.23553309f32, -0.23255032f32, -0.22956537f32, -0.22657827f32, -0.22358902f32, -0.22059767f32, -0.21760425f32, -0.21460877f32, -0.21161129f32, -0.2086118f32, -0.20561035f32, -0.20260696f32, -0.19960168f32, -0.1965945f32, -0.19358549f32, -0.19057465f32, -0.18756202f32, -0.18454762f32, -0.18153147f32, -0.17851363f32, -0.1754941f32, -0.17247292f32, -0.16945012f32, -0.16642573f32, -0.16339977f32, -0.16037227f32, -0.15734325f32, -0.15431276f32, -0.15128082f32, -0.14824745f32, -0.1452127f32, -0.14217655f32, -0.13913909f32, -0.1361003f32, -0.13306026f32, -0.13001895f32, -0.12697642f32, -0.123932675f32, -0.12088778f32, -0.11784175f32, -0.114794604f32, -0.11174638f32, -0.1086971f32, -0.105646804f32, -0.10259551f32, -0.09954325f32, -0.096490055f32, -0.09343595f32, -0.09038097f32, -0.08732513f32, -0.08426895f32, -0.0812115f32, -0.07815329f32, -0.07509434f32, -0.07203468f32, -0.068974346f32, -0.065913364f32, -0.062851764f32, -0.059789564f32, -0.056726806f32, -0.053663515f32, -0.050599717f32, -0.047535446f32, -0.044470724f32, -0.041405585f32, -0.038340054f32, -0.035274163f32, -0.032207943f32, -0.029141417f32, -0.026074616f32, -0.023007572f32, -0.019940311f32, -0.01687286f32, -0.0138052525f32, -0.010737515f32, -0.0076696756f32, -0.0046017645f32, -0.0015338097f32, ];
#[rustfmt::skip] #[allow(clippy::all)] static COS: [f32; SIZE] = [1f32, 0.9999894f32, 0.9999706f32, 0.99994236f32, 0.9999047f32, 0.99985766f32, 0.99980116f32, 0.9997353f32, 0.99966f32, 0.9995753f32, 0.9994812f32, 0.99937767f32, 0.9992648f32, 0.9991424f32, 0.9990107f32, 0.99886954f32, 0.99871904f32, 0.99855906f32, 0.9983897f32, 0.998211f32, 0.99802285f32, 0.9978253f32, 0.99761844f32, 0.99740213f32, 0.9971764f32, 0.9969413f32, 0.9966969f32, 0.99644303f32, 0.9961798f32, 0.99590725f32, 0.99562526f32, 0.9953339f32, 0.9950332f32, 0.99472314f32, 0.99440366f32, 0.9940749f32, 0.99373674f32, 0.9933892f32, 0.99303234f32, 0.9926661f32, 0.9922906f32, 0.9919057f32, 0.99151146f32, 0.99110794f32, 0.990695f32, 0.9902728f32, 0.9898413f32, 0.98940045f32, 0.98895025f32, 0.9884908f32, 0.988022f32, 0.98754394f32, 0.98705655f32, 0.9865599f32, 0.98605394f32, 0.9855387f32, 0.98501426f32, 0.98448044f32, 0.98393744f32, 0.9833851f32, 0.98282355f32, 0.9822527f32, 0.9816727f32, 0.9810834f32, 0.98048484f32, 0.9798771f32, 0.97926015f32, 0.97863394f32, 0.9779985f32, 0.9773539f32, 0.97670007f32, 0.9760371f32, 0.97536486f32, 0.9746835f32, 0.97399294f32, 0.97329324f32, 0.97258437f32, 0.9718663f32, 0.97113913f32, 0.97040284f32, 0.96965736f32, 0.9689028f32, 0.9681391f32, 0.9673663f32, 0.9665844f32, 0.9657934f32, 0.96499324f32, 0.96418405f32, 0.9633658f32, 0.9625385f32, 0.96170205f32, 0.9608566f32, 0.9600021f32, 0.95913863f32, 0.9582661f32, 0.9573845f32, 0.9564939f32, 0.9555943f32, 0.95468575f32, 0.9537682f32, 0.95284164f32, 0.95190614f32, 0.95096165f32, 0.9500082f32, 0.9490459f32, 0.9480746f32, 0.9470944f32, 0.94610524f32, 0.94510716f32, 0.94410026f32, 0.9430844f32, 0.94205976f32, 0.94102615f32, 0.9399837f32, 0.9389325f32, 0.93787235f32, 0.93680346f32, 0.9357257f32, 0.9346391f32, 0.9335438f32, 0.9324396f32, 0.9313267f32, 0.930205f32, 0.9290746f32, 0.92793536f32, 0.9267875f32, 0.9256308f32, 0.9244655f32, 0.9232914f32, 0.92210865f32, 0.9209172f32, 0.91971713f32, 0.9185084f32, 0.917291f32, 0.916065f32, 0.9148303f32, 0.91358703f32, 0.91233516f32, 0.91107476f32, 0.9098057f32, 0.9085281f32, 0.907242f32, 0.90594727f32, 0.9046441f32, 0.90333235f32, 0.9020121f32, 0.9006834f32, 0.89934623f32, 0.8980006f32, 0.89664644f32, 0.89528394f32, 0.893913f32, 0.89253354f32, 0.89114577f32, 0.8897496f32, 0.888345f32, 0.88693213f32, 0.88551086f32, 0.88408124f32, 0.88264334f32, 0.8811971f32, 0.87974256f32, 0.8782798f32, 0.8768087f32, 0.8753294f32, 0.8738418f32, 0.87234604f32, 0.87084204f32, 0.86932987f32, 0.8678095f32, 0.866281f32, 0.86474425f32, 0.8631994f32, 0.8616465f32, 0.86008537f32, 0.8585162f32, 0.85693896f32, 0.85535365f32, 0.8537603f32, 0.8521589f32, 0.85054946f32, 0.848932f32, 0.8473066f32, 0.84567326f32, 0.8440319f32, 0.8423826f32, 0.84072536f32, 0.83906025f32, 0.8373872f32, 0.83570623f32, 0.8340175f32, 0.83232087f32, 0.8306164f32, 0.8289041f32, 0.827184f32, 0.82545614f32, 0.8237205f32, 0.82197714f32, 0.82022595f32, 0.81846714f32, 0.8167006f32, 0.8149263f32, 0.8131444f32, 0.8113548f32, 0.8095577f32, 0.80775285f32, 0.8059404f32, 0.80412036f32, 0.80229276f32, 0.80045766f32, 0.798615f32, 0.7967648f32, 0.79490715f32, 0.79304194f32, 0.79116935f32, 0.78928924f32, 0.78740174f32, 0.7855068f32, 0.7836045f32, 0.7816948f32, 0.77977777f32, 0.77785337f32, 0.7759217f32, 0.77398264f32, 0.7720364f32, 0.77008283f32, 0.768122f32, 0.766154f32, 0.76417875f32, 0.7621963f32, 0.76020664f32, 0.7582099f32, 0.756206f32, 0.754195f32, 0.7521768f32, 0.75015163f32, 0.74811935f32, 0.74608004f32, 0.7440337f32, 0.74198043f32, 0.7399201f32, 0.7378528f32, 0.73577857f32, 0.7336974f32, 0.73160934f32, 0.7295144f32, 0.72741264f32, 0.72530395f32, 0.72318846f32, 0.7210662f32, 0.7189371f32, 0.7168012f32, 0.7146587f32, 0.7125094f32, 0.7103533f32, 0.7081906f32, 0.70602125f32, 0.7038452f32, 0.70166254f32, 0.6994733f32, 0.6972775f32, 0.6950751f32, 0.69286615f32, 0.6906507f32, 0.6884287f32, 0.68620026f32, 0.68396544f32, 0.6817241f32, 0.6794763f32, 0.67722213f32, 0.6749616f32, 0.67269474f32, 0.67042154f32, 0.668142f32, 0.66585624f32, 0.66356415f32, 0.66126585f32, 0.6589613f32, 0.65665054f32, 0.6543336f32, 0.65201056f32, 0.6496813f32, 0.64734596f32, 0.6450045f32, 0.642657f32, 0.64030343f32, 0.63794386f32, 0.63557833f32, 0.6332067f32, 0.6308292f32, 0.62844574f32, 0.6260564f32, 0.6236611f32, 0.6212599f32, 0.618853f32, 0.6164402f32, 0.61402154f32, 0.6115971f32, 0.609167f32, 0.6067311f32, 0.6042895f32, 0.6018422f32, 0.5993893f32, 0.5969307f32, 0.5944665f32, 0.59199667f32, 0.5895213f32, 0.58704036f32, 0.58455396f32, 0.582062f32, 0.5795646f32, 0.57706165f32, 0.5745533f32, 0.5720396f32, 0.5695205f32, 0.56699604f32, 0.56446624f32, 0.56193113f32, 0.55939066f32, 0.556845f32, 0.5542941f32, 0.55173796f32, 0.54917663f32, 0.5466102f32, 0.54403853f32, 0.54146177f32, 0.53887993f32, 0.53629297f32, 0.533701f32, 0.53110397f32, 0.528502f32, 0.525895f32, 0.52328306f32, 0.52066624f32, 0.5180445f32, 0.5154178f32, 0.5127863f32, 0.51015f32, 0.50750893f32, 0.50486314f32, 0.50221246f32, 0.4995571f32, 0.49689704f32, 0.4942323f32, 0.4915629f32, 0.48888886f32, 0.48621026f32, 0.48352703f32, 0.48083928f32, 0.478147f32, 0.47545022f32, 0.47274897f32, 0.47004324f32, 0.46733323f32, 0.4646187f32, 0.4618998f32, 0.45917654f32, 0.45644897f32, 0.4537171f32, 0.45098096f32, 0.44824058f32, 0.44549596f32, 0.44274718f32, 0.43999422f32, 0.4372371f32, 0.4344759f32, 0.43171057f32, 0.4289413f32, 0.4261679f32, 0.42339048f32, 0.42060906f32, 0.4178237f32, 0.4150344f32, 0.4122412f32, 0.40944412f32, 0.40664318f32, 0.4038384f32, 0.40102983f32, 0.3982175f32, 0.3954014f32, 0.39258158f32, 0.3897582f32, 0.386931f32, 0.3841002f32, 0.38126576f32, 0.37842774f32, 0.37558615f32, 0.37274104f32, 0.36989242f32, 0.3670403f32, 0.36418474f32, 0.36132574f32, 0.35846335f32, 0.3555976f32, 0.35272858f32, 0.34985614f32, 0.34698042f32, 0.34410143f32, 0.3412192f32, 0.33833376f32, 0.33544514f32, 0.33255333f32, 0.32965842f32, 0.3267604f32, 0.3238593f32, 0.32095516f32, 0.318048f32, 0.31513783f32, 0.31222484f32, 0.30930877f32, 0.3063898f32, 0.30346793f32, 0.30054322f32, 0.29761568f32, 0.29468533f32, 0.29175222f32, 0.28881636f32, 0.2858778f32, 0.2829365f32, 0.27999258f32, 0.277046f32, 0.27409682f32, 0.27114516f32, 0.26819086f32, 0.26523402f32, 0.2622747f32, 0.2593129f32, 0.25634867f32, 0.253382f32, 0.25041297f32, 0.24744157f32, 0.24446784f32, 0.24149182f32, 0.23851351f32, 0.23553298f32, 0.23255022f32, 0.22956538f32, 0.22657827f32, 0.22358903f32, 0.22059768f32, 0.21760426f32, 0.21460879f32, 0.21161129f32, 0.20861182f32, 0.20561036f32, 0.20260698f32, 0.1996017f32, 0.19659452f32, 0.1935855f32, 0.19057466f32, 0.18756214f32, 0.18454774f32, 0.18153161f32, 0.17851377f32, 0.17549424f32, 0.17247306f32, 0.16945025f32, 0.16642585f32, 0.16339989f32, 0.16037239f32, 0.15734339f32, 0.1543129f32, 0.15128095f32, 0.14824758f32, 0.14521293f32, 0.1421768f32, 0.13913934f32, 0.13610056f32, 0.1330605f32, 0.13001919f32, 0.12697665f32, 0.12393293f32, 0.12088803f32, 0.117841996f32, 0.11479485f32, 0.11174663f32, 0.108697355f32, 0.10564717f32, 0.10259588f32, 0.099543616f32, 0.09649042f32, 0.093436316f32, 0.09038134f32, 0.0873255f32, 0.084268846f32, 0.081211396f32, 0.07815318f32, 0.07509423f32, 0.072034575f32, 0.06897424f32, 0.06591326f32, 0.06285177f32, 0.059789576f32, 0.05672682f32, 0.053663526f32, 0.05059973f32, 0.047535457f32, 0.044470735f32, 0.041405596f32, 0.038340066f32, 0.035274178f32, 0.032207955f32, 0.029141428f32, 0.02607463f32, 0.023007585f32, 0.019940441f32, 0.016872993f32, 0.013805384f32, 0.010737646f32, 0.007669807f32, 0.0046018953f32, 0.0015339408f32, -0.00000004371139f32, -0.004601983f32, -0.007669894f32, -0.010737733f32, -0.013805471f32, -0.01687308f32, -0.019940529f32, -0.02300767f32, -0.026074717f32, -0.029141515f32, -0.03220804f32, -0.035274263f32, -0.038340155f32, -0.04140568f32, -0.044470824f32, -0.047535542f32, -0.050599817f32, -0.053663615f32, -0.056726906f32, -0.059789665f32, -0.06285186f32, -0.06591334f32, -0.06897433f32, -0.072034664f32, -0.07509432f32, -0.07815327f32, -0.081211485f32, -0.084268935f32, -0.08732559f32, -0.09038142f32, -0.093436405f32, -0.09649051f32, -0.099543706f32, -0.10259596f32, -0.10564726f32, -0.10869744f32, -0.11174671f32, -0.11479494f32, -0.117842086f32, -0.120888114f32, -0.12393301f32, -0.12697674f32, -0.13001928f32, -0.13306059f32, -0.13610065f32, -0.13913943f32, -0.1421769f32, -0.14521302f32, -0.14824767f32, -0.15128103f32, -0.15431298f32, -0.15734348f32, -0.16037248f32, -0.16339998f32, -0.16642594f32, -0.16945034f32, -0.17247315f32, -0.17549431f32, -0.17851384f32, -0.1815317f32, -0.18454783f32, -0.18756223f32, -0.19057475f32, -0.19358559f32, -0.19659461f32, -0.19960177f32, -0.20260707f32, -0.20561045f32, -0.20861189f32, -0.21161138f32, -0.21460888f32, -0.21760434f32, -0.22059777f32, -0.2235891f32, -0.22657835f32, -0.22956547f32, -0.23255031f32, -0.23553306f32, -0.2385136f32, -0.2414919f32, -0.24446793f32, -0.24744165f32, -0.25041306f32, -0.2533821f32, -0.25634873f32, -0.259313f32, -0.26227477f32, -0.2652341f32, -0.26819095f32, -0.27114525f32, -0.2740969f32, -0.27704608f32, -0.27999267f32, -0.2829366f32, -0.28587785f32, -0.28881645f32, -0.2917523f32, -0.29468542f32, -0.29761577f32, -0.3005433f32, -0.30346802f32, -0.3063899f32, -0.30930886f32, -0.31222492f32, -0.31513792f32, -0.3180481f32, -0.32095525f32, -0.3238594f32, -0.32676047f32, -0.3296585f32, -0.33255342f32, -0.3354452f32, -0.33833385f32, -0.34121928f32, -0.34410152f32, -0.3469805f32, -0.34985623f32, -0.35272866f32, -0.35559767f32, -0.35846344f32, -0.36132583f32, -0.36418483f32, -0.36704037f32, -0.36989248f32, -0.3727411f32, -0.37558624f32, -0.37842783f32, -0.38126585f32, -0.3841003f32, -0.3869311f32, -0.38975826f32, -0.39258167f32, -0.39540148f32, -0.3982176f32, -0.4010299f32, -0.4038385f32, -0.40664324f32, -0.40944418f32, -0.41224128f32, -0.41503447f32, -0.41782367f32, -0.42060903f32, -0.42339045f32, -0.42616788f32, -0.42894128f32, -0.43171066f32, -0.43447596f32, -0.43723717f32, -0.4399943f32, -0.44274727f32, -0.44549605f32, -0.44824067f32, -0.45098105f32, -0.45371717f32, -0.45644906f32, -0.45917663f32, -0.46189988f32, -0.46461877f32, -0.46733332f32, -0.47004345f32, -0.47274914f32, -0.4754504f32, -0.47814718f32, -0.48083946f32, -0.4835272f32, -0.48621044f32, -0.48888886f32, -0.49156287f32, -0.49423227f32, -0.496897f32, -0.4995571f32, -0.50221246f32, -0.5048631f32, -0.507509f32, -0.51015013f32, -0.51278645f32, -0.5154179f32, -0.51804453f32, -0.5206663f32, -0.5232831f32, -0.52589506f32, -0.52850205f32, -0.5311041f32, -0.53370106f32, -0.5362931f32, -0.53888f32, -0.5414619f32, -0.54403865f32, -0.5466103f32, -0.54917675f32, -0.55173814f32, -0.5542942f32, -0.5568452f32, -0.55939084f32, -0.5619311f32, -0.5644662f32, -0.56699604f32, -0.5695205f32, -0.5720396f32, -0.5745534f32, -0.57706165f32, -0.5795646f32, -0.582062f32, -0.58455396f32, -0.5870404f32, -0.58952135f32, -0.5919967f32, -0.59446657f32, -0.59693074f32, -0.5993894f32, -0.60184234f32, -0.6042896f32, -0.60673124f32, -0.6091671f32, -0.61159724f32, -0.61402166f32, -0.6164403f32, -0.6188531f32, -0.6212601f32, -0.6236612f32, -0.62605655f32, -0.62844574f32, -0.6308292f32, -0.6332067f32, -0.6355783f32, -0.63794386f32, -0.6403035f32, -0.64265704f32, -0.6450046f32, -0.64734596f32, -0.6496813f32, -0.65201056f32, -0.65433365f32, -0.6566506f32, -0.65896136f32, -0.6612659f32, -0.6635642f32, -0.6658563f32, -0.6681421f32, -0.67042166f32, -0.67269486f32, -0.67496175f32, -0.67722225f32, -0.67947644f32, -0.6817242f32, -0.6839655f32, -0.68620044f32, -0.6884289f32, -0.6906508f32, -0.69286615f32, -0.6950751f32, -0.6972775f32, -0.6994733f32, -0.7016626f32, -0.70384526f32, -0.70602125f32, -0.7081906f32, -0.7103534f32, -0.7125094f32, -0.71465874f32, -0.7168013f32, -0.71893716f32, -0.72106624f32, -0.7231885f32, -0.725304f32, -0.7274127f32, -0.7295145f32, -0.73160946f32, -0.73369753f32, -0.7357787f32, -0.73785293f32, -0.7399202f32, -0.7419805f32, -0.7440339f32, -0.74608016f32, -0.7481195f32, -0.75015175f32, -0.7521768f32, -0.754195f32, -0.756206f32, -0.7582099f32, -0.7602067f32, -0.7621963f32, -0.76417875f32, -0.766154f32, -0.768122f32, -0.77008283f32, -0.77203643f32, -0.7739827f32, -0.77592176f32, -0.7778534f32, -0.7797778f32, -0.7816949f32, -0.78360456f32, -0.7855069f32, -0.7874018f32, -0.78928936f32, -0.7911694f32, -0.79304206f32, -0.7949072f32, -0.7967649f32, -0.7986151f32, -0.8004578f32, -0.8022929f32, -0.80412036f32, -0.8059404f32, -0.8077528f32, -0.8095576f32, -0.8113548f32, -0.8131444f32, -0.8149263f32, -0.8167006f32, -0.81846714f32, -0.820226f32, -0.82197714f32, -0.8237205f32, -0.8254562f32, -0.8271841f32, -0.82890415f32, -0.8306165f32, -0.8323209f32, -0.8340176f32, -0.83570635f32, -0.83738726f32, -0.8390603f32, -0.8407254f32, -0.84238267f32, -0.844032f32, -0.8456733f32, -0.8473067f32, -0.84893215f32, -0.8505496f32, -0.8521589f32, -0.8537603f32, -0.85535365f32, -0.85693896f32, -0.8585162f32, -0.86008537f32, -0.8616465f32, -0.8631994f32, -0.86474425f32, -0.866281f32, -0.86780953f32, -0.86932987f32, -0.8708421f32, -0.8723461f32, -0.8738419f32, -0.87532943f32, -0.87680876f32, -0.87827986f32, -0.8797426f32, -0.88119715f32, -0.8826434f32, -0.8840813f32, -0.8855109f32, -0.8869322f32, -0.8883451f32, -0.88974965f32, -0.8911458f32, -0.89253366f32, -0.8939129f32, -0.89528394f32, -0.89664644f32, -0.8980006f32, -0.89934623f32, -0.9006834f32, -0.90201217f32, -0.90333235f32, -0.90464413f32, -0.9059473f32, -0.907242f32, -0.90852815f32, -0.9098057f32, -0.91107476f32, -0.9123352f32, -0.9135871f32, -0.9148303f32, -0.91606504f32, -0.91729105f32, -0.91850847f32, -0.9197172f32, -0.9209173f32, -0.9221087f32, -0.9232915f32, -0.92446554f32, -0.9256309f32, -0.92678756f32, -0.92793536f32, -0.9290746f32, -0.930205f32, -0.9313267f32, -0.9324396f32, -0.9335438f32, -0.93463916f32, -0.9357257f32, -0.93680346f32, -0.9378724f32, -0.9389325f32, -0.9399838f32, -0.9410262f32, -0.94205976f32, -0.9430845f32, -0.94410026f32, -0.9451072f32, -0.94610524f32, -0.9470944f32, -0.94807464f32, -0.9490459f32, -0.9500083f32, -0.9509617f32, -0.9519062f32, -0.9528417f32, -0.95376825f32, -0.9546858f32, -0.9555944f32, -0.9564939f32, -0.95738447f32, -0.9582661f32, -0.95913863f32, -0.9600021f32, -0.9608566f32, -0.9617021f32, -0.9625385f32, -0.9633658f32, -0.96418405f32, -0.96499324f32, -0.9657934f32, -0.9665844f32, -0.96736634f32, -0.9681391f32, -0.9689028f32, -0.9696574f32, -0.97040284f32, -0.9711392f32, -0.97186637f32, -0.9725844f32, -0.9732933f32, -0.973993f32, -0.9746835f32, -0.9753649f32, -0.97603714f32, -0.9767001f32, -0.97735393f32, -0.9779985f32, -0.97863394f32, -0.97926015f32, -0.9798771f32, -0.98048484f32, -0.9810834f32, -0.9816727f32, -0.9822528f32, -0.98282355f32, -0.98338515f32, -0.98393744f32, -0.98448044f32, -0.98501426f32, -0.9855387f32, -0.986054f32, -0.9865599f32, -0.9870566f32, -0.98754394f32, -0.988022f32, -0.9884908f32, -0.9889503f32, -0.98940045f32, -0.9898413f32, -0.9902728f32, -0.99069506f32, -0.99110794f32, -0.9915115f32, -0.9919057f32, -0.9922906f32, -0.9926661f32, -0.99303234f32, -0.9933892f32, -0.99373674f32, -0.9940749f32, -0.99440366f32, -0.99472314f32, -0.9950332f32, -0.9953339f32, -0.99562526f32, -0.99590725f32, -0.9961798f32, -0.99644303f32, -0.9966969f32, -0.9969414f32, -0.99717647f32, -0.99740213f32, -0.99761844f32, -0.9978254f32, -0.99802285f32, -0.998211f32, -0.9983897f32, -0.99855906f32, -0.99871904f32, -0.99886954f32, -0.9990107f32, -0.9991424f32, -0.9992647f32, -0.99937767f32, -0.9994812f32, -0.9995753f32, -0.99966f32, -0.9997353f32, -0.99980116f32, -0.99985766f32, -0.9999047f32, -0.99994236f32, -0.9999706f32, -0.9999894f32, -0.9999988f32, -1f32, -0.9999894f32, -0.9999706f32, -0.99994236f32, -0.9999047f32, -0.99985766f32, -0.99980116f32, -0.9997353f32, -0.99966f32, -0.9995753f32, -0.9994812f32, -0.99937767f32, -0.9992647f32, -0.9991424f32, -0.9990107f32, -0.99886954f32, -0.99871904f32, -0.99855906f32, -0.9983897f32, -0.998211f32, -0.99802285f32, -0.9978253f32, -0.99761844f32, -0.99740213f32, -0.9971764f32, -0.9969413f32, -0.9966969f32, -0.99644303f32, -0.9961798f32, -0.99590725f32, -0.99562526f32, -0.9953339f32, -0.9950332f32, -0.9947231f32, -0.99440366f32, -0.9940749f32, -0.9937367f32, -0.9933892f32, -0.99303234f32, -0.9926661f32, -0.99229056f32, -0.9919057f32, -0.99151146f32, -0.99110794f32, -0.990695f32, -0.9902728f32, -0.9898413f32, -0.98940045f32, -0.98895025f32, -0.98849076f32, -0.988022f32, -0.98754394f32, -0.98705655f32, -0.9865599f32, -0.98605394f32, -0.9855387f32, -0.9850142f32, -0.98448044f32, -0.9839374f32, -0.9833851f32, -0.98282355f32, -0.9822527f32, -0.98167264f32, -0.98108333f32, -0.98048484f32, -0.97987705f32, -0.9792601f32, -0.9786339f32, -0.9779985f32, -0.97735393f32, -0.97670007f32, -0.9760371f32, -0.97536486f32, -0.9746835f32, -0.97399294f32, -0.97329324f32, -0.97258437f32, -0.9718663f32, -0.97113913f32, -0.97040284f32, -0.96965736f32, -0.96890277f32, -0.9681391f32, -0.9673663f32, -0.9665843f32, -0.9657933f32, -0.96499324f32, -0.96418405f32, -0.96336573f32, -0.9625384f32, -0.96170205f32, -0.96085656f32, -0.9600021f32, -0.9591386f32, -0.958266f32, -0.95738447f32, -0.95649385f32, -0.95559436f32, -0.95468575f32, -0.9537682f32, -0.95284164f32, -0.95190614f32, -0.95096165f32, -0.9500082f32, -0.9490459f32, -0.9480746f32, -0.9470943f32, -0.9461052f32, -0.94510716f32, -0.9441002f32, -0.9430844f32, -0.9420597f32, -0.94102615f32, -0.9399837f32, -0.9389324f32, -0.93787235f32, -0.9368034f32, -0.9357256f32, -0.9346391f32, -0.9335437f32, -0.93243957f32, -0.9313266f32, -0.9302049f32, -0.9290745f32, -0.9279353f32, -0.9267875f32, -0.9256308f32, -0.9244655f32, -0.9232914f32, -0.92210865f32, -0.9209172f32, -0.91971713f32, -0.91850835f32, -0.917291f32, -0.9160649f32, -0.91483027f32, -0.91358703f32, -0.91233516f32, -0.9110747f32, -0.90980566f32, -0.9085281f32, -0.90724194f32, -0.90594727f32, -0.904644f32, -0.9033323f32, -0.90201205f32, -0.90068334f32, -0.8993462f32, -0.8980005f32, -0.8966464f32, -0.8952838f32, -0.89391285f32, -0.89253354f32, -0.89114577f32, -0.8897496f32, -0.888345f32, -0.88693213f32, -0.88551086f32, -0.88408124f32, -0.88264334f32, -0.8811971f32, -0.87974256f32, -0.87827975f32, -0.8768087f32, -0.8753294f32, -0.8738418f32, -0.872346f32, -0.870842f32, -0.8693298f32, -0.8678094f32, -0.8662809f32, -0.8647442f32, -0.86319935f32, -0.86164635f32, -0.8600853f32, -0.85851616f32, -0.8569389f32, -0.85535353f32, -0.8537602f32, -0.8521588f32, -0.85054946f32, -0.848932f32, -0.8473066f32, -0.84567326f32, -0.8440319f32, -0.8423826f32, -0.84072536f32, -0.8390602f32, -0.83738714f32, -0.83570623f32, -0.83401746f32, -0.8323208f32, -0.83061635f32, -0.82890403f32, -0.82718396f32, -0.8254561f32, -0.82372046f32, -0.821977f32, -0.8202259f32, -0.818467f32, -0.81670046f32, -0.8149262f32, -0.8131443f32, -0.81135476f32, -0.8095575f32, -0.80775267f32, -0.8059403f32, -0.80412024f32, -0.8022928f32, -0.80045766f32, -0.798615f32, -0.7967648f32, -0.7949071f32, -0.79304194f32, -0.7911693f32, -0.78928924f32, -0.78740174f32, -0.7855068f32, -0.78360444f32, -0.78169477f32, -0.7797777f32, -0.7778533f32, -0.77592164f32, -0.7739826f32, -0.7720363f32, -0.7700828f32, -0.76812196f32, -0.7661539f32, -0.76417863f32, -0.7621962f32, -0.7602066f32, -0.75820976f32, -0.75620586f32, -0.75419486f32, -0.7521767f32, -0.75015163f32, -0.74811935f32, -0.74608004f32, -0.74403375f32, -0.7419804f32, -0.7399201f32, -0.7378528f32, -0.73577857f32, -0.7336974f32, -0.73160934f32, -0.72951436f32, -0.7274126f32, -0.7253039f32, -0.7231884f32, -0.7210661f32, -0.71893704f32, -0.71680117f32, -0.7146586f32, -0.7125093f32, -0.71035326f32, -0.7081905f32, -0.70602113f32, -0.7038451f32, -0.7016625f32, -0.6994732f32, -0.69727737f32, -0.695075f32, -0.692866f32, -0.6906507f32, -0.68842876f32, -0.6862003f32, -0.6839654f32, -0.6817241f32, -0.6794763f32, -0.67722213f32, -0.6749616f32, -0.67269474f32, -0.6704215f32, -0.66814196f32, -0.6658562f32, -0.6635641f32, -0.66126573f32, -0.6589612f32, -0.6566504f32, -0.65433353f32, -0.6520106f32, -0.6496814f32, -0.647346f32, -0.6450046f32, -0.6426571f32, -0.6403035f32, -0.6379439f32, -0.63557833f32, -0.6332068f32, -0.6308292f32, -0.62844574f32, -0.6260564f32, -0.6236611f32, -0.62126f32, -0.618853f32, -0.6164401f32, -0.61402154f32, -0.6115971f32, -0.609167f32, -0.60673106f32, -0.6042895f32, -0.60184216f32, -0.5993892f32, -0.5969306f32, -0.5944664f32, -0.5919966f32, -0.5895212f32, -0.5870403f32, -0.58455384f32, -0.5820619f32, -0.57956445f32, -0.57706153f32, -0.5745532f32, -0.5720395f32, -0.56952035f32, -0.56699586f32, -0.56446606f32, -0.56193095f32, -0.55939054f32, -0.55684483f32, -0.55429393f32, -0.5517378f32, -0.54917645f32, -0.54660994f32, -0.5440383f32, -0.5414615f32, -0.53887963f32, -0.53629273f32, -0.5337007f32, -0.53110373f32, -0.5285017f32, -0.5258947f32, -0.5232832f32, -0.52066636f32, -0.5180446f32, -0.51541793f32, -0.51278645f32, -0.51015013f32, -0.50750905f32, -0.50486314f32, -0.5022125f32, -0.49955714f32, -0.49689707f32, -0.49423233f32, -0.49156293f32, -0.4888889f32, -0.4862103f32, -0.48352706f32, -0.4808393f32, -0.47814703f32, -0.47545025f32, -0.472749f32, -0.47004327f32, -0.46733314f32, -0.46461862f32, -0.46189973f32, -0.45917648f32, -0.45644888f32, -0.45371702f32, -0.45098087f32, -0.4482405f32, -0.4454959f32, -0.4427471f32, -0.43999413f32, -0.43723702f32, -0.4344758f32, -0.4317105f32, -0.42894113f32, -0.42616773f32, -0.4233903f32, -0.42060888f32, -0.41782352f32, -0.4150342f32, -0.412241f32, -0.4094439f32, -0.40664297f32, -0.40383822f32, -0.40102965f32, -0.3982173f32, -0.3954012f32, -0.3925814f32, -0.3897579f32, -0.3869307f32, -0.3840999f32, -0.38126546f32, -0.37842745f32, -0.37558585f32, -0.37274116f32, -0.36989254f32, -0.36704043f32, -0.3641849f32, -0.3613259f32, -0.3584635f32, -0.35559773f32, -0.3527286f32, -0.34985617f32, -0.34698045f32, -0.34410146f32, -0.34121922f32, -0.3383338f32, -0.33544517f32, -0.33255336f32, -0.32965845f32, -0.32676044f32, -0.32385933f32, -0.3209552f32, -0.31804803f32, -0.31513786f32, -0.31222475f32, -0.30930868f32, -0.30638972f32, -0.30346787f32, -0.30054316f32, -0.29761562f32, -0.29468527f32, -0.29175213f32, -0.28881627f32, -0.2858777f32, -0.28293642f32, -0.2799925f32, -0.2770459f32, -0.27409673f32, -0.271145f32, -0.26819068f32, -0.26523384f32, -0.2622745f32, -0.2593127f32, -0.25634846f32, -0.25338182f32, -0.25041276f32, -0.24744137f32, -0.24446765f32, -0.24149162f32, -0.23851332f32, -0.23553278f32, -0.23255001f32, -0.22956507f32, -0.22657795f32, -0.2235887f32, -0.22059736f32, -0.21760394f32, -0.21460846f32, -0.21161097f32, -0.20861195f32, -0.20561051f32, -0.20260713f32, -0.19960184f32, -0.19659467f32, -0.19358565f32, -0.19057481f32, -0.18756217f32, -0.18454777f32, -0.18153164f32, -0.1785138f32, -0.17549427f32, -0.17247309f32, -0.16945028f32, -0.16642588f32, -0.16339992f32, -0.16037242f32, -0.15734342f32, -0.15431292f32, -0.15128098f32, -0.14824761f32, -0.14521284f32, -0.14217672f32, -0.13913925f32, -0.13610047f32, -0.13306041f32, -0.1300191f32, -0.12697656f32, -0.12393284f32, -0.12088794f32, -0.11784191f32, -0.11479477f32, -0.11174654f32, -0.108697265f32, -0.10564697f32, -0.10259567f32, -0.099543415f32, -0.09649022f32, -0.093436114f32, -0.09038113f32, -0.0873253f32, -0.08426864f32, -0.08121119f32, -0.07815298f32, -0.07509403f32, -0.07203437f32, -0.06897403f32, -0.06591305f32, -0.06285145f32, -0.05978925f32, -0.056726493f32, -0.0536632f32, -0.050599404f32, -0.04753513f32, -0.04447041f32, -0.041405745f32, -0.03834022f32, -0.035274327f32, -0.032208104f32, -0.029141579f32, -0.02607478f32, -0.023007736f32, -0.019940473f32, -0.016873024f32, -0.0138054155f32, -0.010737678f32, -0.0076698386f32, -0.0046019275f32, -0.0015339726f32, 0.000000011924881f32, 0.004601951f32, 0.0076698624f32, 0.010737701f32, 0.01380544f32, 0.016873049f32, 0.019940497f32, 0.023007758f32, 0.026074804f32, 0.029141603f32, 0.03220813f32, 0.03527435f32, 0.03834024f32, 0.04140577f32, 0.04447091f32, 0.04753563f32, 0.050599903f32, 0.0536637f32, 0.056726992f32, 0.05978975f32, 0.06285195f32, 0.06591355f32, 0.06897453f32, 0.072034866f32, 0.07509453f32, 0.078153476f32, 0.081211686f32, 0.084269136f32, 0.0873258f32, 0.09038163f32, 0.09343661f32, 0.09649072f32, 0.099543914f32, 0.10259617f32, 0.10564747f32, 0.108697765f32, 0.11174704f32, 0.11479526f32, 0.117842406f32, 0.12088844f32, 0.12393334f32, 0.12697707f32, 0.13001913f32, 0.13306044f32, 0.1361005f32, 0.13913928f32, 0.14217675f32, 0.14521287f32, 0.14824763f32, 0.151281f32, 0.15431295f32, 0.15734343f32, 0.16037245f32, 0.16339995f32, 0.16642591f32, 0.16945031f32, 0.1724731f32, 0.17549428f32, 0.17851381f32, 0.18153165f32, 0.1845478f32, 0.1875622f32, 0.19057482f32, 0.19358568f32, 0.19659469f32, 0.19960186f32, 0.20260715f32, 0.20561053f32, 0.20861198f32, 0.21161146f32, 0.21460895f32, 0.21760443f32, 0.22059785f32, 0.2235892f32, 0.22657844f32, 0.22956555f32, 0.2325505f32, 0.23553327f32, 0.23851381f32, 0.24149211f32, 0.24446814f32, 0.24744186f32, 0.25041324f32, 0.2533823f32, 0.25634894f32, 0.2593132f32, 0.26227498f32, 0.26523432f32, 0.26819116f32, 0.27114546f32, 0.27409723f32, 0.2770464f32, 0.27999297f32, 0.2829369f32, 0.28587818f32, 0.28881675f32, 0.29175264f32, 0.29468527f32, 0.29761562f32, 0.30054316f32, 0.30346787f32, 0.30638975f32, 0.3093087f32, 0.31222478f32, 0.3151379f32, 0.31804806f32, 0.32095522f32, 0.32385936f32, 0.32676044f32, 0.32965848f32, 0.3325534f32, 0.33544517f32, 0.33833382f32, 0.34121925f32, 0.3441015f32, 0.34698048f32, 0.3498562f32, 0.35272864f32, 0.35559773f32, 0.35846353f32, 0.36132592f32, 0.3641849f32, 0.36704046f32, 0.36989257f32, 0.3727412f32, 0.37558633f32, 0.3784279f32, 0.38126594f32, 0.38410035f32, 0.38693118f32, 0.38975835f32, 0.39258185f32, 0.3954017f32, 0.39821777f32, 0.4010301f32, 0.40383866f32, 0.40664345f32, 0.4094444f32, 0.41224146f32, 0.41503468f32, 0.41782397f32, 0.42060933f32, 0.42339075f32, 0.42616817f32, 0.42894158f32, 0.43171096f32, 0.43447626f32, 0.43723747f32, 0.43999457f32, 0.44274756f32, 0.44549635f32, 0.44824094f32, 0.4509809f32, 0.45371705f32, 0.4564489f32, 0.45917648f32, 0.46189973f32, 0.46461865f32, 0.46733317f32, 0.4700433f32, 0.47274902f32, 0.47545028f32, 0.47814706f32, 0.48083934f32, 0.4835271f32, 0.4862103f32, 0.48888892f32, 0.49156296f32, 0.49423236f32, 0.4968971f32, 0.49955717f32, 0.5022125f32, 0.5048632f32, 0.50750905f32, 0.5101502f32, 0.5127865f32, 0.515418f32, 0.5180446f32, 0.52066636f32, 0.52328324f32, 0.5258952f32, 0.52850217f32, 0.53110415f32, 0.5337012f32, 0.53629315f32, 0.53888005f32, 0.54146194f32, 0.5440387f32, 0.54661036f32, 0.5491769f32, 0.5517382f32, 0.55429435f32, 0.55684525f32, 0.5593909f32, 0.5619314f32, 0.5644665f32, 0.5669963f32, 0.5695208f32, 0.5720399f32, 0.5745536f32, 0.57706195f32, 0.5795648f32, 0.58206224f32, 0.58455426f32, 0.58704066f32, 0.58952165f32, 0.591997f32, 0.5944668f32, 0.5969306f32, 0.59938926f32, 0.6018422f32, 0.6042895f32, 0.6067311f32, 0.609167f32, 0.6115971f32, 0.61402154f32, 0.6164402f32, 0.618853f32, 0.62126f32, 0.6236611f32, 0.62605643f32, 0.6284458f32, 0.6308293f32, 0.6332068f32, 0.6355784f32, 0.637944f32, 0.64030355f32, 0.6426571f32, 0.64500463f32, 0.647346f32, 0.6496814f32, 0.6520106f32, 0.6543337f32, 0.65665066f32, 0.6589614f32, 0.66126597f32, 0.66356426f32, 0.66585636f32, 0.6681422f32, 0.6704217f32, 0.6726949f32, 0.6749618f32, 0.6772223f32, 0.6794765f32, 0.68172425f32, 0.68396556f32, 0.6862005f32, 0.68842894f32, 0.6906509f32, 0.6928664f32, 0.69507533f32, 0.6972777f32, 0.69947356f32, 0.70166284f32, 0.70384544f32, 0.7060215f32, 0.70819086f32, 0.7103536f32, 0.71250963f32, 0.7146589f32, 0.7168015f32, 0.7189374f32, 0.7210665f32, 0.72318846f32, 0.72530395f32, 0.7274126f32, 0.7295144f32, 0.73160934f32, 0.7336974f32, 0.73577857f32, 0.7378528f32, 0.7399201f32, 0.74198043f32, 0.74403375f32, 0.7460801f32, 0.7481194f32, 0.7501517f32, 0.7521769f32, 0.75419503f32, 0.75620604f32, 0.75820994f32, 0.76020676f32, 0.76219636f32, 0.7641788f32, 0.76615405f32, 0.7681221f32, 0.7700829f32, 0.7720365f32, 0.77398276f32, 0.77592176f32, 0.7778535f32, 0.7797779f32, 0.78169495f32, 0.7836046f32, 0.78550696f32, 0.78740185f32, 0.78928936f32, 0.79116946f32, 0.7930421f32, 0.7949073f32, 0.79676497f32, 0.79861516f32, 0.80045784f32, 0.80229294f32, 0.80412054f32, 0.80594057f32, 0.80775297f32, 0.8095578f32, 0.81135505f32, 0.8131446f32, 0.8149265f32, 0.81670076f32, 0.8184673f32, 0.8202262f32, 0.8219773f32, 0.8237207f32, 0.8254564f32, 0.82718426f32, 0.8289041f32, 0.83061635f32, 0.8323208f32, 0.83401746f32, 0.83570623f32, 0.8373872f32, 0.83906025f32, 0.84072536f32, 0.8423826f32, 0.8440319f32, 0.84567326f32, 0.84730667f32, 0.8489321f32, 0.8505495f32, 0.8521589f32, 0.8537603f32, 0.8553537f32, 0.856939f32, 0.8585163f32, 0.8600854f32, 0.86164653f32, 0.8631995f32, 0.8647443f32, 0.86628103f32, 0.86780953f32, 0.8693299f32, 0.87084216f32, 0.87234616f32, 0.87384194f32, 0.8753295f32, 0.8768088f32, 0.87827986f32, 0.8797427f32, 0.8811972f32, 0.88264346f32, 0.88408136f32, 0.885511f32, 0.88693225f32, 0.8883451f32, 0.8897497f32, 0.8911459f32, 0.89253366f32, 0.8939131f32, 0.89528406f32, 0.8966466f32, 0.8980007f32, 0.89934635f32, 0.9006836f32, 0.9020123f32, 0.90333253f32, 0.90464425f32, 0.90594745f32, 0.9072421f32, 0.90852827f32, 0.9098059f32, 0.9110749f32, 0.91233516f32, 0.91358703f32, 0.91483027f32, 0.916065f32, 0.917291f32, 0.9185084f32, 0.91971713f32, 0.9209172f32, 0.92210865f32, 0.92329144f32, 0.9244655f32, 0.92563087f32, 0.9267875f32, 0.9279354f32, 0.9290746f32, 0.93020505f32, 0.93132675f32, 0.9324397f32, 0.9335438f32, 0.93463916f32, 0.93572575f32, 0.93680346f32, 0.9378724f32, 0.93893254f32, 0.9399838f32, 0.9410262f32, 0.9420598f32, 0.9430845f32, 0.9441003f32, 0.9451073f32, 0.9461053f32, 0.94709444f32, 0.94807464f32, 0.94904596f32, 0.95000833f32, 0.95096177f32, 0.9519062f32, 0.9528417f32, 0.95376825f32, 0.9546858f32, 0.9555944f32, 0.956494f32, 0.9573846f32, 0.95826614f32, 0.9591387f32, 0.96000224f32, 0.96085674f32, 0.96170217f32, 0.96253854f32, 0.9633659f32, 0.96418417f32, 0.96499336f32, 0.96579343f32, 0.9665845f32, 0.9673664f32, 0.9681391f32, 0.96890277f32, 0.96965736f32, 0.97040284f32, 0.97113913f32, 0.9718663f32, 0.97258437f32, 0.97329324f32, 0.97399294f32, 0.9746835f32, 0.97536486f32, 0.9760371f32, 0.97670007f32, 0.97735393f32, 0.97799855f32, 0.97863394f32, 0.97926015f32, 0.9798771f32, 0.9804849f32, 0.9810834f32, 0.9816727f32, 0.9822528f32, 0.98282355f32, 0.98338515f32, 0.98393744f32, 0.9844805f32, 0.98501426f32, 0.9855388f32, 0.986054f32, 0.9865599f32, 0.9870566f32, 0.987544f32, 0.988022f32, 0.9884908f32, 0.9889503f32, 0.98940045f32, 0.98984134f32, 0.9902728f32, 0.99069506f32, 0.99110794f32, 0.9915115f32, 0.99190575f32, 0.9922906f32, 0.9926662f32, 0.9930324f32, 0.99338925f32, 0.99373674f32, 0.99407494f32, 0.9944037f32, 0.99472314f32, 0.99503326f32, 0.99533397f32, 0.9956253f32, 0.99590725f32, 0.9961799f32, 0.99644303f32, 0.9966969f32, 0.9969413f32, 0.9971764f32, 0.99740213f32, 0.99761844f32, 0.9978253f32, 0.99802285f32, 0.998211f32, 0.9983897f32, 0.99855906f32, 0.99871904f32, 0.99886954f32, 0.9990107f32, 0.9991424f32, 0.9992648f32, 0.99937767f32, 0.9994812f32, 0.9995753f32, 0.99966f32, 0.9997353f32, 0.99980116f32, 0.99985766f32, 0.9999047f32, 0.99994236f32, 0.9999706f32, 0.9999894f32, 0.9999988f32, ];
