# STM32WLxx Hardware Abstraction Layer

## Usage

```toml
[dependencies.stm32wlxx-hal]
version = "0.2.0"
features = [
    # use exactly one of the following depending on your target hardware
    "stm32wl5x_cm0p",
    "stm32wl5x_cm4",
    "stm32wle5",
    # optional: use the cortex-m-rt interrupt interface
    "rt",
    # optional: use defmt
    "defmt",
]

# include cortex-m-rt directly in your crate if you need interrupts
# use the interrupt macro from the hal with `use stm32wlxx_hal::pac::interrupt;`
# DO NOT use the interrupt macro from cortex-m-rt, it will fail to compile
[dependencies]
cortex-m-rt = "0.7"
```

**Note:** To avoid version mismatches do not include `cortex-m`, `embedded-hal`,
or `stm32wl` (the PAC) directly, these are re-exported by the hal.

```rust
use stm32wlxx_hal as hal;

use hal::cortex_m;
use hal::cortex_m_rt; // requires "rt" feature
use hal::embedded_hal;
use hal::pac; // published as "stm32wl" on crates.io
```

## Design

### Peripheral Access

The layout of device memory for the STM32WL is provided from the vendor in a
format called system view description (SVD).
The SVD is not perfect, so there is a set of community maintained SVD
patches at [stm32-rs].
After the SVD is patched it gets run through [svd2rust] which generates
the peripheral access crate (PAC) containing read/write/modify operations for
the device registers.

### Abstraction

_The fundamental theorem of software engineering_

> We can solve any problem by introducing an extra level of abstraction except
> for the problem of too many layers of abstraction.

You will be using the registers directly in some cases.
Not everything is abstracted, and even when this crate is complete some
functionality (e.g. CRC) may not be abstracted because the register interface is
simple enough. That being said if you find somthing missing it is likely
because this crate is incomplete, and not an intentional design decision.

```rust
use stm32wlxx_hal as hal;
use hal::pac;

use hal::{aes::Aes, pka::Pka};
use pac::{AES, PKA};

// The first thing you do is grab the peripherals from the PAC, this is a
// singleton, the `unwrap` call will panic only if you call this twice.
// This strategy prevents you from generating multiple peripheral drivers
// without the use on `unsafe`
let mut dp: pac::Peripherals = pac::Peripherals::take().unwrap();

// The drivers provided typically work by consuming the peripheral access
// structure generated by the PAC to provide structures with higher level
// functionality
let aes: Aes = Aes::new(dp.AES, &mut dp.RCC);
let pka: Pka = Pka::new(dp.PKA, &mut dp.RCC);
```

[stm32-rs]: https://github.com/stm32-rs/stm32-rs
[svd2rust]: https://github.com/rust-embedded/svd2rust
