//! Inter-Integrated Circuit (I2C) bus

use core::sync::atomic::{self, Ordering};

use crate::stm32::{I2C1, I2C3};
use cast::u8;

use crate::gpio::gpioa::{PA10, PA7, PA9};
use crate::gpio::gpiob::{PB10, PB11, PB13, PB14, PB4, PB6, PB7, PB8, PB9};
use crate::gpio::gpioc::{PC0, PC1};
use crate::gpio::{Alternate, OpenDrain, Output, AF4};
use crate::hal::blocking::i2c::{Read, Write, WriteIter, WriteIterRead, WriteRead};
use crate::rcc::Rcc;
use crate::time::Hertz;

use crate::dma::{Receive, RxDma, Transfer, TransferPayload, Transmit, TxDma, R, W};
use crate::dmamux::DmaMuxIndex;
use embedded_dma::{StaticReadBuffer, StaticWriteBuffer};
/// I2C error
#[derive(Debug)]
#[non_exhaustive]
pub enum Error {
    /// Bus error
    Bus,
    /// Arbitration loss
    Arbitration,
    /// NACK
    Nack,
    // Overrun, // slave mode only
    // Pec, // SMBUS mode only
    Timeout,

    /// An error occurred during DMA transfer.
    DmaTransferError,
    // Alert, // SMBUS mode only
}

// FIXME these should be "closed" traits
/// SCL pin -- DO NOT IMPLEMENT THIS TRAIT
pub unsafe trait SclPin<I2C> {}

/// SDA pin -- DO NOT IMPLEMENT THIS TRAIT
pub unsafe trait SdaPin<I2C> {}

unsafe impl SclPin<I2C1> for PA9<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C1> for PA10<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C1> for PB6<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C1> for PB7<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C1> for PB8<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C1> for PB9<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C3> for PA7<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C3> for PB4<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C3> for PB10<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C3> for PB11<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C3> for PB13<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C3> for PB14<Alternate<AF4, Output<OpenDrain>>> {}

unsafe impl SclPin<I2C3> for PC0<Alternate<AF4, Output<OpenDrain>>> {}
unsafe impl SdaPin<I2C3> for PC1<Alternate<AF4, Output<OpenDrain>>> {}

/// I2C peripheral operating in master mode
pub struct I2c<I2C, PINS> {
    i2c: I2C,
    pins: PINS,
}

macro_rules! busy_wait {
    ($i2c:expr, $flag:ident) => {
        let mut i = 0;

        loop {
            let isr = $i2c.isr.read();

            if isr.berr().bit_is_set() {
                return Err(Error::Bus);
            } else if isr.arlo().bit_is_set() {
                return Err(Error::Arbitration);
            } else if isr.nackf().bit_is_set() {
                return Err(Error::Nack);
            } else if isr.$flag().bit_is_set() {
                break;
            } else {
                // try again
                i += 1;

                if i > 1_000_000 {
                    return Err(Error::Timeout);
                }
            }
        }
    };
}

macro_rules! hal {
    ($($I2CX:ident: ($i2cX:ident, $i2cXen:ident, $i2cXrst:ident),)+) => {
        $(
            impl<SCL, SDA> I2c<$I2CX, (SCL, SDA)> {
                /// Configures the I2C peripheral to work in master mode
                /// by using raw timing data generated by CubeMX tool
                pub fn from_raw_timing(i2c: $I2CX, pins: (SCL, SDA), timing: u32) -> Self where
                    SCL: SclPin<$I2CX>,
                    SDA: SdaPin<$I2CX>,
                {
                    i2c.timingr.write(|w| unsafe {
                        w.bits(timing)
                    });

                    // Enable the peripheral
                    i2c.cr1.write(|w| w.pe().set_bit());

                    I2c { i2c, pins }
                }

                /// Configures the I2C peripheral to work in master mode
                pub fn $i2cX<F>(
                    i2c: $I2CX,
                    pins: (SCL, SDA),
                    freq: F,
                    rcc: &mut Rcc,
                ) -> Self where
                    F: Into<Hertz>,
                    SCL: SclPin<$I2CX>,
                    SDA: SdaPin<$I2CX>,
                {
                    // All I2Cs are located on APB1-related RCC registers
                    rcc.rb.apb1enr1.modify(|_, w| w.$i2cXen().set_bit());
                    rcc.rb.apb1rstr1.modify(|_, w| w.$i2cXrst().set_bit());
                    rcc.rb.apb1rstr1.modify(|_, w| w.$i2cXrst().clear_bit());

                    let freq = freq.into().0;

                    assert!(freq <= 1_000_000);

                    // TODO review compliance with the timing requirements of I2C
                    // t_I2CCLK = 1 / PCLK1
                    // t_PRESC  = (PRESC + 1) * t_I2CCLK
                    // t_SCLL   = (SCLL + 1) * t_PRESC
                    // t_SCLH   = (SCLH + 1) * t_PRESC
                    //
                    // t_SYNC1 + t_SYNC2 > 4 * t_I2CCLK
                    // t_SCL ~= t_SYNC1 + t_SYNC2 + t_SCLL + t_SCLH
                    let i2cclk = rcc.clocks.pclk1().0;
                    let ratio = i2cclk / freq - 4;
                    let (presc, scll, sclh, sdadel, scldel) = if freq >= 100_000 {
                        // fast-mode or fast-mode plus
                        // here we pick SCLL + 1 = 2 * (SCLH + 1)
                        let presc = ratio / 387;

                        let sclh = ((ratio / (presc + 1)) - 3) / 3;
                        let scll = 2 * (sclh + 1) - 1;

                        let (sdadel, scldel) = if freq > 400_000 {
                            // fast-mode plus
                            let sdadel = 0;
                            let scldel = i2cclk / 4_000_000 / (presc + 1) - 1;

                            (sdadel, scldel)
                        } else {
                            // fast-mode
                            let sdadel = i2cclk / 8_000_000 / (presc + 1);
                            let scldel = i2cclk / 2_000_000 / (presc + 1) - 1;

                            (sdadel, scldel)
                        };

                        (presc, scll, sclh, sdadel, scldel)
                    } else {
                        // standard-mode
                        // here we pick SCLL = SCLH
                        let presc = ratio / 514;

                        let sclh = ((ratio / (presc + 1)) - 2) / 2;
                        let scll = sclh;

                        let sdadel = i2cclk / 2_000_000 / (presc + 1);
                        let scldel = i2cclk / 800_000 / (presc + 1) - 1;

                        (presc, scll, sclh, sdadel, scldel)
                    };

                    let presc = u8(presc).unwrap();
                    assert!(presc < 16);
                    let scldel = u8(scldel).unwrap();
                    assert!(scldel < 16);
                    let sdadel = u8(sdadel).unwrap();
                    assert!(sdadel < 16);
                    let sclh = u8(sclh).unwrap();
                    let scll = u8(scll).unwrap();

                    // Configure for "fast mode" (400 KHz)
                    i2c.timingr.write(|w| unsafe {
                        w.presc()
                            .bits(presc)
                            .scll()
                            .bits(scll)
                            .sclh()
                            .bits(sclh)
                            .sdadel()
                            .bits(sdadel)
                            .scldel()
                            .bits(scldel)
                    });

                    // Enable clock stretching
                    i2c.cr1.modify(|_, w| w.nostretch().clear_bit());

                    // Enable the peripheral
                    i2c.cr1.write(|w| w.pe().set_bit());

                    I2c { i2c, pins }
                }

                /// Releases the I2C peripheral and associated pins
                pub fn free(self) -> ($I2CX, (SCL, SDA)) {
                    (self.i2c, self.pins)
                }

                /// Enables or disables interrupt for errors
                pub fn set_error_interrupt(&mut self, enabled: bool) {
                    self.i2c.cr1.modify(|_, w| w.errie().bit(enabled));
                }

                /// Clears errors that may trigger interrupt
                pub fn clear_error_interrupt(&mut self) {
                    self.i2c.icr.write(|w| w.berrcf().set_bit().arlocf().set_bit());
                }

                /// Enables or disables event interrupt associated with a NACK
                pub fn set_nack_interrupt(&mut self, enabled: bool) {
                    self.i2c.cr1.modify(|_, w| w.nackie().bit(enabled));
                }

                /// Clears NACK flag that may trigger interrupt
                pub fn clear_nack_interrupt(&mut self) {
                    self.i2c.icr.write(|w| w.nackcf().set_bit());
                }
            }

            impl<PINS> Write for I2c<$I2CX, PINS> {
                type Error = Error;

                fn write(&mut self, addr: u8, bytes: &[u8]) -> Result<(), Error> {
                    // TODO support transfers of more than 255 bytes
                    assert!(bytes.len() < 256 && bytes.len() > 0);

                    // START and prepare to send `bytes`
                    self.i2c.cr2.write(|w| unsafe {
                        w.sadd()
                            .bits((addr as u16) << 1)
                            .rd_wrn()
                            .clear_bit()
                            .nbytes()
                            .bits(bytes.len() as u8)
                            .start()
                            .set_bit()
                            .autoend()
                            .set_bit()
                    });

                    for byte in bytes {
                        // Wait until we are allowed to send data (START has been ACKed or last byte
                        // when through)
                        busy_wait!(self.i2c, txis);

                        // put byte on the wire
                        self.i2c.txdr.write(unsafe { |w| { w.txdata().bits(*byte) } });
                    }

                    // automatic STOP

                    Ok(())
                }
            }

            impl<PINS> WriteIter for I2c<$I2CX, PINS> {
                type Error = Error;

                fn write<B: IntoIterator<Item=u8>>(&mut self, addr: u8, bytes: B) -> Result<(), Error> {
                    let bytes = bytes.into_iter();

                    let mut bytes_allocated = [0u8; 255];
                    let mut len = 0;
                    for byte in bytes.into_iter() {
                        assert!(len < 256);
                        bytes_allocated[len] = byte;
                        len += 1;
                    }
                    Write::write(self, addr, &bytes_allocated[..len])
                }
            }

            impl<PINS> Read for I2c<$I2CX, PINS> {
                type Error = Error;

                fn read(&mut self,
                    addr: u8,
                    buffer: &mut [u8],) -> Result<(), Error> {
                    self.i2c.cr2.write(|w| unsafe {
                        w.sadd()
                            .bits((addr as u16) << 1)
                            .rd_wrn()
                            .set_bit()
                            .nbytes()
                            .bits(buffer.len() as u8)
                            .start()
                            .set_bit()
                            .autoend()
                            .set_bit()
                    });

                    for byte in buffer {
                        // Wait until we have received something
                        busy_wait!(self.i2c, rxne);

                        *byte = self.i2c.rxdr.read().rxdata().bits();
                    }

                    Ok(())
                }
            }

            impl<PINS> WriteRead for I2c<$I2CX, PINS> {
                type Error = Error;

                fn write_read(
                    &mut self,
                    addr: u8,
                    bytes: &[u8],
                    buffer: &mut [u8],
                ) -> Result<(), Error> {
                    // TODO support transfers of more than 255 bytes
                    assert!(bytes.len() < 256 && bytes.len() > 0);
                    assert!(buffer.len() < 256 && buffer.len() > 0);

                    // TODO do we have to explicitly wait here if the bus is busy (e.g. another
                    // master is communicating)?

                    // START and prepare to send `bytes`
                    self.i2c.cr2.write(|w| unsafe {
                        w.sadd()
                            .bits((addr as u16) << 1)
                            .rd_wrn()
                            .clear_bit()
                            .nbytes()
                            .bits(bytes.len() as u8)
                            .start()
                            .set_bit()
                            .autoend()
                            .clear_bit()
                    });

                    for byte in bytes {
                        // Wait until we are allowed to send data (START has been ACKed or last byte
                        // when through)
                        busy_wait!(self.i2c, txis);

                        // put byte on the wire
                        self.i2c.txdr.write(|w| unsafe { w.txdata().bits(*byte) });
                    }

                    // Wait until the last transmission is finished
                    busy_wait!(self.i2c, tc);

                    // reSTART and prepare to receive bytes into `buffer`
                    self.i2c.cr2.write(|w| unsafe {
                        w.sadd()
                            .bits((addr as u16) << 1)
                            .rd_wrn()
                            .set_bit()
                            .nbytes()
                            .bits(buffer.len() as u8)
                            .start()
                            .set_bit()
                            .autoend()
                            .set_bit()
                    });

                    for byte in buffer {
                        // Wait until we have received something
                        busy_wait!(self.i2c, rxne);

                        *byte = self.i2c.rxdr.read().rxdata().bits();
                    }

                    // automatic STOP - due to autoend

                    Ok(())
                }
            }

            impl<PINS> WriteIterRead for I2c<$I2CX, PINS> {
                type Error = Error;

                fn write_iter_read<B: IntoIterator<Item=u8>>(
                    &mut self,
                    addr: u8,
                    bytes: B,
                    buffer: &mut [u8]
                ) -> Result<(), Error> {
                    let bytes = bytes.into_iter();

                    let mut bytes_allocated = [0u8; 255];
                    let mut len = 0;
                    for byte in bytes.into_iter() {
                        assert!(len < 256);
                        bytes_allocated[len] = byte;
                        len += 1;
                    }
                    WriteRead::write_read(self, addr, &bytes_allocated[..len], buffer)
                }
            }
        )+
    }
}

hal! {
    I2C1: (i2c1, i2c1en, i2c1rst),
    I2C3: (i2c3, i2c3en, i2c3rst),
}

// DMA

pub struct I2cPayload<I2C, PINS> {
    i2c: I2c<I2C, PINS>,
    address: u8,
    autoend: bool,
}

pub type I2cTxDma<I2C, PINS, CHANNEL> = TxDma<I2cPayload<I2C, PINS>, CHANNEL>;
pub type I2cRxDma<I2C, PINS, CHANNEL> = RxDma<I2cPayload<I2C, PINS>, CHANNEL>;

macro_rules! i2c_dma {
    ($I2Ci:ident, $Ci:ident, $dmamuxTX:path, $dmamuxRX:path) => {
        impl<PINS> I2c<$I2Ci, PINS> {
            pub fn with_tx_dma(
                self,
                channel: $Ci,
                address: u8,
                autoend: bool,
            ) -> I2cTxDma<$I2Ci, PINS, $Ci> {
                let payload = I2cPayload {
                    i2c: self,
                    address,
                    autoend,
                };
                I2cTxDma { payload, channel }
            }

            pub fn with_rx_dma(
                self,
                channel: $Ci,
                address: u8,
                autoend: bool,
            ) -> I2cRxDma<$I2Ci, PINS, $Ci> {
                let payload = I2cPayload {
                    i2c: self,
                    address,
                    autoend,
                };
                I2cRxDma { payload, channel }
            }
        }

        // TX DMA

        impl<PINS> I2cTxDma<$I2Ci, PINS, $Ci> {
            pub fn free(self) -> ($Ci, I2c<$I2Ci, PINS>) {
                (self.channel, self.payload.i2c)
            }
        }

        impl<PINS> Transmit for I2cTxDma<$I2Ci, PINS, $Ci> {
            type TxChannel = $Ci;
            type ReceivedWord = u8;
        }

        impl<PINS> TransferPayload for I2cTxDma<$I2Ci, PINS, $Ci> {
            fn start(&mut self) {
                self.payload
                    .i2c
                    .i2c
                    .cr1
                    .modify(|_, w| w.txdmaen().set_bit());
                self.channel.start();
            }
            fn stop(&mut self) {
                self.payload
                    .i2c
                    .i2c
                    .cr1
                    .modify(|_, w| w.txdmaen().clear_bit());
                self.channel.stop();
            }
        }

        impl<B, PINS> crate::dma::WriteDma<B, u8> for I2cTxDma<$I2Ci, PINS, $Ci>
        where
            B: StaticReadBuffer<Word = u8>,
        {
            fn write(mut self, buffer: B) -> Transfer<R, B, Self> {
                // NOTE(unsafe) We own the buffer now and we won't call other `&mut` on it
                // until the end of the transfer.
                let (ptr, len) = unsafe { buffer.static_read_buffer() };
                atomic::compiler_fence(Ordering::Release);
                self.channel.set_circular_mode(false);
                self.channel.listen(crate::dma::Event::TransferComplete);
                self.channel.listen(crate::dma::Event::TransferError);
                self.channel.select_peripheral($dmamuxTX);
                self.channel.set_peripheral_address(
                    unsafe { &(*$I2Ci::ptr()).txdr as *const _ as u32 },
                    false,
                );
                self.channel.set_memory_address(ptr as u32, true);
                self.channel.set_transfer_length(len);
                self.channel.set_mem2mem(false);
                self.channel
                    .set_direction(crate::dma::Direction::FromMemory);
                self.channel
                    .set_priority_level(crate::dma::Priority::Medium);
                self.channel.set_word_size(crate::dma::WordSize::BITS8);

                // Prepare to send `bytes`
                let addr = self.payload.address;
                let autoend = self.payload.autoend;
                self.start();
                let i2c = &mut self.payload.i2c.i2c;
                i2c.cr2.write(|w| unsafe {
                    w.sadd()
                        .bits((addr as u16) << 1)
                        .rd_wrn()
                        .clear_bit()
                        .nbytes()
                        .bits(len as u8)
                        .start()
                        .set_bit()
                        .autoend()
                        .bit(autoend)
                });
                Transfer::r(buffer, self)
            }
        }

        // RX DMA

        impl<PINS> I2cRxDma<$I2Ci, PINS, $Ci> {
            pub fn free(self) -> ($Ci, I2c<$I2Ci, PINS>) {
                (self.channel, self.payload.i2c)
            }
        }

        impl<PINS> Receive for I2cRxDma<$I2Ci, PINS, $Ci> {
            type RxChannel = $Ci;
            type TransmittedWord = u8;
        }

        impl<PINS> TransferPayload for I2cRxDma<$I2Ci, PINS, $Ci> {
            fn start(&mut self) {
                self.payload
                    .i2c
                    .i2c
                    .cr1
                    .modify(|_, w| w.rxdmaen().set_bit());
                self.channel.start();
            }
            fn stop(&mut self) {
                self.payload
                    .i2c
                    .i2c
                    .cr1
                    .modify(|_, w| w.rxdmaen().clear_bit());
                self.channel.stop();
            }
        }

        impl<B, PINS> crate::dma::ReadDma<B, u8> for I2cRxDma<$I2Ci, PINS, $Ci>
        where
            B: StaticWriteBuffer<Word = u8>,
        {
            fn read(mut self, mut buffer: B) -> Transfer<W, B, Self> {
                // NOTE(unsafe) We own the buffer now and we won't call other `&mut` on it
                // until the end of the transfer.
                let (ptr, len) = unsafe { buffer.static_write_buffer() };
                atomic::compiler_fence(Ordering::Release);
                self.channel.set_circular_mode(false);
                self.channel.listen(crate::dma::Event::TransferComplete);
                self.channel.listen(crate::dma::Event::TransferError);
                self.channel.select_peripheral($dmamuxRX);
                self.channel.set_peripheral_address(
                    unsafe { &(*$I2Ci::ptr()).rxdr as *const _ as u32 },
                    false,
                );
                self.channel.set_memory_address(ptr as u32, true);
                self.channel.set_transfer_length(len);
                self.channel.set_mem2mem(false);
                self.channel
                    .set_direction(crate::dma::Direction::FromPeripheral);
                self.channel
                    .set_priority_level(crate::dma::Priority::Medium);
                self.channel.set_word_size(crate::dma::WordSize::BITS8);

                // Prepare to send `bytes`
                let addr = self.payload.address;
                let autoend = self.payload.autoend;
                self.start();
                let i2c = &mut self.payload.i2c.i2c;
                i2c.cr2.write(|w| unsafe {
                    w.sadd()
                        .bits((addr as u16) << 1)
                        .rd_wrn()
                        .set_bit()
                        .nbytes()
                        .bits(len as u8)
                        .start()
                        .set_bit()
                        .autoend()
                        .bit(autoend)
                });
                Transfer::w(buffer, self)
            }
        }
    };
}

use crate::dma::dma1impl::{C1, C2};

i2c_dma!(I2C1, C1, DmaMuxIndex::I2C1_TX, DmaMuxIndex::I2C1_RX);
i2c_dma!(I2C3, C2, DmaMuxIndex::I2C3_TX, DmaMuxIndex::I2C3_RX);
