#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! General-purpose timers
//!
//! Used by: stm32wl5x_cm0p, stm32wl5x_cm4, stm32wle5

use crate::{RWRegister, WORegister};
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// TIM16/TIM17 control register 1
pub mod CR1 {

    /// UIF status bit remapping
    pub mod UIFREMAP {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No remapping. UIF status bit is not copied to TIMx_CNT register bit 31
            pub const Disabled: u32 = 0b0;

            /// 0b1: Remapping enabled. UIF status bit is copied to TIMx_CNT register bit 31
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Clock division
    pub mod CKD {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: t_DTS = t_CK_INT
            pub const Div1: u32 = 0b00;

            /// 0b01: t_DTS = 2 × t_CK_INT
            pub const Div2: u32 = 0b01;

            /// 0b10: t_DTS = 4 × t_CK_INT
            pub const Div4: u32 = 0b10;
        }
    }

    /// Auto-reload preload enable
    pub mod ARPE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: TIMx_APRR register is not buffered
            pub const Disabled: u32 = 0b0;

            /// 0b1: TIMx_APRR register is buffered
            pub const Enabled: u32 = 0b1;
        }
    }

    /// One pulse mode
    pub mod OPM {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Not stopped at update event
            pub const NotStopped: u32 = 0b0;

            /// 0b1: Counter stops counting at next update event
            pub const Stopped: u32 = 0b1;
        }
    }

    /// Update request source
    pub mod URS {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Any of counter overflow/underflow, setting UG, or update through slave mode, generates an update interrupt or DMA request
            pub const AnyEvent: u32 = 0b0;

            /// 0b1: Only counter overflow/underflow generates an update interrupt or DMA request
            pub const CounterOnly: u32 = 0b1;
        }
    }

    /// Update disable
    pub mod UDIS {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Update event enabled
            pub const Enabled: u32 = 0b0;

            /// 0b1: Update event disabled
            pub const Disabled: u32 = 0b1;
        }
    }

    /// Counter enable
    pub mod CEN {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Counter disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Counter enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// TIM16/TIM17 control register 2
pub mod CR2 {

    /// OIS1N
    pub mod OIS1N {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1N=0 after a dead-time when MOE=0
            pub const Low: u32 = 0b0;

            /// 0b1: OC1N=1 after a dead-time when MOE=0
            pub const High: u32 = 0b1;
        }
    }

    /// OIS1
    pub mod OIS1 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1=0 (after a dead-time if OC1N is implemented) when MOE=0
            pub const Low: u32 = 0b0;

            /// 0b1: OC1=1 (after a dead-time if OC1N is implemented) when MOE=0
            pub const High: u32 = 0b1;
        }
    }

    /// CCDS
    pub mod CCDS {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CCx DMA request sent when CCx event occurs
            pub const OnCompare: u32 = 0b0;

            /// 0b1: CCx DMA request sent when update event occurs
            pub const OnUpdate: u32 = 0b1;
        }
    }

    /// CCUS
    pub mod CCUS {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Capture/compare are updated only by setting the COMG bit
            pub const Default: u32 = 0b0;

            /// 0b1: Capture/compare are updated by setting the COMG bit or when an rising edge occurs on TRGI
            pub const WithRisingEdge: u32 = 0b1;
        }
    }

    /// CCPC
    pub mod CCPC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CCxE, CCxNE and OCxM bits are not preloaded
            pub const NotPreloaded: u32 = 0b0;

            /// 0b1: CCxE, CCxNE and OCxM bits are preloaded
            pub const Preloaded: u32 = 0b1;
        }
    }
}

/// TIM16/TIM17 DMA/interrupt enable register
pub mod DIER {

    /// Capture/Compare 1 DMA request enable
    pub mod CC1DE {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC1 DMA request disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: CC1 DMA request enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Update DMA request enable
    pub mod UDE {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Update DMA request disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Update DMA request enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Break interrupt enable
    pub mod BIE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Break interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// COM interrupt enable
    pub mod COMIE {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: COM interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: COM interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 interrupt enable
    pub mod CC1IE {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC1 interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: CC1 interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Update interrupt enable
    pub mod UIE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Update interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Update interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// TIM16/TIM17 status register
pub mod SR {

    /// Capture/Compare 1 overcapture flag
    pub mod CC1OF {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No overcapture has been detected
            pub const NoOvercapture: u32 = 0b0;

            /// 0b1: The counter value has been captured in TIMx_CCRx register while CCxIF flag was already set
            pub const Overcapture: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b0: Clear flag
            pub const Clear: u32 = 0b0;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Break interrupt flag
    pub mod BIF {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No break event occurred
            pub const NoBreak: u32 = 0b0;

            /// 0b1: Break interrupt pending
            pub const Break: u32 = 0b1;
        }
        pub use super::CC1OF::W;
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// COM interrupt flag
    pub mod COMIF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No COM event occurred
            pub const NoCOM: u32 = 0b0;

            /// 0b1: COM interrupt pending
            pub const COM: u32 = 0b1;
        }
        pub use super::CC1OF::W;
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 1 interrupt flag
    pub mod CC1IF {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No campture/compare has been detected
            pub const NoMatch: u32 = 0b0;

            /// 0b1: If CC1 is an output: The content of the counter TIMx_CNT matches the content of the TIMx_CCR1 register. If CC1 is an input: The counter value has been captured in TIMx_CCR1 register
            pub const Match: u32 = 0b1;
        }
        pub use super::CC1OF::W;
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Update interrupt flag
    pub mod UIF {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No update occurred
            pub const Clear: u32 = 0b0;

            /// 0b1: Update interrupt pending.
            pub const UpdatePending: u32 = 0b1;
        }
    }
}

/// TIM16/TIM17 event generation register
pub mod EGR {

    /// Break generation
    pub mod BG {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: A break event is generated. MOE bit is cleared and BIF flag is set. Related interrupt or DMA transfer can occur if enabled
            pub const Trigger: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare control update generation
    pub mod COMG {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: When CCPC bit is set, it allows CCxE, CCxNE and OCxM bits to be updated
            pub const Trigger: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 1 generation
    pub mod CC1G {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: If CC1 is an output: CC1IF flag is set, Corresponding interrupt or DMA request is sent if enabled. If CC1 is an input: The current value of the counter is captured in TIMx_CCR1 register
            pub const Trigger: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Update generation
    pub mod UG {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: Re-initializes the timer counter and generates an update of the registers.
            pub const Update: u32 = 0b1;
        }
    }
}

/// CCMR1_Output and CCMR1_Input
/// CCMR1_Output: TIM16/TIM17 capture/compare mode register 1
/// CCMR1_Input: TIM16/TIM17 capture/compare mode register 1
pub mod CCMR1 {

    /// OC1M
    pub mod OC1M_3 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// OC1M
    pub mod OC1M {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: The comparison between the output compare register TIMx_CCRy and the counter TIMx_CNT has no effect on the outputs
            pub const Frozen: u32 = 0b000;

            /// 0b001: Set channel to active level on match. OCyREF signal is forced high when the counter matches the capture/compare register
            pub const ActiveOnMatch: u32 = 0b001;

            /// 0b010: Set channel to inactive level on match. OCyREF signal is forced low when the counter matches the capture/compare register
            pub const InactiveOnMatch: u32 = 0b010;

            /// 0b011: OCyREF toggles when TIMx_CNT=TIMx_CCRy
            pub const Toggle: u32 = 0b011;

            /// 0b100: OCyREF is forced low
            pub const ForceInactive: u32 = 0b100;

            /// 0b101: OCyREF is forced high
            pub const ForceActive: u32 = 0b101;

            /// 0b110: In upcounting, channel is active as long as TIMx_CNT<TIMx_CCRy else inactive. In downcounting, channel is inactive as long as TIMx_CNT>TIMx_CCRy else active
            pub const PwmMode1: u32 = 0b110;

            /// 0b111: Inversely to PwmMode1
            pub const PwmMode2: u32 = 0b111;
        }
    }

    /// OC1PE
    pub mod OC1PE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Preload register on CCRx disabled. New values written to CCRx are taken into account immediately
            pub const Disabled: u32 = 0b0;

            /// 0b1: Preload register on CCRx enabled. Preload value is loaded into active register on each update event
            pub const Enabled: u32 = 0b1;
        }
    }

    /// OC1FE
    pub mod OC1FE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Fast output disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Fast output enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// CC1S
    pub mod CC1S {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CCx channel is configured as output
            pub const Output: u32 = 0b00;

            /// 0b01: CCx channel is configured as input, ICx is mapped on TI1
            pub const TI1: u32 = 0b01;
        }
    }

    /// IC1F
    pub mod IC1F {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (4 bits: 0b1111 << 4)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: No filter, sampling is done at fDTS
            pub const NoFilter: u32 = 0b0000;

            /// 0b0001: fSAMPLING=fCK_INT, N=2
            pub const FCK_INT_N2: u32 = 0b0001;

            /// 0b0010: fSAMPLING=fCK_INT, N=4
            pub const FCK_INT_N4: u32 = 0b0010;

            /// 0b0011: fSAMPLING=fCK_INT, N=8
            pub const FCK_INT_N8: u32 = 0b0011;

            /// 0b0100: fSAMPLING=fDTS/2, N=6
            pub const FDTS_Div2_N6: u32 = 0b0100;

            /// 0b0101: fSAMPLING=fDTS/2, N=8
            pub const FDTS_Div2_N8: u32 = 0b0101;

            /// 0b0110: fSAMPLING=fDTS/4, N=6
            pub const FDTS_Div4_N6: u32 = 0b0110;

            /// 0b0111: fSAMPLING=fDTS/4, N=8
            pub const FDTS_Div4_N8: u32 = 0b0111;

            /// 0b1000: fSAMPLING=fDTS/8, N=6
            pub const FDTS_Div8_N6: u32 = 0b1000;

            /// 0b1001: fSAMPLING=fDTS/8, N=8
            pub const FDTS_Div8_N8: u32 = 0b1001;

            /// 0b1010: fSAMPLING=fDTS/16, N=5
            pub const FDTS_Div16_N5: u32 = 0b1010;

            /// 0b1011: fSAMPLING=fDTS/16, N=6
            pub const FDTS_Div16_N6: u32 = 0b1011;

            /// 0b1100: fSAMPLING=fDTS/16, N=8
            pub const FDTS_Div16_N8: u32 = 0b1100;

            /// 0b1101: fSAMPLING=fDTS/32, N=5
            pub const FDTS_Div32_N5: u32 = 0b1101;

            /// 0b1110: fSAMPLING=fDTS/32, N=6
            pub const FDTS_Div32_N6: u32 = 0b1110;

            /// 0b1111: fSAMPLING=fDTS/32, N=8
            pub const FDTS_Div32_N8: u32 = 0b1111;
        }
    }

    /// IC1PSC
    pub mod IC1PSC {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (2 bits: 0b11 << 2)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CCx channel is configured as output
            pub const Output: u32 = 0b00;

            /// 0b01: Capture is done once every 2 events
            pub const Capture_2: u32 = 0b01;

            /// 0b10: Capture is done once every 4 events
            pub const Capture_4: u32 = 0b10;

            /// 0b11: Capture is done once every 8 events
            pub const Capture_8: u32 = 0b11;
        }
    }
}

/// TIM16/TIM17 capture/compare enable register
pub mod CCER {

    /// Capture/Compare 1 complementary output polarity
    pub mod CC1NP {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OCxN active high
            pub const ActiveHigh: u32 = 0b0;

            /// 0b1: OCxN active low
            pub const ActiveLow: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 complementary output enable
    pub mod CC1NE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Complementary output disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Complementary output enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 output polarity
    pub mod CC1P {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Noninverted/rising edge
            pub const RisingEdge: u32 = 0b0;

            /// 0b1: Inverted/falling edge
            pub const FallingEdge: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 output enable
    pub mod CC1E {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Capture disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Capture enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// TIM16/TIM17 counter
pub mod CNT {

    /// UIF Copy
    pub mod UIFCPYorRes {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// CNT
    pub mod CNT {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 prescaler
pub mod PSC {

    /// Prescaler value
    pub mod PSC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 auto-reload register
pub mod ARR {

    /// Auto-reload value
    pub mod ARR {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 repetition counter register
pub mod RCR {

    /// Repetition counter value
    pub mod REP {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (8 bits: 0xff << 0)
        pub const mask: u32 = 0xff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 capture/compare register 1
pub mod CCR1 {

    /// Capture/Compare 1 value
    pub mod CCR1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 break and dead-time register
pub mod BDTR {

    /// Break Bidirectional
    pub mod BKBID {
        /// Offset (28 bits)
        pub const offset: u32 = 28;
        /// Mask (1 bit: 1 << 28)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK in input mode
            pub const Input: u32 = 0b0;

            /// 0b1: Break input BRK in bidirectional mode
            pub const Bidirectional: u32 = 0b1;
        }
    }

    /// Break Disarm
    pub mod BKDSRM {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (1 bit: 1 << 26)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK is armed
            pub const Armed: u32 = 0b0;

            /// 0b1: Break input BRK is disarmed
            pub const Disarmed: u32 = 0b1;
        }
    }

    /// Main output enable
    pub mod MOE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC and OCN outputs are disabled or forced to idle state depending on the OSSI bit
            pub const Disabled: u32 = 0b0;

            /// 0b1: OC and OCN outputs are enabled if their respective enable bits are set (CCxE, CCxNE in TIMx_CCER register)
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Automatic output enable
    pub mod AOE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: MOE can be set only by software
            pub const Disabled: u32 = 0b0;

            /// 0b1: MOE can be set by software or automatically at the next update event (if none of the break inputs BRK and BRK2 is active)
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Break polarity
    pub mod BKP {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK is active low
            pub const ActiveLow: u32 = 0b0;

            /// 0b1: Break input BRK is active high
            pub const ActiveHigh: u32 = 0b1;
        }
    }

    /// Break enable
    pub mod BKE {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break inputs (BRK and CCS clock failure event) disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Break inputs (BRK and CCS clock failure event) enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Off-state selection for Run mode
    pub mod OSSR {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC/OCN outputs are disabled when inactive
            pub const Disabled: u32 = 0b0;

            /// 0b1: OC/OCN outputs are enabled with their inactive level as soon as CCxE=1 or CCxNE=1
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Off-state selection for Idle mode
    pub mod OSSI {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC/OCN outputs are disabled when inactive
            pub const Disabled: u32 = 0b0;

            /// 0b1: OC/OCN outputs are first forced with their inactive level then forced to their idle level after the deadtime
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Lock configuration
    pub mod LOCK {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: No write protection
            pub const Off: u32 = 0b00;

            /// 0b01: Level 1 write protection
            pub const Level1: u32 = 0b01;

            /// 0b10: Level 2 write protection
            pub const Level2: u32 = 0b10;

            /// 0b11: Level 3 write protection
            pub const Level3: u32 = 0b11;
        }
    }

    /// Dead-time generator setup
    pub mod DTG {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (8 bits: 0xff << 0)
        pub const mask: u32 = 0xff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 DMA control register
pub mod DCR {

    /// DMA burst length
    pub mod DBL {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (5 bits: 0b11111 << 8)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// DMA base address
    pub mod DBA {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (5 bits: 0b11111 << 0)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16/TIM17 DMA address for full transfer
pub mod DMAR {

    /// DMA register for burst accesses
    pub mod DMAB {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM16 option register 1
pub mod OR1 {

    /// Timer 17 input 1 connection
    pub mod TI1_RMP {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: TI1 is connected to GPIO
            pub const GPIO: u32 = 0b00;

            /// 0b01: TI1 is connected to LSI
            pub const LSI: u32 = 0b01;

            /// 0b10: TI1 is connected to LSE
            pub const LSE: u32 = 0b10;

            /// 0b11: TI1 is connected to RTC wake-up interrupt
            pub const RTC: u32 = 0b11;
        }
    }
}

/// TIM16 alternate function register 1
pub mod AF1 {

    /// BRK COMP2 input polarity
    pub mod BKCMP2P {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Input polarity not inverted
            pub const NotInverted: u32 = 0b0;

            /// 0b1: Input polarity inverted
            pub const Inverted: u32 = 0b1;
        }
    }

    /// BRK COMP1 input polarity
    pub mod BKCMP1P {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::BKCMP2P::RW;
    }

    /// BRK BKIN input polarity
    pub mod BKINP {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::BKCMP2P::RW;
    }

    /// BRK COMP2 enable
    pub mod BKCMP2E {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: COMP2 input disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: COMP2 input enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// BRK COMP1 enable
    pub mod BKCMP1E {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: COMP1 input disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: COMP1 input enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// BRK BKIN input enable
    pub mod BKINE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: BKIN input disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: BKIN input enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// TIM16 input selection register
pub mod TISEL {

    /// TISEL
    pub mod TISEL {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: TIM1_CH1 input selected
            pub const Selected: u32 = 0b0000;
        }
    }
}
#[repr(C)]
pub struct RegisterBlock {
    /// TIM16/TIM17 control register 1
    pub CR1: RWRegister<u32>,

    /// TIM16/TIM17 control register 2
    pub CR2: RWRegister<u32>,

    _reserved1: [u32; 1],

    /// TIM16/TIM17 DMA/interrupt enable register
    pub DIER: RWRegister<u32>,

    /// TIM16/TIM17 status register
    pub SR: RWRegister<u32>,

    /// TIM16/TIM17 event generation register
    pub EGR: WORegister<u32>,

    /// CCMR1_Output and CCMR1_Input
    /// CCMR1_Output: TIM16/TIM17 capture/compare mode register 1
    /// CCMR1_Input: TIM16/TIM17 capture/compare mode register 1
    pub CCMR1: RWRegister<u32>,

    _reserved2: [u32; 1],

    /// TIM16/TIM17 capture/compare enable register
    pub CCER: RWRegister<u32>,

    /// TIM16/TIM17 counter
    pub CNT: RWRegister<u32>,

    /// TIM16/TIM17 prescaler
    pub PSC: RWRegister<u32>,

    /// TIM16/TIM17 auto-reload register
    pub ARR: RWRegister<u32>,

    /// TIM16/TIM17 repetition counter register
    pub RCR: RWRegister<u32>,

    /// TIM16/TIM17 capture/compare register 1
    pub CCR1: RWRegister<u32>,

    _reserved3: [u32; 3],

    /// TIM16/TIM17 break and dead-time register
    pub BDTR: RWRegister<u32>,

    /// TIM16/TIM17 DMA control register
    pub DCR: RWRegister<u32>,

    /// TIM16/TIM17 DMA address for full transfer
    pub DMAR: RWRegister<u32>,

    /// TIM16 option register 1
    pub OR1: RWRegister<u32>,

    _reserved4: [u32; 3],

    /// TIM16 alternate function register 1
    pub AF1: RWRegister<u32>,

    _reserved5: [u32; 1],

    /// TIM16 input selection register
    pub TISEL: RWRegister<u32>,
}
pub struct ResetValues {
    pub CR1: u32,
    pub CR2: u32,
    pub DIER: u32,
    pub SR: u32,
    pub EGR: u32,
    pub CCMR1: u32,
    pub CCER: u32,
    pub CNT: u32,
    pub PSC: u32,
    pub ARR: u32,
    pub RCR: u32,
    pub CCR1: u32,
    pub BDTR: u32,
    pub DCR: u32,
    pub DMAR: u32,
    pub OR1: u32,
    pub AF1: u32,
    pub TISEL: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}
