#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! Real-time clock
//!
//! Used by: stm32wl5x_cm0p, stm32wl5x_cm4, stm32wle5

use crate::{RORegister, RWRegister, WORegister};
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// Time register
pub mod TR {

    /// AM/PM notation
    pub mod PM {
        /// Offset (22 bits)
        pub const offset: u32 = 22;
        /// Mask (1 bit: 1 << 22)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: AM or 24-hour format
            pub const AM: u32 = 0b0;

            /// 0b1: PM
            pub const PM: u32 = 0b1;
        }
    }

    /// Hour tens in BCD format
    pub mod HT {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (2 bits: 0b11 << 20)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Hour units in BCD format
    pub mod HU {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Minute tens in BCD format
    pub mod MNT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Minute units in BCD format
    pub mod MNU {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Second tens in BCD format
    pub mod ST {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Second units in BCD format
    pub mod SU {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Date register
pub mod DR {

    /// Year tens in BCD format
    pub mod YT {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (4 bits: 0b1111 << 20)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Year units in BCD format
    pub mod YU {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Week day units
    pub mod WDU {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (3 bits: 0b111 << 13)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Month tens in BCD format
    pub mod MT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Month units in BCD format
    pub mod MU {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Date tens in BCD format
    pub mod DT {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (2 bits: 0b11 << 4)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Date units in BCD format
    pub mod DU {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Sub second register
pub mod SSR {

    /// Synchronous binary counter
    pub mod SS {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Initialization control and status register
pub mod ICSR {

    /// Recalibration pending Flag
    pub mod RECALPF {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b1: The RECALPF status flag is automatically set to 1 when software writes to the RTC_CALR register, indicating that the RTC_CALR register is blocked. When the new calibration settings are taken into account, this bit returns to 0
            pub const Pending: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// BCD update
    pub mod BCDU {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (3 bits: 0b111 << 10)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: 1s increment each time SS\[7:0\]=0
            pub const Bit7: u32 = 0b000;

            /// 0b001: 1s increment each time SS\[8:0\]=0
            pub const Bit8: u32 = 0b001;

            /// 0b010: 1s increment each time SS\[9:0\]=0
            pub const Bit9: u32 = 0b010;

            /// 0b011: 1s increment each time SS\[10:0\]=0
            pub const Bit10: u32 = 0b011;

            /// 0b100: 1s increment each time SS\[11:0\]=0
            pub const Bit11: u32 = 0b100;

            /// 0b101: 1s increment each time SS\[12:0\]=0
            pub const Bit12: u32 = 0b101;

            /// 0b110: 1s increment each time SS\[13:0\]=0
            pub const Bit13: u32 = 0b110;

            /// 0b111: 1s increment each time SS\[14:0\]=0
            pub const Bit14: u32 = 0b111;
        }
    }

    /// Binary mode
    pub mod BIN {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Free running BCD calendar mode (Binary mode disabled)
            pub const BCD: u32 = 0b00;

            /// 0b01: Free running Binary mode (BCD mode disabled)
            pub const Binary: u32 = 0b01;

            /// 0b10: Free running BCD calendar and Binary modes
            pub const BinBCD: u32 = 0b10;

            /// 0b11: Free running BCD calendar and Binary modes
            pub const BinBCD2: u32 = 0b11;
        }
    }

    /// Initialization mode
    pub mod INIT {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Free running mode
            pub const FreeRunningMode: u32 = 0b0;

            /// 0b1: Initialization mode used to program time and date register (RTC_TR and RTC_DR), and prescaler register (RTC_PRER). Counters are stopped and start counting from the new value when INIT is reset.
            pub const InitMode: u32 = 0b1;
        }
    }

    /// Initialization flag
    pub mod INITF {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Calendar registers update is not allowed
            pub const NotAllowed: u32 = 0b0;

            /// 0b1: Calendar registers update is allowed
            pub const Allowed: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Registers synchronization flag
    pub mod RSF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Calendar shadow registers not yet synchronized
            pub const NotSynced: u32 = 0b0;

            /// 0b1: Calendar shadow registers synchronized
            pub const Synced: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b0: This flag is cleared by software by writing 0
            pub const Clear: u32 = 0b0;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Initialization status flag
    pub mod INITS {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Calendar has not been initialized
            pub const NotInitalized: u32 = 0b0;

            /// 0b1: Calendar has been initialized
            pub const Initalized: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Shift operation pending
    pub mod SHPF {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No shift operation is pending
            pub const NoShiftPending: u32 = 0b0;

            /// 0b1: A shift operation is pending
            pub const ShiftPending: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Wakeup timer write flag
    pub mod WUTWF {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Wakeup timer configuration update not allowed
            pub const UpdateNotAllowed: u32 = 0b0;

            /// 0b1: Wakeup timer configuration update allowed
            pub const UpdateAllowed: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Pre-scaler register
pub mod PRER {

    /// Asynchronous prescaler factor
    pub mod PREDIV_A {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (7 bits: 0x7f << 16)
        pub const mask: u32 = 0x7f << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Synchronous prescaler factor
    pub mod PREDIV_S {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (15 bits: 0x7fff << 0)
        pub const mask: u32 = 0x7fff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Wakeup timer register
pub mod WUTR {

    /// Wakeup auto-reload output clear value
    pub mod WUTOCLR {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (16 bits: 0xffff << 16)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Wakeup auto-reload value bits
    pub mod WUT {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Control register
pub mod CR {

    /// RTC_OUT2 output enable
    pub mod OUT2EN {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: RTC output 2 disable
            pub const Disabled: u32 = 0b0;

            /// 0b1: RTC output 2 enable
            pub const Enabled: u32 = 0b1;
        }
    }

    /// TAMPALRM output type
    pub mod TAMPALRM_TYPE {
        /// Offset (30 bits)
        pub const offset: u32 = 30;
        /// Mask (1 bit: 1 << 30)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: TAMPALRM is push-pull output
            pub const PushPull: u32 = 0b0;

            /// 0b1: TAMPALRM is open-drain output
            pub const OpenDrain: u32 = 0b1;
        }
    }

    /// TAMPALRM pull-up enable
    pub mod TAMPALRM_PU {
        /// Offset (29 bits)
        pub const offset: u32 = 29;
        /// Mask (1 bit: 1 << 29)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No pull-up is applied on TAMPALRM output
            pub const NoPullUp: u32 = 0b0;

            /// 0b1: A pull-up is applied on TAMPALRM output
            pub const PullUp: u32 = 0b1;
        }
    }

    /// Tamper detection output enable on TAMPALRM
    pub mod TAMPOE {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (1 bit: 1 << 26)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: The tamper flag is not routed on TAMPALRM
            pub const Disabled: u32 = 0b0;

            /// 0b1: The tamper flag is routed on TAMPALRM, combined with the signal provided by OSEL and with the polarity provided by POL
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Activate timestamp on tamper detection event
    pub mod TAMPTS {
        /// Offset (25 bits)
        pub const offset: u32 = 25;
        /// Mask (1 bit: 1 << 25)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Tamper detection event does not cause a RTC timestamp to be saved
            pub const Disabled: u32 = 0b0;

            /// 0b1: Save RTC timestamp on tamper detection event
            pub const Enabled: u32 = 0b1;
        }
    }

    /// timestamp on internal event enable
    pub mod ITSE {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Internal event timestamp disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Internal event timestamp enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Calibration output enable
    pub mod COE {
        /// Offset (23 bits)
        pub const offset: u32 = 23;
        /// Mask (1 bit: 1 << 23)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calibration output disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Calibration output enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Output selection
    pub mod OSEL {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (2 bits: 0b11 << 21)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Output disabled
            pub const Disabled: u32 = 0b00;

            /// 0b01: Alarm A output enabled
            pub const AlarmA: u32 = 0b01;

            /// 0b10: Alarm B output enabled
            pub const AlarmB: u32 = 0b10;

            /// 0b11: Wakeup output enabled
            pub const Wakeup: u32 = 0b11;
        }
    }

    /// Output polarity
    pub mod POL {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (1 bit: 1 << 20)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: The pin is high when ALRAF/ALRBF/WUTF is asserted (depending on OSEL\[1:0\])
            pub const High: u32 = 0b0;

            /// 0b1: The pin is low when ALRAF/ALRBF/WUTF is asserted (depending on OSEL\[1:0\])
            pub const Low: u32 = 0b1;
        }
    }

    /// Calibration output selection
    pub mod COSEL {
        /// Offset (19 bits)
        pub const offset: u32 = 19;
        /// Mask (1 bit: 1 << 19)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calibration output is 512 Hz (with default prescaler setting)
            pub const CalFreq_512Hz: u32 = 0b0;

            /// 0b1: Calibration output is 1 Hz (with default prescaler setting)
            pub const CalFreq_1Hz: u32 = 0b1;
        }
    }

    /// Backup
    pub mod BKP {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (1 bit: 1 << 18)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Daylight Saving Time change has not been performed
            pub const DSTNotChanged: u32 = 0b0;

            /// 0b1: Daylight Saving Time change has been performed
            pub const DSTChanged: u32 = 0b1;
        }
    }

    /// Subtract 1 hour (winter time change)
    pub mod SUB1H {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: Subtracts 1 hour to the current time. This can be used for winter time change outside initialization mode
            pub const Sub1: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Add 1 hour (summer time change)
    pub mod ADD1H {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: Adds 1 hour to the current time. This can be used for summer time change outside initialization mode
            pub const Add1: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Timestamp interrupt enable
    pub mod TSIE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Time-stamp Interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Time-stamp Interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Wakeup timer interrupt enable
    pub mod WUTIE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Wakeup timer interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Wakeup timer interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Alarm B interrupt enable
    pub mod ALRBIE {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Alarm B Interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Alarm B Interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Alarm A interrupt enable
    pub mod ALRAIE {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Alarm A interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Alarm A interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// timestamp enable
    pub mod TSE {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Timestamp disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Timestamp enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Wakeup timer enable
    pub mod WUTE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Wakeup timer disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Wakeup timer enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Alarm B enable
    pub mod ALRBE {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Alarm B disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Alarm B enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Alarm A enable
    pub mod ALRAE {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Alarm A disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Alarm A enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// SSR underflow interrupt enable
    pub mod SSRUIE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: SSR underflow interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: SSR underflow interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Hour format
    pub mod FMT {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: 24 hour/day format
            pub const TwentyFourHour: u32 = 0b0;

            /// 0b1: AM/PM hour format
            pub const AmPm: u32 = 0b1;
        }
    }

    /// Bypass the shadow registers
    pub mod BYPSHAD {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calendar values (when reading from RTC_SSR, RTC_TR, and RTC_DR) are taken from the shadow registers, which are updated once every two RTCCLK cycles
            pub const ShadowReg: u32 = 0b0;

            /// 0b1: Calendar values (when reading from RTC_SSR, RTC_TR, and RTC_DR) are taken directly from the calendar counters
            pub const BypassShadowReg: u32 = 0b1;
        }
    }

    /// RTC_REFIN reference clock detection enable (50 or 60 Hz)
    pub mod REFCKON {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: RTC_REFIN detection disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: RTC_REFIN detection enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Timestamp event active edge
    pub mod TSEDGE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: RTC_TS input rising edge generates a time-stamp event
            pub const RisingEdge: u32 = 0b0;

            /// 0b1: RTC_TS input falling edge generates a time-stamp event
            pub const FallingEdge: u32 = 0b1;
        }
    }

    /// Wakeup clock selection
    pub mod WUCKSEL {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (3 bits: 0b111 << 0)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: RTC/16 clock is selected
            pub const Div16: u32 = 0b000;

            /// 0b001: RTC/8 clock is selected
            pub const Div8: u32 = 0b001;

            /// 0b010: RTC/4 clock is selected
            pub const Div4: u32 = 0b010;

            /// 0b011: RTC/2 clock is selected
            pub const Div2: u32 = 0b011;

            /// 0b100: ck_spre (usually 1 Hz) clock is selected
            pub const ClockSpare: u32 = 0b100;

            /// 0b110: ck_spre (usually 1 Hz) clock is selected and 2^16 is added to the WUT counter value
            pub const ClockSpareWithOffset: u32 = 0b110;
        }
    }
}

/// Write protection register
pub mod WPR {

    /// Write protection key
    pub mod KEY {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (8 bits: 0xff << 0)
        pub const mask: u32 = 0xff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b11001010: Key 1
            pub const Deactivate1: u32 = 0b11001010;

            /// 0b01010011: Key 2
            pub const Deactivate2: u32 = 0b01010011;

            /// 0b00000000: Activate write protection (any value that is not the keys)
            pub const Activate: u32 = 0b00000000;
        }
    }
}

/// Calibration register
pub mod CALR {

    /// Use an 8-second calibration cycle period
    pub mod CALP {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No RTCCLK pulses are added
            pub const NoChange: u32 = 0b0;

            /// 0b1: One RTCCLK pulse is effectively inserted every 2^11 pulses (frequency increased by 488.5 ppm)
            pub const IncreaseFreq: u32 = 0b1;
        }
    }

    /// Use a 16-second calibration cycle period
    pub mod CALW8 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: When CALW8 is set to ‘1’, the 8-second calibration cycle period is selected
            pub const EightSeconds: u32 = 0b1;
        }
    }

    /// CALW16
    pub mod CALW16 {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: When CALW16 is set to ‘1’, the 16-second calibration cycle period is selected.This bit must not be set to ‘1’ if CALW8=1
            pub const SixteenSeconds: u32 = 0b1;
        }
    }

    /// Calibration low-power mode
    pub mod LPCAL {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calibration window is 220 RTCCLK, which is a high-consumption mode. This mode should be set only when less than 32s calibration window is required
            pub const RTCCLK: u32 = 0b0;

            /// 0b1: Calibration window is 220 ck_apre, which is the required configuration for ultra-low consumption mode
            pub const CkApre: u32 = 0b1;
        }
    }

    /// Calibration minus
    pub mod CALM {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (9 bits: 0x1ff << 0)
        pub const mask: u32 = 0x1ff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Shift control register
pub mod SHIFTR {

    /// Add one second
    pub mod ADD1S {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: Add one second to the clock/calendar
            pub const Add1: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Subtract a fraction of a second
    pub mod SUBFS {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (15 bits: 0x7fff << 0)
        pub const mask: u32 = 0x7fff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Timestamp time register
pub mod TSTR {

    /// AM/PM notation
    pub mod PM {
        /// Offset (22 bits)
        pub const offset: u32 = 22;
        /// Mask (1 bit: 1 << 22)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Hour tens in BCD format.
    pub mod HT {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (2 bits: 0b11 << 20)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Hour units in BCD format.
    pub mod HU {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Minute tens in BCD format.
    pub mod MNT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Minute units in BCD format.
    pub mod MNU {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Second tens in BCD format.
    pub mod ST {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Second units in BCD format.
    pub mod SU {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Timestamp date register
pub mod TSDR {

    /// Week day units
    pub mod WDU {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (3 bits: 0b111 << 13)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Month tens in BCD format
    pub mod MT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Month units in BCD format
    pub mod MU {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Date tens in BCD format
    pub mod DT {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (2 bits: 0b11 << 4)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Date units in BCD format
    pub mod DU {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Timestamp sub second register
pub mod TSSSR {
    pub use super::SSR::SS;
}

/// Alarm A register
pub mod ALRMAR {

    /// Alarm A date mask
    pub mod MSK4 {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Alarm set if the date/day match
            pub const Mask: u32 = 0b0;

            /// 0b1: Date/day don’t care in Alarm comparison
            pub const NotMask: u32 = 0b1;
        }
    }

    /// Week day selection
    pub mod WDSEL {
        /// Offset (30 bits)
        pub const offset: u32 = 30;
        /// Mask (1 bit: 1 << 30)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: DU\[3:0\] represents the date units
            pub const DateUnits: u32 = 0b0;

            /// 0b1: DU\[3:0\] represents the week day. DT\[1:0\] is don’t care.
            pub const WeekDay: u32 = 0b1;
        }
    }

    /// Date tens in BCD format
    pub mod DT {
        /// Offset (28 bits)
        pub const offset: u32 = 28;
        /// Mask (2 bits: 0b11 << 28)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Date units or day in BCD format
    pub mod DU {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (4 bits: 0b1111 << 24)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Alarm A hours mask
    pub mod MSK3 {
        /// Offset (23 bits)
        pub const offset: u32 = 23;
        /// Mask (1 bit: 1 << 23)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::MSK4::RW;
    }

    /// AM/PM notation
    pub mod PM {
        /// Offset (22 bits)
        pub const offset: u32 = 22;
        /// Mask (1 bit: 1 << 22)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: AM or 24-hour format
            pub const AM: u32 = 0b0;

            /// 0b1: PM
            pub const PM: u32 = 0b1;
        }
    }

    /// Hour tens in BCD format
    pub mod HT {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (2 bits: 0b11 << 20)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Hour units in BCD format
    pub mod HU {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Alarm A minutes mask
    pub mod MSK2 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::MSK4::RW;
    }

    /// Minute tens in BCD format
    pub mod MNT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Minute units in BCD format
    pub mod MNU {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Alarm A seconds mask
    pub mod MSK1 {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::MSK4::RW;
    }

    /// Second tens in BCD format.
    pub mod ST {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Second units in BCD format.
    pub mod SU {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Alarm A sub second register
pub mod ALRMASSR {

    /// Clear synchronous counter on alarm (Binary mode only)
    pub mod SSCLR {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: The synchronous binary counter (SS\[31:0\] in RTC_SSR) is free-running
            pub const FreeRunning: u32 = 0b0;

            /// 0b1: The synchronous binary counter (SS\[31:0\] in RTC_SSR) is running from 0xFFFF FFFF to RTC_ALRMABINR → SS\[31:0\] value and is automatically reloaded with 0xFFFF FFFF when reaching RTC_ALRMABINR → SS\[31:0\]
            pub const ALRMBINR: u32 = 0b1;
        }
    }

    /// Mask the most-significant bits starting at this bit
    pub mod MASKSS {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (6 bits: 0x3f << 24)
        pub const mask: u32 = 0x3f << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Sub seconds value
    pub mod SS {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (15 bits: 0x7fff << 0)
        pub const mask: u32 = 0x7fff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// Alarm B register
pub mod ALRMBR {
    pub use super::ALRMAR::DT;
    pub use super::ALRMAR::DU;
    pub use super::ALRMAR::HT;
    pub use super::ALRMAR::HU;
    pub use super::ALRMAR::MNT;
    pub use super::ALRMAR::MNU;
    pub use super::ALRMAR::MSK1;
    pub use super::ALRMAR::MSK2;
    pub use super::ALRMAR::MSK3;
    pub use super::ALRMAR::MSK4;
    pub use super::ALRMAR::PM;
    pub use super::ALRMAR::ST;
    pub use super::ALRMAR::SU;
    pub use super::ALRMAR::WDSEL;
}

/// Alarm B sub second register
pub mod ALRMBSSR {
    pub use super::ALRMASSR::MASKSS;
    pub use super::ALRMASSR::SS;
    pub use super::ALRMASSR::SSCLR;
}

/// Status register (interrupts)
pub mod SR {

    /// SSR underflow flag
    pub mod SSRUF {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the SSR rolls under 0. SSRUF is not set when SSCLR=1
            pub const Underflow: u32 = 0b1;
        }
    }

    /// Internal timestamp flag
    pub mod ITSF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a timestamp on the internal event occurs
            pub const TimestampEvent: u32 = 0b1;
        }
    }

    /// Timestamp overflow flag
    pub mod TSOVF {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a time-stamp event occurs while TSF is already set
            pub const Overflow: u32 = 0b1;
        }
    }

    /// Timestamp flag
    pub mod TSF {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a time-stamp event occurs
            pub const TimestampEvent: u32 = 0b1;
        }
    }

    /// Wakeup timer flag
    pub mod WUTF {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the wakeup auto-reload counter reaches 0
            pub const Zero: u32 = 0b1;
        }
    }

    /// Alarm B flag
    pub mod ALRBF {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the time/date registers (RTC_TR and RTC_DR) match the Alarm B register (RTC_ALRMBR)
            pub const Match: u32 = 0b1;
        }
    }

    /// Alarm A flag
    pub mod ALRAF {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the time/date registers (RTC_TR and RTC_DR) match the Alarm A register (RTC_ALRMAR)
            pub const Match: u32 = 0b1;
        }
    }
}

/// Masked interrupt status register
pub mod MISR {

    /// SSR underflow masked flag
    pub mod SSRUMF {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the SSR rolls under 0. SSRUF is not set when SSCLR=1
            pub const Underflow: u32 = 0b1;
        }
    }

    /// Internal timestamp masked flag
    pub mod ITSMF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a timestamp on the internal event occurs
            pub const TimestampEvent: u32 = 0b1;
        }
    }

    /// Timestamp overflow masked flag
    pub mod TSOVMF {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a time-stamp event occurs while TSF is already set
            pub const Overflow: u32 = 0b1;
        }
    }

    /// Timestamp masked flag
    pub mod TSMF {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when a time-stamp event occurs
            pub const TimestampEvent: u32 = 0b1;
        }
    }

    /// Wakeup timer masked flag
    pub mod WUTMF {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the wakeup auto-reload counter reaches 0
            pub const Zero: u32 = 0b1;
        }
    }

    /// Alarm B masked flag
    pub mod ALRBMF {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the time/date registers (RTC_TR and RTC_DR) match the Alarm B register (RTC_ALRMBR)
            pub const Match: u32 = 0b1;
        }
    }

    /// Alarm A masked flag
    pub mod ALRAMF {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: This flag is set by hardware when the time/date registers (RTC_TR and RTC_DR) match the Alarm A register (RTC_ALRMAR)
            pub const Match: u32 = 0b1;
        }
    }
}

/// Status clear register (interrupts)
pub mod SCR {

    /// Clear SSR underflow flag
    pub mod CSSRUF {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: Clear interrupt flag by writing 1
            pub const Clear: u32 = 0b1;
        }
    }

    /// Clear internal timestamp flag
    pub mod CITSF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }

    /// Clear timestamp overflow flag
    pub mod CTSOVF {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }

    /// Clear timestamp flag
    pub mod CTSF {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }

    /// Clear wakeup timer flag
    pub mod CWUTF {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }

    /// Clear alarm B flag
    pub mod CALRBF {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }

    /// Clear alarm A flag
    pub mod CALRAF {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::CSSRUF::RW;
    }
}

/// RTC alarm A binary mode register
pub mod ALRABINR {

    /// Synchronous counter alarm value in Binary mode
    pub mod SS {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// RTC alarm B binary mode register
pub mod ALRBBINR {
    pub use super::ALRABINR::SS;
}
#[repr(C)]
pub struct RegisterBlock {
    /// Time register
    pub TR: RWRegister<u32>,

    /// Date register
    pub DR: RWRegister<u32>,

    /// Sub second register
    pub SSR: RORegister<u32>,

    /// Initialization control and status register
    pub ICSR: RWRegister<u32>,

    /// Pre-scaler register
    pub PRER: RWRegister<u32>,

    /// Wakeup timer register
    pub WUTR: RWRegister<u32>,

    /// Control register
    pub CR: RWRegister<u32>,

    _reserved1: [u32; 2],

    /// Write protection register
    pub WPR: WORegister<u32>,

    /// Calibration register
    pub CALR: RWRegister<u32>,

    /// Shift control register
    pub SHIFTR: WORegister<u32>,

    /// Timestamp time register
    pub TSTR: RORegister<u32>,

    /// Timestamp date register
    pub TSDR: RORegister<u32>,

    /// Timestamp sub second register
    pub TSSSR: RORegister<u32>,

    _reserved2: [u32; 1],

    /// Alarm A register
    pub ALRMAR: RWRegister<u32>,

    /// Alarm A sub second register
    pub ALRMASSR: RWRegister<u32>,

    /// Alarm B register
    pub ALRMBR: RWRegister<u32>,

    /// Alarm B sub second register
    pub ALRMBSSR: RWRegister<u32>,

    /// Status register (interrupts)
    pub SR: RORegister<u32>,

    /// Masked interrupt status register
    pub MISR: RORegister<u32>,

    _reserved3: [u32; 1],

    /// Status clear register (interrupts)
    pub SCR: WORegister<u32>,

    _reserved4: [u32; 4],

    /// RTC alarm A binary mode register
    pub ALRABINR: RWRegister<u32>,

    /// RTC alarm B binary mode register
    pub ALRBBINR: RWRegister<u32>,
}
pub struct ResetValues {
    pub TR: u32,
    pub DR: u32,
    pub SSR: u32,
    pub ICSR: u32,
    pub PRER: u32,
    pub WUTR: u32,
    pub CR: u32,
    pub WPR: u32,
    pub CALR: u32,
    pub SHIFTR: u32,
    pub TSTR: u32,
    pub TSDR: u32,
    pub TSSSR: u32,
    pub ALRMAR: u32,
    pub ALRMASSR: u32,
    pub ALRMBR: u32,
    pub ALRMBSSR: u32,
    pub SR: u32,
    pub MISR: u32,
    pub SCR: u32,
    pub ALRABINR: u32,
    pub ALRBBINR: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}
