#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! FMC
//!
//! Used by: stm32l552, stm32l562

#[cfg(not(feature = "nosync"))]
pub use crate::stm32l5::peripherals::fmc::Instance;
pub use crate::stm32l5::peripherals::fmc::{RegisterBlock, ResetValues};
pub use crate::stm32l5::peripherals::fmc::{
    FMC_BCR1, FMC_BCR2, FMC_BCR3, FMC_BCR4, FMC_BTR1, FMC_BTR2, FMC_BTR3, FMC_BTR4, FMC_BWTR1,
    FMC_BWTR2, FMC_BWTR3, FMC_BWTR4, FMC_ECCR, FMC_PATT, FMC_PCR, FMC_PMEM, FMC_SR, PCSCNTR,
};

/// Access functions for the FMC peripheral instance
pub mod FMC {
    use super::ResetValues;

    #[cfg(not(feature = "nosync"))]
    use super::Instance;

    #[cfg(not(feature = "nosync"))]
    const INSTANCE: Instance = Instance {
        addr: 0x44020000,
        _marker: ::core::marker::PhantomData,
    };

    /// Reset values for each field in FMC
    pub const reset: ResetValues = ResetValues {
        FMC_BCR1: 0x000030DB,
        FMC_BCR2: 0x000030D2,
        FMC_BCR3: 0x000030D2,
        FMC_BCR4: 0x000030D2,
        FMC_BTR1: 0x0FFFFFFF,
        FMC_BTR2: 0x0FFFFFFF,
        FMC_BTR3: 0x0FFFFFFF,
        FMC_BTR4: 0x0FFFFFFF,
        FMC_PCR: 0x00000018,
        FMC_SR: 0x00000040,
        FMC_PMEM: 0xFCFCFCFC,
        FMC_PATT: 0xFCFCFCFC,
        FMC_ECCR: 0x00000000,
        FMC_BWTR1: 0x0FFFFFFF,
        FMC_BWTR2: 0x0FFFFFFF,
        FMC_BWTR3: 0x0FFFFFFF,
        FMC_BWTR4: 0x0FFFFFFF,
        PCSCNTR: 0x00000000,
    };

    #[cfg(not(feature = "nosync"))]
    #[allow(renamed_and_removed_lints)]
    #[allow(private_no_mangle_statics)]
    #[no_mangle]
    static mut FMC_TAKEN: bool = false;

    /// Safe access to FMC
    ///
    /// This function returns `Some(Instance)` if this instance is not
    /// currently taken, and `None` if it is. This ensures that if you
    /// do get `Some(Instance)`, you are ensured unique access to
    /// the peripheral and there cannot be data races (unless other
    /// code uses `unsafe`, of course). You can then pass the
    /// `Instance` around to other functions as required. When you're
    /// done with it, you can call `release(instance)` to return it.
    ///
    /// `Instance` itself dereferences to a `RegisterBlock`, which
    /// provides access to the peripheral's registers.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn take() -> Option<Instance> {
        external_cortex_m::interrupt::free(|_| unsafe {
            if FMC_TAKEN {
                None
            } else {
                FMC_TAKEN = true;
                Some(INSTANCE)
            }
        })
    }

    /// Release exclusive access to FMC
    ///
    /// This function allows you to return an `Instance` so that it
    /// is available to `take()` again. This function will panic if
    /// you return a different `Instance` or if this instance is not
    /// already taken.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn release(inst: Instance) {
        external_cortex_m::interrupt::free(|_| unsafe {
            if FMC_TAKEN && inst.addr == INSTANCE.addr {
                FMC_TAKEN = false;
            } else {
                panic!("Released a peripheral which was not taken");
            }
        });
    }

    /// Unsafely steal FMC
    ///
    /// This function is similar to take() but forcibly takes the
    /// Instance, marking it as taken irregardless of its previous
    /// state.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub unsafe fn steal() -> Instance {
        FMC_TAKEN = true;
        INSTANCE
    }
}

/// Raw pointer to FMC
///
/// Dereferencing this is unsafe because you are not ensured unique
/// access to the peripheral, so you may encounter data races with
/// other users of this peripheral. It is up to you to ensure you
/// will not cause data races.
///
/// This constant is provided for ease of use in unsafe code: you can
/// simply call for example `write_reg!(gpio, GPIOA, ODR, 1);`.
pub const FMC: *const RegisterBlock = 0x44020000 as *const _;

/// Access functions for the SEC_FMC peripheral instance
pub mod SEC_FMC {
    use super::ResetValues;

    #[cfg(not(feature = "nosync"))]
    use super::Instance;

    #[cfg(not(feature = "nosync"))]
    const INSTANCE: Instance = Instance {
        addr: 0x54020000,
        _marker: ::core::marker::PhantomData,
    };

    /// Reset values for each field in SEC_FMC
    pub const reset: ResetValues = ResetValues {
        FMC_BCR1: 0x000030DB,
        FMC_BCR2: 0x000030D2,
        FMC_BCR3: 0x000030D2,
        FMC_BCR4: 0x000030D2,
        FMC_BTR1: 0x0FFFFFFF,
        FMC_BTR2: 0x0FFFFFFF,
        FMC_BTR3: 0x0FFFFFFF,
        FMC_BTR4: 0x0FFFFFFF,
        FMC_PCR: 0x00000018,
        FMC_SR: 0x00000040,
        FMC_PMEM: 0xFCFCFCFC,
        FMC_PATT: 0xFCFCFCFC,
        FMC_ECCR: 0x00000000,
        FMC_BWTR1: 0x0FFFFFFF,
        FMC_BWTR2: 0x0FFFFFFF,
        FMC_BWTR3: 0x0FFFFFFF,
        FMC_BWTR4: 0x0FFFFFFF,
        PCSCNTR: 0x00000000,
    };

    #[cfg(not(feature = "nosync"))]
    #[allow(renamed_and_removed_lints)]
    #[allow(private_no_mangle_statics)]
    #[no_mangle]
    static mut SEC_FMC_TAKEN: bool = false;

    /// Safe access to SEC_FMC
    ///
    /// This function returns `Some(Instance)` if this instance is not
    /// currently taken, and `None` if it is. This ensures that if you
    /// do get `Some(Instance)`, you are ensured unique access to
    /// the peripheral and there cannot be data races (unless other
    /// code uses `unsafe`, of course). You can then pass the
    /// `Instance` around to other functions as required. When you're
    /// done with it, you can call `release(instance)` to return it.
    ///
    /// `Instance` itself dereferences to a `RegisterBlock`, which
    /// provides access to the peripheral's registers.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn take() -> Option<Instance> {
        external_cortex_m::interrupt::free(|_| unsafe {
            if SEC_FMC_TAKEN {
                None
            } else {
                SEC_FMC_TAKEN = true;
                Some(INSTANCE)
            }
        })
    }

    /// Release exclusive access to SEC_FMC
    ///
    /// This function allows you to return an `Instance` so that it
    /// is available to `take()` again. This function will panic if
    /// you return a different `Instance` or if this instance is not
    /// already taken.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn release(inst: Instance) {
        external_cortex_m::interrupt::free(|_| unsafe {
            if SEC_FMC_TAKEN && inst.addr == INSTANCE.addr {
                SEC_FMC_TAKEN = false;
            } else {
                panic!("Released a peripheral which was not taken");
            }
        });
    }

    /// Unsafely steal SEC_FMC
    ///
    /// This function is similar to take() but forcibly takes the
    /// Instance, marking it as taken irregardless of its previous
    /// state.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub unsafe fn steal() -> Instance {
        SEC_FMC_TAKEN = true;
        INSTANCE
    }
}

/// Raw pointer to SEC_FMC
///
/// Dereferencing this is unsafe because you are not ensured unique
/// access to the peripheral, so you may encounter data races with
/// other users of this peripheral. It is up to you to ensure you
/// will not cause data races.
///
/// This constant is provided for ease of use in unsafe code: you can
/// simply call for example `write_reg!(gpio, GPIOA, ODR, 1);`.
pub const SEC_FMC: *const RegisterBlock = 0x54020000 as *const _;
