#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! Analog to Digital Converter
//!
//! Used by: stm32h743, stm32h753

use crate::{RORegister, RWRegister};
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// ADC interrupt and status register
pub mod ISR {

    /// ADC group injected contexts queue overflow flag
    pub mod JQOVF {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No injected context queue overflow has occurred
            pub const NoOverflow: u32 = 0b0;

            /// 0b1: Injected context queue overflow has occurred
            pub const Overflow: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear injected context queue overflow flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC analog watchdog 3 flag
    pub mod AWD3 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No analog watchdog event occurred
            pub const NoEvent: u32 = 0b0;

            /// 0b1: Analog watchdog event occurred
            pub const Event: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear analog watchdog event occurred flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC analog watchdog 2 flag
    pub mod AWD2 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        pub use super::AWD3::R;
        pub use super::AWD3::W;
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC analog watchdog 1 flag
    pub mod AWD1 {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        pub use super::AWD3::R;
        pub use super::AWD3::W;
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected end of sequence conversions flag
    pub mod JEOS {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Injected sequence is not complete
            pub const NotComplete: u32 = 0b0;

            /// 0b1: Injected sequence complete
            pub const Complete: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear Injected sequence complete flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected end of unitary conversion flag
    pub mod JEOC {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Injected conversion is not complete
            pub const NotComplete: u32 = 0b0;

            /// 0b1: Injected conversion complete
            pub const Complete: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear injected conversion complete flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular overrun flag
    pub mod OVR {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No overrun occurred
            pub const NoOverrun: u32 = 0b0;

            /// 0b1: Overrun occurred
            pub const Overrun: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear overrun occurred flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular end of sequence conversions flag
    pub mod EOS {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Regular sequence is not complete
            pub const NotComplete: u32 = 0b0;

            /// 0b1: Regular sequence complete
            pub const Complete: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear regular sequence complete flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular end of unitary conversion flag
    pub mod EOC {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: Regular conversion is not complete
            pub const NotComplete: u32 = 0b0;

            /// 0b1: Regular conversion complete
            pub const Complete: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear regular conversion complete flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular end of sampling flag
    pub mod EOSMP {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: End of sampling phase no yet reached
            pub const NotEnded: u32 = 0b0;

            /// 0b1: End of sampling phase reached
            pub const Ended: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear end of sampling phase reached flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC ready flag
    pub mod ADRDY {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: ADC is not ready to start conversion
            pub const NotReady: u32 = 0b0;

            /// 0b1: ADC is ready to start conversion
            pub const Ready: u32 = 0b1;
        }
        /// Write-only values
        pub mod W {

            /// 0b1: Clear ADC is ready to start conversion flag
            pub const Clear: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC interrupt enable register
pub mod IER {

    /// ADC group injected contexts queue overflow interrupt
    pub mod JQOVFIE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Injected context queue overflow interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Injected context queue overflow interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 3 interrupt
    pub mod AWD3IE {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Analog watchdog interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Analog watchdog interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 2 interrupt
    pub mod AWD2IE {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3IE::RW;
    }

    /// ADC analog watchdog 1 interrupt
    pub mod AWD1IE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3IE::RW;
    }

    /// ADC group injected end of sequence conversions interrupt
    pub mod JEOSIE {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: End of injected sequence interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: End of injected sequence interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group injected end of unitary conversion interrupt
    pub mod JEOCIE {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: End of injected conversion interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: End of injected conversion interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group regular overrun interrupt
    pub mod OVRIE {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Overrun interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Overrun interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group regular end of sequence conversions interrupt
    pub mod EOSIE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: End of regular sequence interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: End of regular sequence interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group regular end of unitary conversion interrupt
    pub mod EOCIE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: End of regular conversion interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: End of regular conversion interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group regular end of sampling interrupt
    pub mod EOSMPIE {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: End of regular conversion sampling phase interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: End of regular conversion sampling phase interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC ready interrupt
    pub mod ADRDYIE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: ADC ready interrupt disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: ADC ready interrupt enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// ADC control register
pub mod CR {

    /// ADC calibration
    pub mod ADCAL {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calibration complete
            pub const Complete: u32 = 0b0;

            /// 0b1: Start the calibration of the ADC
            pub const Calibration: u32 = 0b1;
        }
    }

    /// ADC differential mode for calibration
    pub mod ADCALDIF {
        /// Offset (30 bits)
        pub const offset: u32 = 30;
        /// Mask (1 bit: 1 << 30)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Calibration for single-ended mode
            pub const SingleEnded: u32 = 0b0;

            /// 0b1: Calibration for differential mode
            pub const Differential: u32 = 0b1;
        }
    }

    /// ADC deep power down enable
    pub mod DEEPPWD {
        /// Offset (29 bits)
        pub const offset: u32 = 29;
        /// Mask (1 bit: 1 << 29)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: ADC not in deep power down
            pub const PowerUp: u32 = 0b0;

            /// 0b1: ADC in deep power down
            pub const PowerDown: u32 = 0b1;
        }
    }

    /// ADC voltage regulator enable
    pub mod ADVREGEN {
        /// Offset (28 bits)
        pub const offset: u32 = 28;
        /// Mask (1 bit: 1 << 28)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: ADC voltage regulator disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: ADC voltage regulator enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Linearity calibration ready Word 6
    pub mod LINCALRDYW6 {
        /// Offset (27 bits)
        pub const offset: u32 = 27;
        /// Mask (1 bit: 1 << 27)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: LINCALFACT Word Read
            pub const Reset: u32 = 0b0;

            /// 0b1: LINCALFACT Word Write
            pub const Set: u32 = 0b1;
        }
    }

    /// Linearity calibration ready Word 5
    pub mod LINCALRDYW5 {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (1 bit: 1 << 26)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::LINCALRDYW6::RW;
    }

    /// Linearity calibration ready Word 4
    pub mod LINCALRDYW4 {
        /// Offset (25 bits)
        pub const offset: u32 = 25;
        /// Mask (1 bit: 1 << 25)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::LINCALRDYW6::RW;
    }

    /// Linearity calibration ready Word 3
    pub mod LINCALRDYW3 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::LINCALRDYW6::RW;
    }

    /// Linearity calibration ready Word 2
    pub mod LINCALRDYW2 {
        /// Offset (23 bits)
        pub const offset: u32 = 23;
        /// Mask (1 bit: 1 << 23)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::LINCALRDYW6::RW;
    }

    /// Linearity calibration ready Word 1
    pub mod LINCALRDYW1 {
        /// Offset (22 bits)
        pub const offset: u32 = 22;
        /// Mask (1 bit: 1 << 22)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::LINCALRDYW6::RW;
    }

    /// Linearity calibration
    pub mod ADCALLIN {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: ADC calibration without linearaity calibration
            pub const NoLinearity: u32 = 0b0;

            /// 0b1: ADC calibration with linearaity calibration
            pub const Linearity: u32 = 0b1;
        }
    }

    /// Boost mode control
    pub mod BOOST {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Boost mode off. Used when ADC clock < 20MHz
            pub const Off: u32 = 0b0;

            /// 0b1: Boost mode on. Used when ADC clock > 20MHz
            pub const On: u32 = 0b1;
        }
    }

    /// ADC group injected conversion stop
    pub mod JADSTP {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: Stop conversion of channel
            pub const Stop: u32 = 0b1;
        }
    }

    /// ADC group regular conversion stop
    pub mod ADSTP {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::JADSTP::RW;
    }

    /// ADC group injected conversion start
    pub mod JADSTART {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b1: Starts conversion of channel
            pub const Start: u32 = 0b1;
        }
    }

    /// ADC group regular conversion start
    pub mod ADSTART {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::JADSTART::RW;
    }

    /// ADC disable
    pub mod ADDIS {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b0: Disable ADC conversion and go to power down mode
            pub const Disable: u32 = 0b0;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC enable
    pub mod ADEN {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values
        pub mod W {

            /// 0b1: Enable ADC
            pub const Enable: u32 = 0b1;
        }
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC configuration register 1
pub mod CFGR {

    /// ADC group injected contexts queue disable
    pub mod JQDIS {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Injected Queue enabled
            pub const Enabled: u32 = 0b0;

            /// 0b1: Injected Queue disabled
            pub const Disabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 1 monitored channel selection
    pub mod AWD1CH {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (5 bits: 0b11111 << 26)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected automatic trigger mode
    pub mod JAUTO {
        /// Offset (25 bits)
        pub const offset: u32 = 25;
        /// Mask (1 bit: 1 << 25)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Automatic injected group conversion disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Automatic injected group conversion enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 1 enable on scope ADC group injected
    pub mod JAWD1EN {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Analog watchdog 1 disabled on injected channels
            pub const Disabled: u32 = 0b0;

            /// 0b1: Analog watchdog 1 enabled on injected channels
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 1 enable on scope ADC group regular
    pub mod AWD1EN {
        /// Offset (23 bits)
        pub const offset: u32 = 23;
        /// Mask (1 bit: 1 << 23)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Analog watchdog 1 disabled on regular channels
            pub const Disabled: u32 = 0b0;

            /// 0b1: Analog watchdog 1 enabled on regular channels
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 1 monitoring a single channel or all channels
    pub mod AWD1SGL {
        /// Offset (22 bits)
        pub const offset: u32 = 22;
        /// Mask (1 bit: 1 << 22)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Analog watchdog 1 enabled on all channels
            pub const All: u32 = 0b0;

            /// 0b1: Analog watchdog 1 enabled on single channel selected in AWD1CH
            pub const Single: u32 = 0b1;
        }
    }

    /// ADC group injected contexts queue mode
    pub mod JQM {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (1 bit: 1 << 21)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: JSQR Mode 0: Queue maintains the last written configuration into JSQR
            pub const Mode0: u32 = 0b0;

            /// 0b1: JSQR Mode 1: An empty queue disables software and hardware triggers of the injected sequence
            pub const Mode1: u32 = 0b1;
        }
    }

    /// ADC group injected sequencer discontinuous mode
    pub mod JDISCEN {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (1 bit: 1 << 20)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Discontinuous mode on injected channels disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Discontinuous mode on injected channels enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC group regular sequencer discontinuous number of ranks
    pub mod DISCNUM {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (3 bits: 0b111 << 17)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer discontinuous mode
    pub mod DISCEN {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Discontinuous mode on regular channels disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Discontinuous mode on regular channels enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC low power auto wait
    pub mod AUTDLY {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Auto delayed conversion mode off
            pub const Off: u32 = 0b0;

            /// 0b1: Auto delayed conversion mode on
            pub const On: u32 = 0b1;
        }
    }

    /// ADC group regular continuous conversion mode
    pub mod CONT {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Single conversion mode
            pub const Single: u32 = 0b0;

            /// 0b1: Continuous conversion mode
            pub const Continuous: u32 = 0b1;
        }
    }

    /// ADC group regular overrun configuration
    pub mod OVRMOD {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Preserve DR register when an overrun is detected
            pub const Preserve: u32 = 0b0;

            /// 0b1: Overwrite DR register when an overrun is detected
            pub const Overwrite: u32 = 0b1;
        }
    }

    /// ADC group regular external trigger polarity
    pub mod EXTEN {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (2 bits: 0b11 << 10)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Trigger detection disabled
            pub const Disabled: u32 = 0b00;

            /// 0b01: Trigger detection on the rising edge
            pub const RisingEdge: u32 = 0b01;

            /// 0b10: Trigger detection on the falling edge
            pub const FallingEdge: u32 = 0b10;

            /// 0b11: Trigger detection on both the rising and falling edges
            pub const BothEdges: u32 = 0b11;
        }
    }

    /// ADC group regular external trigger source
    pub mod EXTSEL {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (5 bits: 0b11111 << 5)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00101: Timer 4 CC4 event
            pub const TIM4_CC4: u32 = 0b00101;

            /// 0b00111: Timer 8 TRGO event
            pub const TIM8_TRGO: u32 = 0b00111;

            /// 0b01000: Timer 8 TRGO2 event
            pub const TIM8_TRGO2: u32 = 0b01000;

            /// 0b01100: Timer 4 TRGO event
            pub const TIM4_TRGO: u32 = 0b01100;

            /// 0b10000: HRTIM1_ADCTRG1 event
            pub const HRTIM1_ADCTRG1: u32 = 0b10000;

            /// 0b10001: HRTIM1_ADCTRG3 event
            pub const HRTIM1_ADCTRG3: u32 = 0b10001;

            /// 0b10010: LPTIM1_OUT event
            pub const LPTIM1_OUT: u32 = 0b10010;

            /// 0b10011: LPTIM2_OUT event
            pub const LPTIM2_OUT: u32 = 0b10011;

            /// 0b10100: LPTIM3_OUT event
            pub const LPTIM3_OUT: u32 = 0b10100;

            /// 0b00000: Timer 1 CC1 event
            pub const TIM1_CC1: u32 = 0b00000;

            /// 0b00001: Timer 1 CC2 event
            pub const TIM1_CC2: u32 = 0b00001;

            /// 0b00010: Timer 1 CC3 event
            pub const TIM1_CC3: u32 = 0b00010;

            /// 0b00011: Timer 2 CC2 event
            pub const TIM2_CC2: u32 = 0b00011;

            /// 0b00100: Timer 3 TRGO event
            pub const TIM3_TRGO: u32 = 0b00100;

            /// 0b00110: EXTI line 11
            pub const EXTI11: u32 = 0b00110;

            /// 0b01001: Timer 1 TRGO event
            pub const TIM1_TRGO: u32 = 0b01001;

            /// 0b01010: Timer 1 TRGO2 event
            pub const TIM1_TRGO2: u32 = 0b01010;

            /// 0b01011: Timer 2 TRGO event
            pub const TIM2_TRGO: u32 = 0b01011;

            /// 0b01101: Timer 6 TRGO event
            pub const TIM6_TRGO: u32 = 0b01101;

            /// 0b01110: Timer 15 TRGO event
            pub const TIM15_TRGO: u32 = 0b01110;

            /// 0b01111: Timer 3 CC4 event
            pub const TIM3_CC4: u32 = 0b01111;
        }
    }

    /// ADC data resolution
    pub mod RES {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (3 bits: 0b111 << 2)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: 16-bit resolution
            pub const SixteenBit: u32 = 0b000;

            /// 0b001: 14-bit resolution
            pub const FourteenBit: u32 = 0b001;

            /// 0b010: 12-bit resolution
            pub const TwelveBit: u32 = 0b010;

            /// 0b011: 10-bit resolution
            pub const TenBit: u32 = 0b011;

            /// 0b100: 8-bit resolution
            pub const EightBit: u32 = 0b100;
        }
    }

    /// ADC DMA transfer enable
    pub mod DMNGT {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Store output data in DR only
            pub const DR: u32 = 0b00;

            /// 0b01: DMA One Shot Mode selected
            pub const DMA_OneShot: u32 = 0b01;

            /// 0b10: DFSDM mode selected
            pub const DFSDM: u32 = 0b10;

            /// 0b11: DMA Circular Mode selected
            pub const DMA_Circular: u32 = 0b11;
        }
    }
}

/// ADC configuration register 2
pub mod CFGR2 {

    /// ADC oversampler enable on scope ADC group regular
    pub mod ROVSE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Regular oversampling disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Regular oversampling enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC oversampler enable on scope ADC group injected
    pub mod JOVSE {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Injected oversampling disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Injected oversampling enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC oversampling shift
    pub mod OVSS {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (4 bits: 0b1111 << 5)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC oversampling discontinuous mode (triggered mode) for ADC group regular
    pub mod TROVS {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: All oversampled conversions for a channel are run following a trigger
            pub const Automatic: u32 = 0b0;

            /// 0b1: Each oversampled conversion for a channel needs a new trigger
            pub const Triggered: u32 = 0b1;
        }
    }

    /// Regular Oversampling mode
    pub mod ROVSM {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Oversampling is temporary stopped and continued after injection sequence
            pub const Continued: u32 = 0b0;

            /// 0b1: Oversampling is aborted and resumed from start after injection sequence
            pub const Resumed: u32 = 0b1;
        }
    }

    /// Right-shift data after Offset 1 correction
    pub mod RSHIFT1 {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Right-shifting disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Data is right-shifted 1-bit
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Right-shift data after Offset 2 correction
    pub mod RSHIFT2 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::RSHIFT1::RW;
    }

    /// Right-shift data after Offset 3 correction
    pub mod RSHIFT3 {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::RSHIFT1::RW;
    }

    /// Right-shift data after Offset 4 correction
    pub mod RSHIFT4 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::RSHIFT1::RW;
    }

    /// Oversampling ratio
    pub mod OSVR {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (10 bits: 0x3ff << 16)
        pub const mask: u32 = 0x3ff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Left shift factor
    pub mod LSHIFT {
        /// Offset (28 bits)
        pub const offset: u32 = 28;
        /// Mask (4 bits: 0b1111 << 28)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC sampling time register 1
pub mod SMPR1 {

    /// ADC channel 9 sampling time selection
    pub mod SMP9 {
        /// Offset (27 bits)
        pub const offset: u32 = 27;
        /// Mask (3 bits: 0b111 << 27)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: 1.5 ADC clock cycles
            pub const Cycles1_5: u32 = 0b000;

            /// 0b001: 2.5 ADC clock cycles
            pub const Cycles2_5: u32 = 0b001;

            /// 0b010: 8.5 ADC clock cycles
            pub const Cycles8_5: u32 = 0b010;

            /// 0b011: 16.5 ADC clock cycles
            pub const Cycles16_5: u32 = 0b011;

            /// 0b100: 32.5 ADC clock cycles
            pub const Cycles32_5: u32 = 0b100;

            /// 0b101: 64.5 ADC clock cycles
            pub const Cycles64_5: u32 = 0b101;

            /// 0b110: 387.5 ADC clock cycles
            pub const Cycles387_5: u32 = 0b110;

            /// 0b111: 810.5 ADC clock cycles
            pub const Cycles810_5: u32 = 0b111;
        }
    }

    /// ADC channel 8 sampling time selection
    pub mod SMP8 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (3 bits: 0b111 << 24)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 7 sampling time selection
    pub mod SMP7 {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (3 bits: 0b111 << 21)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 6 sampling time selection
    pub mod SMP6 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (3 bits: 0b111 << 18)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 5 sampling time selection
    pub mod SMP5 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (3 bits: 0b111 << 15)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 4 sampling time selection
    pub mod SMP4 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 3 sampling time selection
    pub mod SMP3 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (3 bits: 0b111 << 9)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 2 sampling time selection
    pub mod SMP2 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (3 bits: 0b111 << 6)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 1 sampling time selection
    pub mod SMP1 {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (3 bits: 0b111 << 3)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }

    /// ADC channel 0 sampling time
    pub mod SMP0 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (3 bits: 0b111 << 0)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP9::RW;
    }
}

/// ADC sampling time register 2
pub mod SMPR2 {

    /// ADC channel 19 sampling time
    pub mod SMP19 {
        /// Offset (27 bits)
        pub const offset: u32 = 27;
        /// Mask (3 bits: 0b111 << 27)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: 1.5 ADC clock cycles
            pub const Cycles1_5: u32 = 0b000;

            /// 0b001: 2.5 ADC clock cycles
            pub const Cycles2_5: u32 = 0b001;

            /// 0b010: 8.5 ADC clock cycles
            pub const Cycles8_5: u32 = 0b010;

            /// 0b011: 16.5 ADC clock cycles
            pub const Cycles16_5: u32 = 0b011;

            /// 0b100: 32.5 ADC clock cycles
            pub const Cycles32_5: u32 = 0b100;

            /// 0b101: 64.5 ADC clock cycles
            pub const Cycles64_5: u32 = 0b101;

            /// 0b110: 387.5 ADC clock cycles
            pub const Cycles387_5: u32 = 0b110;

            /// 0b111: 810.5 ADC clock cycles
            pub const Cycles810_5: u32 = 0b111;
        }
    }

    /// ADC channel 18 sampling time selection
    pub mod SMP18 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (3 bits: 0b111 << 24)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 17 sampling time selection
    pub mod SMP17 {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (3 bits: 0b111 << 21)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 16 sampling time selection
    pub mod SMP16 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (3 bits: 0b111 << 18)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 15 sampling time selection
    pub mod SMP15 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (3 bits: 0b111 << 15)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 14 sampling time selection
    pub mod SMP14 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 13 sampling time selection
    pub mod SMP13 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (3 bits: 0b111 << 9)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 12 sampling time selection
    pub mod SMP12 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (3 bits: 0b111 << 6)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 11 sampling time selection
    pub mod SMP11 {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (3 bits: 0b111 << 3)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }

    /// ADC channel 10 sampling time selection
    pub mod SMP10 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (3 bits: 0b111 << 0)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::SMP19::RW;
    }
}

/// ADC analog watchdog 1 threshold register
pub mod LTR1 {

    /// ADC analog watchdog 1 threshold low
    pub mod LTR1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC analog watchdog 2 threshold register
pub mod HTR1 {

    /// ADC analog watchdog 2 threshold low
    pub mod HTR1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group regular sequencer ranks register 1
pub mod SQR1 {

    /// ADC group regular sequencer rank 4
    pub mod SQ4 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (5 bits: 0b11111 << 24)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 3
    pub mod SQ3 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (5 bits: 0b11111 << 18)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 2
    pub mod SQ2 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (5 bits: 0b11111 << 12)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 1
    pub mod SQ1 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (5 bits: 0b11111 << 6)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// L3
    pub mod L {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group regular sequencer ranks register 2
pub mod SQR2 {

    /// ADC group regular sequencer rank 9
    pub mod SQ9 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (5 bits: 0b11111 << 24)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 8
    pub mod SQ8 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (5 bits: 0b11111 << 18)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 7
    pub mod SQ7 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (5 bits: 0b11111 << 12)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 6
    pub mod SQ6 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (5 bits: 0b11111 << 6)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 5
    pub mod SQ5 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (5 bits: 0b11111 << 0)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group regular sequencer ranks register 3
pub mod SQR3 {

    /// ADC group regular sequencer rank 14
    pub mod SQ14 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (5 bits: 0b11111 << 24)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 13
    pub mod SQ13 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (5 bits: 0b11111 << 18)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 12
    pub mod SQ12 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (5 bits: 0b11111 << 12)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 11
    pub mod SQ11 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (5 bits: 0b11111 << 6)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 10
    pub mod SQ10 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (5 bits: 0b11111 << 0)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group regular sequencer ranks register 4
pub mod SQR4 {

    /// ADC group regular sequencer rank 16
    pub mod SQ16 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (5 bits: 0b11111 << 6)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group regular sequencer rank 15
    pub mod SQ15 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (5 bits: 0b11111 << 0)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group regular conversion data register
pub mod DR {

    /// ADC group regular conversion data
    pub mod RDATA {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group injected sequencer register
pub mod JSQR {

    /// ADC group injected sequencer rank 4
    pub mod JSQ4 {
        /// Offset (27 bits)
        pub const offset: u32 = 27;
        /// Mask (5 bits: 0b11111 << 27)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected sequencer rank 3
    pub mod JSQ3 {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (5 bits: 0b11111 << 21)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected sequencer rank 2
    pub mod JSQ2 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (5 bits: 0b11111 << 15)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected sequencer rank 1
    pub mod JSQ1 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (5 bits: 0b11111 << 9)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC group injected external trigger polarity
    pub mod JEXTEN {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (2 bits: 0b11 << 7)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Trigger detection disabled
            pub const Disabled: u32 = 0b00;

            /// 0b01: Trigger detection on the rising edge
            pub const RisingEdge: u32 = 0b01;

            /// 0b10: Trigger detection on the falling edge
            pub const FallingEdge: u32 = 0b10;

            /// 0b11: Trigger detection on both the rising and falling edges
            pub const BothEdges: u32 = 0b11;
        }
    }

    /// ADC group injected external trigger source
    pub mod JEXTSEL {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (5 bits: 0b11111 << 2)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00101: Timer 4 TRGO event
            pub const TIM4_TRGO: u32 = 0b00101;

            /// 0b00111: Timer 8 CC4 event
            pub const TIM8_CC4: u32 = 0b00111;

            /// 0b01001: Timer 8 TRGO event
            pub const TIM8_TRGO: u32 = 0b01001;

            /// 0b01010: Timer 8 TRGO2 event
            pub const TIM8_TRGO2: u32 = 0b01010;

            /// 0b10000: HRTIM1_ADCTRG2 event
            pub const HRTIM1_ADCTRG2: u32 = 0b10000;

            /// 0b10001: HRTIM1_ADCTRG4 event
            pub const HRTIM1_ADCTRG4: u32 = 0b10001;

            /// 0b10010: LPTIM1_OUT event
            pub const LPTIM1_OUT: u32 = 0b10010;

            /// 0b10011: LPTIM2_OUT event
            pub const LPTIM2_OUT: u32 = 0b10011;

            /// 0b10100: LPTIM3_OUT event
            pub const LPTIM3_OUT: u32 = 0b10100;

            /// 0b00000: Timer 1 TRGO event
            pub const TIM1_TRGO: u32 = 0b00000;

            /// 0b00001: Timer 1 CC4 event
            pub const TIM1_CC4: u32 = 0b00001;

            /// 0b00010: Timer 2 TRGO event
            pub const TIM2_TRGO: u32 = 0b00010;

            /// 0b00011: Timer 2 CC1 event
            pub const TIM2_CC1: u32 = 0b00011;

            /// 0b00100: Timer 3 CC4 event
            pub const TIM3_CC4: u32 = 0b00100;

            /// 0b00110: EXTI line 15
            pub const EXTI15: u32 = 0b00110;

            /// 0b01000: Timer 1 TRGO2 event
            pub const TIM1_TRGO2: u32 = 0b01000;

            /// 0b01011: Timer 3 CC3 event
            pub const TIM3_CC3: u32 = 0b01011;

            /// 0b01100: Timer 3 TRGO event
            pub const TIM3_TRGO: u32 = 0b01100;

            /// 0b01101: Timer 3 CC1 event
            pub const TIM3_CC1: u32 = 0b01101;

            /// 0b01110: Timer 6 TRGO event
            pub const TIM6_TRGO: u32 = 0b01110;

            /// 0b01111: Timer 15 TRGO event
            pub const TIM15_TRGO: u32 = 0b01111;
        }
    }

    /// ADC group injected sequencer scan length
    pub mod JL {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC offset number 1 register
pub mod OFR1 {

    /// Signed saturation enable
    pub mod SSATE {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Offset is subtracted maintaining data integrity and extending result size (9-bit and 17-bit signed format)
            pub const Disabled: u32 = 0b0;

            /// 0b1: Offset is subtracted and result is saturated to maintain result size
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC offset number 1 channel selection
    pub mod OFFSET1_CH {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (5 bits: 0b11111 << 26)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC offset number 1 offset level
    pub mod OFFSET1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC offset number 2 register
pub mod OFR2 {

    /// Signed saturation enable
    pub mod SSATE {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Offset is subtracted maintaining data integrity and extending result size (9-bit and 17-bit signed format)
            pub const Disabled: u32 = 0b0;

            /// 0b1: Offset is subtracted and result is saturated to maintain result size
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC offset number 1 channel selection
    pub mod OFFSET2_CH {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (5 bits: 0b11111 << 26)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC offset number 1 offset level
    pub mod OFFSET2 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC offset number 3 register
pub mod OFR3 {

    /// Signed saturation enable
    pub mod SSATE {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Offset is subtracted maintaining data integrity and extending result size (9-bit and 17-bit signed format)
            pub const Disabled: u32 = 0b0;

            /// 0b1: Offset is subtracted and result is saturated to maintain result size
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC offset number 1 channel selection
    pub mod OFFSET3_CH {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (5 bits: 0b11111 << 26)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC offset number 1 offset level
    pub mod OFFSET3 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC offset number 4 register
pub mod OFR4 {

    /// Signed saturation enable
    pub mod SSATE {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Offset is subtracted maintaining data integrity and extending result size (9-bit and 17-bit signed format)
            pub const Disabled: u32 = 0b0;

            /// 0b1: Offset is subtracted and result is saturated to maintain result size
            pub const Enabled: u32 = 0b1;
        }
    }

    /// ADC offset number 1 channel selection
    pub mod OFFSET4_CH {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (5 bits: 0b11111 << 26)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC offset number 1 offset level
    pub mod OFFSET4 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group injected sequencer rank 1 register
pub mod JDR1 {

    /// ADC group injected sequencer rank 1 conversion data
    pub mod JDATA1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group injected sequencer rank 2 register
pub mod JDR2 {

    /// ADC group injected sequencer rank 2 conversion data
    pub mod JDATA2 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group injected sequencer rank 3 register
pub mod JDR3 {

    /// ADC group injected sequencer rank 3 conversion data
    pub mod JDATA3 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC group injected sequencer rank 4 register
pub mod JDR4 {

    /// ADC group injected sequencer rank 4 conversion data
    pub mod JDATA4 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC analog watchdog 2 configuration register
pub mod AWD2CR {

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH0 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Input channel not monitored by AWDx
            pub const NotMonitored: u32 = 0b0;

            /// 0b1: Input channel monitored by AWDx
            pub const Monitored: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH1 {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH2 {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH3 {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH4 {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH5 {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH6 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH7 {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH8 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH9 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH10 {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH11 {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH12 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH13 {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH14 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH15 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH16 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH17 {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH18 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (1 bit: 1 << 18)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }

    /// ADC analog watchdog 2 monitored channel selection
    pub mod AWD2CH19 {
        /// Offset (19 bits)
        pub const offset: u32 = 19;
        /// Mask (1 bit: 1 << 19)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD2CH0::RW;
    }
}

/// ADC analog watchdog 3 configuration register
pub mod AWD3CR {

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH0 {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Input channel not monitored by AWDx
            pub const NotMonitored: u32 = 0b0;

            /// 0b1: Input channel monitored by AWDx
            pub const Monitored: u32 = 0b1;
        }
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH1 {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH2 {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH3 {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH4 {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH5 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH6 {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH7 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH8 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH9 {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH10 {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH11 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH12 {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH13 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH14 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH15 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH16 {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH17 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (1 bit: 1 << 18)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH18 {
        /// Offset (19 bits)
        pub const offset: u32 = 19;
        /// Mask (1 bit: 1 << 19)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }

    /// ADC analog watchdog 3 monitored channel selection
    pub mod AWD3CH19 {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (1 bit: 1 << 20)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::AWD3CH0::RW;
    }
}

/// ADC channel differential or single-ended mode selection register
pub mod DIFSEL {

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL0 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Input channel is configured in single-ended mode
            pub const SingleEnded: u32 = 0b0;

            /// 0b1: Input channel is configured in differential mode
            pub const Differential: u32 = 0b1;
        }
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL1 {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL2 {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL3 {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL4 {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL5 {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL6 {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL7 {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL8 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL9 {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL10 {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL11 {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL12 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL13 {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL14 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL15 {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL16 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL17 {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL18 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (1 bit: 1 << 18)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }

    /// ADC channel differential or single-ended mode for channel
    pub mod DIFSEL19 {
        /// Offset (19 bits)
        pub const offset: u32 = 19;
        /// Mask (1 bit: 1 << 19)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::DIFSEL0::RW;
    }
}

/// ADC calibration factors register
pub mod CALFACT {

    /// ADC calibration factor in differential mode
    pub mod CALFACT_D {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (11 bits: 0x7ff << 16)
        pub const mask: u32 = 0x7ff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// ADC calibration factor in single-ended mode
    pub mod CALFACT_S {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (11 bits: 0x7ff << 0)
        pub const mask: u32 = 0x7ff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC pre channel selection register
pub mod PCSEL {

    /// Channel x (VINP\[i\]) pre selection
    pub mod PCSEL {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (20 bits: 0xfffff << 0)
        pub const mask: u32 = 0xfffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00000000000000000000: Input channel x is not pre-selected
            pub const NotPreselected: u32 = 0b00000000000000000000;

            /// 0b00000000000000000001: Pre-select input channel x
            pub const Preselected: u32 = 0b00000000000000000001;
        }
    }
}

/// ADC watchdog lower threshold register 2
pub mod LTR2 {

    /// Analog watchdog 2 lower threshold
    pub mod LTR2 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC watchdog higher threshold register 2
pub mod HTR2 {

    /// Analog watchdog 2 higher threshold
    pub mod HTR2 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC watchdog lower threshold register 3
pub mod LTR3 {

    /// Analog watchdog 3 lower threshold
    pub mod LTR3 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC watchdog higher threshold register 3
pub mod HTR3 {

    /// Analog watchdog 3 higher threshold
    pub mod HTR3 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (26 bits: 0x3ffffff << 0)
        pub const mask: u32 = 0x3ffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// ADC Calibration Factor register 2
pub mod CALFACT2 {

    /// Linearity Calibration Factor
    pub mod LINCALFACT {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (30 bits: 0x3fffffff << 0)
        pub const mask: u32 = 0x3fffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}
#[repr(C)]
pub struct RegisterBlock {
    /// ADC interrupt and status register
    pub ISR: RWRegister<u32>,

    /// ADC interrupt enable register
    pub IER: RWRegister<u32>,

    /// ADC control register
    pub CR: RWRegister<u32>,

    /// ADC configuration register 1
    pub CFGR: RWRegister<u32>,

    /// ADC configuration register 2
    pub CFGR2: RWRegister<u32>,

    /// ADC sampling time register 1
    pub SMPR1: RWRegister<u32>,

    /// ADC sampling time register 2
    pub SMPR2: RWRegister<u32>,

    /// ADC pre channel selection register
    pub PCSEL: RWRegister<u32>,

    /// ADC analog watchdog 1 threshold register
    pub LTR1: RWRegister<u32>,

    /// ADC analog watchdog 2 threshold register
    pub HTR1: RWRegister<u32>,

    _reserved1: [u32; 2],

    /// ADC group regular sequencer ranks register 1
    pub SQR1: RWRegister<u32>,

    /// ADC group regular sequencer ranks register 2
    pub SQR2: RWRegister<u32>,

    /// ADC group regular sequencer ranks register 3
    pub SQR3: RWRegister<u32>,

    /// ADC group regular sequencer ranks register 4
    pub SQR4: RWRegister<u32>,

    /// ADC group regular conversion data register
    pub DR: RORegister<u32>,

    _reserved2: [u32; 2],

    /// ADC group injected sequencer register
    pub JSQR: RWRegister<u32>,

    _reserved3: [u32; 4],

    /// ADC offset number 1 register
    pub OFR1: RWRegister<u32>,

    /// ADC offset number 2 register
    pub OFR2: RWRegister<u32>,

    /// ADC offset number 3 register
    pub OFR3: RWRegister<u32>,

    /// ADC offset number 4 register
    pub OFR4: RWRegister<u32>,

    _reserved4: [u32; 4],

    /// ADC group injected sequencer rank 1 register
    pub JDR1: RORegister<u32>,

    /// ADC group injected sequencer rank 2 register
    pub JDR2: RORegister<u32>,

    /// ADC group injected sequencer rank 3 register
    pub JDR3: RORegister<u32>,

    /// ADC group injected sequencer rank 4 register
    pub JDR4: RORegister<u32>,

    _reserved5: [u32; 4],

    /// ADC analog watchdog 2 configuration register
    pub AWD2CR: RWRegister<u32>,

    /// ADC analog watchdog 3 configuration register
    pub AWD3CR: RWRegister<u32>,

    _reserved6: [u32; 2],

    /// ADC watchdog lower threshold register 2
    pub LTR2: RWRegister<u32>,

    /// ADC watchdog higher threshold register 2
    pub HTR2: RWRegister<u32>,

    /// ADC watchdog lower threshold register 3
    pub LTR3: RWRegister<u32>,

    /// ADC watchdog higher threshold register 3
    pub HTR3: RWRegister<u32>,

    /// ADC channel differential or single-ended mode selection register
    pub DIFSEL: RWRegister<u32>,

    /// ADC calibration factors register
    pub CALFACT: RWRegister<u32>,

    /// ADC Calibration Factor register 2
    pub CALFACT2: RWRegister<u32>,
}
pub struct ResetValues {
    pub ISR: u32,
    pub IER: u32,
    pub CR: u32,
    pub CFGR: u32,
    pub CFGR2: u32,
    pub SMPR1: u32,
    pub SMPR2: u32,
    pub PCSEL: u32,
    pub LTR1: u32,
    pub HTR1: u32,
    pub SQR1: u32,
    pub SQR2: u32,
    pub SQR3: u32,
    pub SQR4: u32,
    pub DR: u32,
    pub JSQR: u32,
    pub OFR1: u32,
    pub OFR2: u32,
    pub OFR3: u32,
    pub OFR4: u32,
    pub JDR1: u32,
    pub JDR2: u32,
    pub JDR3: u32,
    pub JDR4: u32,
    pub AWD2CR: u32,
    pub AWD3CR: u32,
    pub LTR2: u32,
    pub HTR2: u32,
    pub LTR3: u32,
    pub HTR3: u32,
    pub DIFSEL: u32,
    pub CALFACT: u32,
    pub CALFACT2: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}
