#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! Advanced-timers

use crate::{RWRegister, WORegister};
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// control register 1
pub mod CR1 {

    /// Counter enable Note: External clock, gated mode and encoder mode can work only if the CEN bit has been previously set by software. However trigger mode can set the CEN bit automatically by hardware.
    pub mod CEN {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Counter disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Counter enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Update disable This bit is set and cleared by software to enable/disable UEV event generation. Counter overflow/underflow Setting the UG bit Update generation through the slave mode controller Buffered registers are then loaded with their preload values.
    pub mod UDIS {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: UEV enabled. The Update (UEV) event is generated by one of the following events:
            pub const B_0x0: u32 = 0b0;

            /// 0b1: UEV disabled. The Update event is not generated, shadow registers keep their value (ARR, PSC, CCRx). However the counter and the prescaler are reinitialized if the UG bit is set or if a hardware reset is received from the slave mode controller.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Update request source This bit is set and cleared by software to select the UEV event sources. Counter overflow/underflow Setting the UG bit Update generation through the slave mode controller
    pub mod URS {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Any of the following events generate an update interrupt or DMA request if enabled. These events can be:
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Only counter overflow/underflow generates an update interrupt or DMA request if enabled.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// One pulse mode
    pub mod OPM {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Counter is not stopped at update event
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Counter stops counting at the next update event (clearing the bit CEN)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Direction Note: This bit is read only when the timer is configured in Center-aligned mode or Encoder mode.
    pub mod DIR {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Counter used as upcounter
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Counter used as downcounter
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Center-aligned mode selection Note: Switch from edge-aligned mode to center-aligned mode as long as the counter is enabled (CEN=1) is not allowed
    pub mod CMS {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (2 bits: 0b11 << 5)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Edge-aligned mode. The counter counts up or down depending on the direction bit (DIR).
            pub const B_0x0: u32 = 0b00;

            /// 0b01: Center-aligned mode 1. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set only when the counter is counting down.
            pub const B_0x1: u32 = 0b01;

            /// 0b10: Center-aligned mode 2. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set only when the counter is counting up.
            pub const B_0x2: u32 = 0b10;

            /// 0b11: Center-aligned mode 3. The counter counts up and down alternatively. Output compare interrupt flags of channels configured in output (CCxS=00 in TIMx_CCMRx register) are set both when the counter is counting up or down.
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Auto-reload preload enable
    pub mod ARPE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: TIMx_ARR register is not buffered
            pub const B_0x0: u32 = 0b0;

            /// 0b1: TIMx_ARR register is buffered
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Clock division This bit-field indicates the division ratio between the timer clock (CK_INT) frequency and the dead-time and sampling clock (tDTS)used by the dead-time generators and the digital filters (ETR, TIx): Note: tDTS = 1/fDTS, tCK_INT = 1/fCK_INT.
    pub mod CKD {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: tDTS=tCK_INT
            pub const B_0x0: u32 = 0b00;

            /// 0b01: tDTS=2*tCK_INT
            pub const B_0x1: u32 = 0b01;

            /// 0b10: tDTS=4*tCK_INT
            pub const B_0x2: u32 = 0b10;

            /// 0b11: Reserved, do not program this value
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// UIF status bit remapping
    pub mod UIFREMAP {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No remapping. UIF status bit is not copied to TIMx_CNT register bit 31.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Remapping enabled. UIF status bit is copied to TIMx_CNT register bit 31.
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// control register 2
pub mod CR2 {

    /// Capture/compare preloaded control Note: This bit acts only on channels that have a complementary output.
    pub mod CCPC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CCxE, CCxNE and OCxM bits are not preloaded
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CCxE, CCxNE and OCxM bits are preloaded, after having been written, they are updated only when a commutation event (COM) occurs (COMG bit set or rising edge detected on TRGI, depending on the CCUS bit).
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/compare control update selection Note: This bit acts only on channels that have a complementary output.
    pub mod CCUS {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit only
            pub const B_0x0: u32 = 0b0;

            /// 0b1: When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit or when an rising edge occurs on TRGI
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/compare DMA selection
    pub mod CCDS {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CCx DMA request sent when CCx event occurs
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CCx DMA requests sent when update event occurs
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Master mode selection These bits allow selected information to be sent in master mode to slave timers for synchronization (TRGO). The combination is as follows: Note: The clock of the slave timer or ADC must be enabled prior to receive events from the master timer, and must not be changed on-the-fly while triggers are received from the master timer.
    pub mod MMS {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: Reset - the UG bit from the TIMx_EGR register is used as trigger output (TRGO). If the reset is generated by the trigger input (slave mode controller configured in reset mode) then the signal on TRGO is delayed compared to the actual reset.
            pub const B_0x0: u32 = 0b000;

            /// 0b001: Enable - the Counter Enable signal CNT_EN is used as trigger output (TRGO). It is useful to start several timers at the same time or to control a window in which a slave timer is enable. The Counter Enable signal is generated by a logic AND between CEN control bit and the trigger input when configured in gated mode. When the Counter Enable signal is controlled by the trigger input, there is a delay on TRGO, except if the master/slave mode is selected (see the MSM bit description in TIMx_SMCR register).
            pub const B_0x1: u32 = 0b001;

            /// 0b010: Update - The update event is selected as trigger output (TRGO). For instance a master timer can then be used as a prescaler for a slave timer.
            pub const B_0x2: u32 = 0b010;

            /// 0b011: Compare Pulse - The trigger output send a positive pulse when the CC1IF flag is to be set (even if it was already high), as soon as a capture or a compare match occurred. (TRGO).
            pub const B_0x3: u32 = 0b011;

            /// 0b100: Compare - OC1REFC signal is used as trigger output (TRGO)
            pub const B_0x4: u32 = 0b100;

            /// 0b101: Compare - OC2REFC signal is used as trigger output (TRGO)
            pub const B_0x5: u32 = 0b101;

            /// 0b110: Compare - OC3REFC signal is used as trigger output (TRGO)
            pub const B_0x6: u32 = 0b110;

            /// 0b111: Compare - OC4REFC signal is used as trigger output (TRGO)
            pub const B_0x7: u32 = 0b111;
        }
    }

    /// TI1 selection
    pub mod TI1S {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: The TIMx_CH1 pin is connected to TI1 input
            pub const B_0x0: u32 = 0b0;

            /// 0b1: The TIMx_CH1, CH2 and CH3 pins are connected to the TI1 input (XOR combination)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Output Idle state 1 (OC1 output) Note: This bit can not be modified as long as LOCK level 1, 2 or 3 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod OIS1 {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1=0 (after a dead-time if OC1N is implemented) when MOE=0
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1=1 (after a dead-time if OC1N is implemented) when MOE=0
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Output Idle state 1 (OC1N output) Note: This bit can not be modified as long as LOCK level 1, 2 or 3 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod OIS1N {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1N=0 after a dead-time when MOE=0
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1N=1 after a dead-time when MOE=0
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Output Idle state 2 (OC2 output) Refer to OIS1 bit
    pub mod OIS2 {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 2 (OC2N output) Refer to OIS1N bit
    pub mod OIS2N {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 3 (OC3 output) Refer to OIS1 bit
    pub mod OIS3 {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 3 (OC3N output) Refer to OIS1N bit
    pub mod OIS3N {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 4 (OC4 output) Refer to OIS1 bit
    pub mod OIS4 {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 5 (OC5 output) Refer to OIS1 bit
    pub mod OIS5 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Idle state 6 (OC6 output) Refer to OIS1 bit
    pub mod OIS6 {
        /// Offset (18 bits)
        pub const offset: u32 = 18;
        /// Mask (1 bit: 1 << 18)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Master mode selection 2 These bits allow the information to be sent to ADC for synchronization (TRGO2) to be selected. The combination is as follows: Note: The clock of the slave timer or ADC must be enabled prior to receive events from the master timer, and must not be changed on-the-fly while triggers are received from the master timer.
    pub mod MMS2 {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (4 bits: 0b1111 << 20)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: Reset - the UG bit from the TIMx_EGR register is used as trigger output (TRGO2). If the reset is generated by the trigger input (slave mode controller configured in reset mode), the signal on TRGO2 is delayed compared to the actual reset.
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: Enable - the Counter Enable signal CNT_EN is used as trigger output (TRGO2). It is useful to start several timers at the same time or to control a window in which a slave timer is enabled. The Counter Enable signal is generated by a logic AND between the CEN control bit and the trigger input when configured in Gated mode. When the Counter Enable signal is controlled by the trigger input, there is a delay on TRGO2, except if the Master/Slave mode is selected (see the MSM bit description in TIMx_SMCR register).
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: Update - the update event is selected as trigger output (TRGO2). For instance, a master timer can then be used as a prescaler for a slave timer.
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: Compare pulse - the trigger output sends a positive pulse when the CC1IF flag is to be set (even if it was already high), as soon as a capture or compare match occurs (TRGO2).
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: Compare - OC1REFC signal is used as trigger output (TRGO2)
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: Compare - OC2REFC signal is used as trigger output (TRGO2)
            pub const B_0x5: u32 = 0b0101;

            /// 0b0110: Compare - OC3REFC signal is used as trigger output (TRGO2)
            pub const B_0x6: u32 = 0b0110;

            /// 0b0111: Compare - OC4REFC signal is used as trigger output (TRGO2)
            pub const B_0x7: u32 = 0b0111;

            /// 0b1000: Compare - OC5REFC signal is used as trigger output (TRGO2)
            pub const B_0x8: u32 = 0b1000;

            /// 0b1001: Compare - OC6REFC signal is used as trigger output (TRGO2)
            pub const B_0x9: u32 = 0b1001;

            /// 0b1010: Compare Pulse - OC4REFC rising or falling edges generate pulses on TRGO2
            pub const B_0xA: u32 = 0b1010;

            /// 0b1011: Compare Pulse - OC6REFC rising or falling edges generate pulses on TRGO2
            pub const B_0xB: u32 = 0b1011;

            /// 0b1100: Compare Pulse - OC4REFC or OC6REFC rising edges generate pulses on TRGO2
            pub const B_0xC: u32 = 0b1100;

            /// 0b1101: Compare Pulse - OC4REFC rising or OC6REFC falling edges generate pulses on TRGO2
            pub const B_0xD: u32 = 0b1101;

            /// 0b1110: Compare Pulse - OC5REFC or OC6REFC rising edges generate pulses on TRGO2
            pub const B_0xE: u32 = 0b1110;

            /// 0b1111: Compare Pulse - OC5REFC rising or OC6REFC falling edges generate pulses on TRGO2
            pub const B_0xF: u32 = 0b1111;
        }
    }
}

/// slave mode control register
pub mod SMCR {

    /// Slave mode selection When external signals are selected the active edge of the trigger signal (TRGI) is linked to the polarity selected on the external input (see Input Control register and Control Register description. Note: The gated mode must not be used if TI1F_ED is selected as the trigger input (TS=00100). Indeed, TI1F_ED outputs 1 pulse for each transition on TI1F, whereas the gated mode checks the level of the trigger signal. Note: The clock of the slave peripherals (timer, ADC, ...) receiving the TRGO or the TRGO2 signals must be enabled prior to receive events from the master timer, and the clock frequency (prescaler) must not be changed on-the-fly while triggers are received from the master timer.
    pub mod SMS1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (3 bits: 0b111 << 0)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// OCREF clear selection This bit is used to select the OCREF clear source.
    pub mod OCCS {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OCREF_CLR_INT is connected to COMP1 or COMP2 output depending on TIM1_OR1.OCREF_CLR
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OCREF_CLR_INT is connected to ETRF
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Trigger selection This bit-field selects the trigger input to be used to synchronize the counter. Others: Reserved See for more details on ITRx meaning for each Timer. Note: These bits must be changed only when they are not used (e.g. when SMS=000) to avoid wrong edge detections at the transition.
    pub mod TS1 {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Master/slave mode
    pub mod MSM {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: The effect of an event on the trigger input (TRGI) is delayed to allow a perfect synchronization between the current timer and its slaves (through TRGO). It is useful if we want to synchronize several timers on a single external event.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// External trigger filter This bit-field then defines the frequency used to sample ETRP signal and the length of the digital filter applied to ETRP. The digital filter is made of an event counter in which N consecutive events are needed to validate a transition on the output:
    pub mod ETF {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: No filter, sampling is done at fDTS
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: fSAMPLING=fCK_INT, N=2
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: fSAMPLING=fCK_INT, N=4
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: fSAMPLING=fCK_INT, N=8
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: fSAMPLING=fDTS/2, N=6
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: fSAMPLING=fDTS/2, N=8
            pub const B_0x5: u32 = 0b0101;

            /// 0b0110: fSAMPLING=fDTS/4, N=6
            pub const B_0x6: u32 = 0b0110;

            /// 0b0111: fSAMPLING=fDTS/4, N=8
            pub const B_0x7: u32 = 0b0111;

            /// 0b1000: fSAMPLING=fDTS/8, N=6
            pub const B_0x8: u32 = 0b1000;

            /// 0b1001: fSAMPLING=fDTS/8, N=8
            pub const B_0x9: u32 = 0b1001;

            /// 0b1010: fSAMPLING=fDTS/16, N=5
            pub const B_0xA: u32 = 0b1010;

            /// 0b1011: fSAMPLING=fDTS/16, N=6
            pub const B_0xB: u32 = 0b1011;

            /// 0b1100: fSAMPLING=fDTS/16, N=8
            pub const B_0xC: u32 = 0b1100;

            /// 0b1101: fSAMPLING=fDTS/32, N=5
            pub const B_0xD: u32 = 0b1101;

            /// 0b1110: fSAMPLING=fDTS/32, N=6
            pub const B_0xE: u32 = 0b1110;

            /// 0b1111: fSAMPLING=fDTS/32, N=8
            pub const B_0xF: u32 = 0b1111;
        }
    }

    /// External trigger prescaler External trigger signal ETRP frequency must be at most 1/4 of fCK_INT frequency. A prescaler can be enabled to reduce ETRP frequency. It is useful when inputting fast external clocks.
    pub mod ETPS {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (2 bits: 0b11 << 12)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: Prescaler OFF
            pub const B_0x0: u32 = 0b00;

            /// 0b01: ETRP frequency divided by 2
            pub const B_0x1: u32 = 0b01;

            /// 0b10: ETRP frequency divided by 4
            pub const B_0x2: u32 = 0b10;

            /// 0b11: ETRP frequency divided by 8
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// External clock enable This bit enables External clock mode 2. Note: Setting the ECE bit has the same effect as selecting external clock mode 1 with TRGI connected to ETRF (SMS=111 and TS=00111). It is possible to simultaneously use external clock mode 2 with the following slave modes: reset mode, gated mode and trigger mode. Nevertheless, TRGI must not be connected to ETRF in this case (TS bits must not be 00111). If external clock mode 1 and external clock mode 2 are enabled at the same time, the external clock input is ETRF.
    pub mod ECE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: External clock mode 2 disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: External clock mode 2 enabled. The counter is clocked by any active edge on the ETRF signal.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// External trigger polarity This bit selects whether ETR or ETR is used for trigger operations
    pub mod ETP {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: ETR is non-inverted, active at high level or rising edge.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: ETR is inverted, active at low level or falling edge.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Slave mode selection When external signals are selected the active edge of the trigger signal (TRGI) is linked to the polarity selected on the external input (see Input Control register and Control Register description. Note: The gated mode must not be used if TI1F_ED is selected as the trigger input (TS=00100). Indeed, TI1F_ED outputs 1 pulse for each transition on TI1F, whereas the gated mode checks the level of the trigger signal. Note: The clock of the slave peripherals (timer, ADC, ...) receiving the TRGO or the TRGO2 signals must be enabled prior to receive events from the master timer, and the clock frequency (prescaler) must not be changed on-the-fly while triggers are received from the master timer.
    pub mod SMS2 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Trigger selection This bit-field selects the trigger input to be used to synchronize the counter. Others: Reserved See for more details on ITRx meaning for each Timer. Note: These bits must be changed only when they are not used (e.g. when SMS=000) to avoid wrong edge detections at the transition.
    pub mod TS2 {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (2 bits: 0b11 << 20)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// DMA/Interrupt enable register
pub mod DIER {

    /// Update interrupt enable
    pub mod UIE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Update interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Update interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 interrupt enable
    pub mod CC1IE {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC1 interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC1 interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 interrupt enable
    pub mod CC2IE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC2 interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC2 interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 3 interrupt enable
    pub mod CC3IE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC3 interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC3 interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 4 interrupt enable
    pub mod CC4IE {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC4 interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC4 interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// COM interrupt enable
    pub mod COMIE {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: COM interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: COM interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Trigger interrupt enable
    pub mod TIE {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Trigger interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Trigger interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break interrupt enable
    pub mod BIE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break interrupt disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break interrupt enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Update DMA request enable
    pub mod UDE {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Update DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Update DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 DMA request enable
    pub mod CC1DE {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC1 DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC1 DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 DMA request enable
    pub mod CC2DE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC2 DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC2 DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 3 DMA request enable
    pub mod CC3DE {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC3 DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC3 DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 4 DMA request enable
    pub mod CC4DE {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC4 DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: CC4 DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// COM DMA request enable
    pub mod COMDE {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: COM DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: COM DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Trigger DMA request enable
    pub mod TDE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Trigger DMA request disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Trigger DMA request enabled
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// status register
pub mod SR {

    /// Update interrupt flag This bit is set by hardware on an update event. It is cleared by software. At overflow or underflow regarding the repetition counter value (update if repetition counter = 0) and if the UDIS=0 in the TIMx_CR1 register. When CNT is reinitialized by software using the UG bit in TIMx_EGR register, if URS=0 and UDIS=0 in the TIMx_CR1 register. When CNT is reinitialized by a trigger event (refer to control register (TIM1_SMCRTIMx_SMCR)N/A), if URS=0 and UDIS=0 in the TIMx_CR1 register.
    pub mod UIF {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No update occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Update interrupt pending. This bit is set by hardware when the registers are updated:
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 interrupt flag This flag is set by hardware. It is cleared by software (input capture or output compare mode) or by reading the TIMx_CCR1 register (input capture mode only). If channel CC1 is configured as output: this flag is set when he content of the counter TIMx_CNT matches the content of the TIMx_CCR1 register. When the content of TIMx_CCR1 is greater than the content of TIMx_ARR, the CC1IF bit goes high on the counter overflow (in up-counting and up/down-counting modes) or underflow (in down-counting mode). There are 3 possible options for flag setting in center-aligned mode, refer to the CMS bits in the TIMx_CR1 register for the full description. If channel CC1 is configured as input: this bit is set when counter value has been captured in TIMx_CCR1 register (an edge has been detected on IC1, as per the edge sensitivity defined with the CC1P and CC1NP bits setting, in TIMx_CCER).
    pub mod CC1IF {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No compare match / No input capture occurred
            pub const B_0x0: u32 = 0b0;

            /// 0b1: A compare match or an input capture occurred.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 interrupt flag Refer to CC1IF description
    pub mod CC2IF {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 interrupt flag Refer to CC1IF description
    pub mod CC3IF {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 interrupt flag Refer to CC1IF description
    pub mod CC4IF {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// COM interrupt flag This flag is set by hardware on COM event (when Capture/compare Control bits - CCxE, CCxNE, OCxM - have been updated). It is cleared by software.
    pub mod COMIF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No COM event occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: COM interrupt pending.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Trigger interrupt flag This flag is set by hardware on the TRG trigger event (active edge detected on TRGI input when the slave mode controller is enabled in all modes but gated mode. It is set when the counter starts or stops when gated mode is selected. It is cleared by software.
    pub mod TIF {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No trigger event occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Trigger interrupt pending.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break interrupt flag This flag is set by hardware as soon as the break input goes active. It can be cleared by software if the break input is not active.
    pub mod BIF {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No break event occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: An active level has been detected on the break input. An interrupt is generated if BIE=1 in the TIMx_DIER register.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break 2 interrupt flag This flag is set by hardware as soon as the break 2 input goes active. It can be cleared by software if the break 2 input is not active.
    pub mod B2IF {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No break event occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: An active level has been detected on the break 2 input. An interrupt is generated if BIE=1 in the TIMx_DIER register.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 overcapture flag This flag is set by hardware only when the corresponding channel is configured in input capture mode. It is cleared by software by writing it to '0â.
    pub mod CC1OF {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No overcapture has been detected.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: The counter value has been captured in TIMx_CCR1 register while CC1IF flag was already set
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 overcapture flag Refer to CC1OF description
    pub mod CC2OF {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 overcapture flag Refer to CC1OF description
    pub mod CC3OF {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 overcapture flag Refer to CC1OF description
    pub mod CC4OF {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// System Break interrupt flag This flag is set by hardware as soon as the system break input goes active. It can be cleared by software if the system break input is not active. This flag must be reset to re-start PWM operation.
    pub mod SBIF {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No break event occurred.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: An active level has been detected on the system break input. An interrupt is generated if BIE=1 in the TIMx_DIER register.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Compare 5 interrupt flag Refer to CC1IF description (Note: Channel 5 can only be configured as output)
    pub mod CC5IF {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Compare 6 interrupt flag Refer to CC1IF description (Note: Channel 6 can only be configured as output)
    pub mod CC6IF {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// event generation register
pub mod EGR {

    /// Update generation This bit can be set by software, it is automatically cleared by hardware.
    pub mod UG {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Reinitialize the counter and generates an update of the registers. The prescaler internal counter is also cleared (the prescaler ratio is not affected). The counter is cleared if the center-aligned mode is selected or if DIR=0 (upcounting), else it takes the auto-reload value (TIMx_ARR) if DIR=1 (downcounting).
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 generation This bit is set by software in order to generate an event, it is automatically cleared by hardware. If channel CC1 is configured as output: CC1IF flag is set, Corresponding interrupt or DMA request is sent if enabled. If channel CC1 is configured as input: The current value of the counter is captured in TIMx_CCR1 register. The CC1IF flag is set, the corresponding interrupt or DMA request is sent if enabled. The CC1OF flag is set if the CC1IF flag was already high.
    pub mod CC1G {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: A capture/compare event is generated on channel 1:
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 generation Refer to CC1G description
    pub mod CC2G {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 generation Refer to CC1G description
    pub mod CC3G {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 generation Refer to CC1G description
    pub mod CC4G {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare control update generation This bit can be set by software, it is automatically cleared by hardware Note: This bit acts only on channels having a complementary output.
    pub mod COMG {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: When CCPC bit is set, it allows CCxE, CCxNE and OCxM bits to be updated.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Trigger generation This bit is set by software in order to generate an event, it is automatically cleared by hardware.
    pub mod TG {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: The TIF flag is set in TIMx_SR register. Related interrupt or DMA transfer can occur if enabled.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break generation This bit is set by software in order to generate an event, it is automatically cleared by hardware.
    pub mod BG {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: A break event is generated. MOE bit is cleared and BIF flag is set. Related interrupt or DMA transfer can occur if enabled.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break 2 generation This bit is set by software in order to generate an event, it is automatically cleared by hardware.
    pub mod B2G {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No action
            pub const B_0x0: u32 = 0b0;

            /// 0b1: A break 2 event is generated. MOE bit is cleared and B2IF flag is set. Related interrupt can occur if enabled.
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// CCMR1_Output and CCMR1_Input
/// CCMR1_Output: capture/compare mode register 1 (output mode)
/// CCMR1_Input: capture/compare mode register 1 (output mode)
pub mod CCMR1 {

    /// Capture/Compare 1 selection This bit-field defines the direction of the channel (input/output) as well as the used input. Note: CC1S bits are writable only when the channel is OFF (CC1E = '0â in TIMx_CCER).
    pub mod CC1S {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CC1 channel is configured as output
            pub const B_0x0: u32 = 0b00;

            /// 0b01: CC1 channel is configured as input, IC1 is mapped on TI1
            pub const B_0x1: u32 = 0b01;

            /// 0b10: CC1 channel is configured as input, IC1 is mapped on TI2
            pub const B_0x2: u32 = 0b10;

            /// 0b11: CC1 channel is configured as input, IC1 is mapped on TRC. This mode is working only if an internal trigger input is selected through TS bit (TIMx_SMCR register)
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Output Compare 1 fast enable This bit decreases the latency between a trigger event and a transition on the timer output. It must be used in one-pulse mode (OPM bit set in TIMx_CR1 register), to have the output pulse starting as soon as possible after the starting trigger.
    pub mod OC1FE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CC1 behaves normally depending on counter and CCR1 values even when the trigger is ON. The minimum delay to activate CC1 output when an edge occurs on the trigger input is 5 clock cycles.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: An active edge on the trigger input acts like a compare match on CC1 output. Then, OC is set to the compare level independently from the result of the comparison. Delay to sample the trigger input and to activate CC1 output is reduced to 3 clock cycles. OCFE acts only if the channel is configured in PWM1 or PWM2 mode.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Output Compare 1 preload enable Note: These bits can not be modified as long as LOCK level 3 has been programmed (LOCK bits in TIMx_BDTR register) and CC1S=â00â (the channel is configured in output). The PWM mode can be used without validating the preload register only in one pulse mode (OPM bit set in TIMx_CR1 register). Else the behavior is not guaranteed.
    pub mod OC1PE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Preload register on TIMx_CCR1 disabled. TIMx_CCR1 can be written at anytime, the new value is taken in account immediately.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Preload register on TIMx_CCR1 enabled. Read/Write operations access the preload register. TIMx_CCR1 preload value is loaded in the active register at each update event.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Output Compare 1 mode These bits define the behavior of the output reference signal OC1REF from which OC1 and OC1N are derived. OC1REF is active high whereas OC1 and OC1N active level depends on CC1P and CC1NP bits. Note: These bits can not be modified as long as LOCK level 3 has been programmed (LOCK bits in TIMx_BDTR register) and CC1S=â00â (the channel is configured in output). Note: In PWM mode, the OCREF level changes only when the result of the comparison changes or when the output compare mode switches from âfrozenâ mode to âPWMâ mode. Note: On channels having a complementary output, this bit field is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the OC1M active bits take the new value from the preloaded bits only when a COM event is generated. Note: The OC1M\[3\] bit is not contiguous, located in bit 16.
    pub mod OC1M1 {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Compare 1 clear enable
    pub mod OC1CE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1Ref is not affected by the ocref_clr_int signal
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1Ref is cleared as soon as a High level is detected on ocref_clr_int signal (OCREF_CLR input or ETRF input)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 selection This bit-field defines the direction of the channel (input/output) as well as the used input. Note: CC2S bits are writable only when the channel is OFF (CC2E = '0â in TIMx_CCER).
    pub mod CC2S {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CC2 channel is configured as output
            pub const B_0x0: u32 = 0b00;

            /// 0b01: CC2 channel is configured as input, IC2 is mapped on TI2
            pub const B_0x1: u32 = 0b01;

            /// 0b10: CC2 channel is configured as input, IC2 is mapped on TI1
            pub const B_0x2: u32 = 0b10;

            /// 0b11: CC2 channel is configured as input, IC2 is mapped on TRC. This mode is working only if an internal trigger input is selected through the TS bit (TIMx_SMCR register)
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Output Compare 2 fast enable Refer to OC1FE description.
    pub mod OC2FE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Compare 2 preload enable Refer to OC1PE description.
    pub mod OC2PE {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Compare 2 mode Refer to OC1M\[3:0\] description.
    pub mod OC2M {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: The comparison between the output compare register TIMx_CCRy and the counter TIMx_CNT has no effect on the outputs / OpmMode1: Retriggerable OPM mode 1 - In up-counting mode, the channel is active until a trigger event is detected (on TRGI signal). In down-counting mode, the channel is inactive
            pub const Frozen: u32 = 0b000;

            /// 0b001: Set channel to active level on match. OCyREF signal is forced high when the counter matches the capture/compare register / OpmMode2: Inversely to OpmMode1
            pub const ActiveOnMatch: u32 = 0b001;

            /// 0b010: Set channel to inactive level on match. OCyREF signal is forced low when the counter matches the capture/compare register / Reserved
            pub const InactiveOnMatch: u32 = 0b010;

            /// 0b011: OCyREF toggles when TIMx_CNT=TIMx_CCRy / Reserved
            pub const Toggle: u32 = 0b011;

            /// 0b100: OCyREF is forced low / CombinedPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC is the logical OR between OC1REF and OC2REF
            pub const ForceInactive: u32 = 0b100;

            /// 0b101: OCyREF is forced high / CombinedPwmMode2: OCyREF has the same behavior as in PWM mode 2. OCyREFC is the logical AND between OC1REF and OC2REF
            pub const ForceActive: u32 = 0b101;

            /// 0b110: In upcounting, channel is active as long as TIMx_CNT<TIMx_CCRy else inactive. In downcounting, channel is inactive as long as TIMx_CNT>TIMx_CCRy else active / AsymmetricPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC outputs OC1REF when the counter is counting up, OC2REF when it is counting down
            pub const PwmMode1: u32 = 0b110;

            /// 0b111: Inversely to PwmMode1 / AsymmetricPwmMode2: Inversely to AsymmetricPwmMode1
            pub const PwmMode2: u32 = 0b111;
        }
    }

    /// Output Compare 2 clear enable Refer to OC1CE description.
    pub mod OC2CE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Compare 1 mode These bits define the behavior of the output reference signal OC1REF from which OC1 and OC1N are derived. OC1REF is active high whereas OC1 and OC1N active level depends on CC1P and CC1NP bits. Note: These bits can not be modified as long as LOCK level 3 has been programmed (LOCK bits in TIMx_BDTR register) and CC1S=â00â (the channel is configured in output). Note: In PWM mode, the OCREF level changes only when the result of the comparison changes or when the output compare mode switches from âfrozenâ mode to âPWMâ mode. Note: On channels having a complementary output, this bit field is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the OC1M active bits take the new value from the preloaded bits only when a COM event is generated. Note: The OC1M\[3\] bit is not contiguous, located in bit 16.
    pub mod OC1M2 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output Compare 2 mode Refer to OC1M\[3:0\] description.
    pub mod OC2M_3 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Normal output compare mode (modes 0-7)
            pub const Normal: u32 = 0b0;

            /// 0b1: Extended output compare mode (modes 7-15)
            pub const Extended: u32 = 0b1;
        }
    }

    /// Input capture 1 prescaler This bit-field defines the ratio of the prescaler acting on CC1 input (IC1). The prescaler is reset as soon as CC1E=â0â (TIMx_CCER register).
    pub mod IC1PSC {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (2 bits: 0b11 << 2)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: no prescaler, capture is done each time an edge is detected on the capture input
            pub const B_0x0: u32 = 0b00;

            /// 0b01: capture is done once every 2 events
            pub const B_0x1: u32 = 0b01;

            /// 0b10: capture is done once every 4 events
            pub const B_0x2: u32 = 0b10;

            /// 0b11: capture is done once every 8 events
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Input capture 1 filter This bit-field defines the frequency used to sample TI1 input and the length of the digital filter applied to TI1. The digital filter is made of an event counter in which N consecutive events are needed to validate a transition on the output:
    pub mod IC1F {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (4 bits: 0b1111 << 4)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: No filter, sampling is done at fDTS
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: fSAMPLING=fCK_INT, N=2
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: fSAMPLING=fCK_INT, N=4
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: fSAMPLING=fCK_INT, N=8
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: fSAMPLING=fDTS/2, N=6
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: fSAMPLING=fDTS/2, N=8
            pub const B_0x5: u32 = 0b0101;

            /// 0b0110: fSAMPLING=fDTS/4, N=6
            pub const B_0x6: u32 = 0b0110;

            /// 0b0111: fSAMPLING=fDTS/4, N=8
            pub const B_0x7: u32 = 0b0111;

            /// 0b1000: fSAMPLING=fDTS/8, N=6
            pub const B_0x8: u32 = 0b1000;

            /// 0b1001: fSAMPLING=fDTS/8, N=8
            pub const B_0x9: u32 = 0b1001;

            /// 0b1010: fSAMPLING=fDTS/16, N=5
            pub const B_0xA: u32 = 0b1010;

            /// 0b1011: fSAMPLING=fDTS/16, N=6
            pub const B_0xB: u32 = 0b1011;

            /// 0b1100: fSAMPLING=fDTS/16, N=8
            pub const B_0xC: u32 = 0b1100;

            /// 0b1101: fSAMPLING=fDTS/32, N=5
            pub const B_0xD: u32 = 0b1101;

            /// 0b1110: fSAMPLING=fDTS/32, N=6
            pub const B_0xE: u32 = 0b1110;

            /// 0b1111: fSAMPLING=fDTS/32, N=8
            pub const B_0xF: u32 = 0b1111;
        }
    }

    /// Input capture 2 prescaler Refer to IC1PSC\[1:0\] description.
    pub mod IC2PSC {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (2 bits: 0b11 << 10)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Input capture 2 filter Refer to IC1F\[3:0\] description.
    pub mod IC2F {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (4 bits: 0b1111 << 12)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// CCMR2_Output and CCMR2_Input
/// CCMR2_Output: capture/compare mode register 2 (output mode)
/// CCMR2_Input: capture/compare mode register 2 (output mode)
pub mod CCMR2 {

    /// Capture/Compare 3 selection This bit-field defines the direction of the channel (input/output) as well as the used input. Note: CC3S bits are writable only when the channel is OFF (CC3E = '0â in TIMx_CCER).
    pub mod CC3S {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (2 bits: 0b11 << 0)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CC3 channel is configured as output
            pub const B_0x0: u32 = 0b00;

            /// 0b01: CC3 channel is configured as input, IC3 is mapped on TI3
            pub const B_0x1: u32 = 0b01;

            /// 0b10: CC3 channel is configured as input, IC3 is mapped on TI4
            pub const B_0x2: u32 = 0b10;

            /// 0b11: CC3 channel is configured as input, IC3 is mapped on TRC. This mode is working only if an internal trigger input is selected through TS bit (TIMx_SMCR register)
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Output compare 3 fast enable Refer to OC1FE description.
    pub mod OC3FE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 3 preload enable Refer to OC1PE description.
    pub mod OC3PE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 3 mode Refer to OC1M\[3:0\] description.
    pub mod OC3M {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: The comparison between the output compare register TIMx_CCRy and the counter TIMx_CNT has no effect on the outputs / OpmMode1: Retriggerable OPM mode 1 - In up-counting mode, the channel is active until a trigger event is detected (on TRGI signal). In down-counting mode, the channel is inactive
            pub const Frozen: u32 = 0b000;

            /// 0b001: Set channel to active level on match. OCyREF signal is forced high when the counter matches the capture/compare register / OpmMode2: Inversely to OpmMode1
            pub const ActiveOnMatch: u32 = 0b001;

            /// 0b010: Set channel to inactive level on match. OCyREF signal is forced low when the counter matches the capture/compare register / Reserved
            pub const InactiveOnMatch: u32 = 0b010;

            /// 0b011: OCyREF toggles when TIMx_CNT=TIMx_CCRy / Reserved
            pub const Toggle: u32 = 0b011;

            /// 0b100: OCyREF is forced low / CombinedPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC is the logical OR between OC1REF and OC2REF
            pub const ForceInactive: u32 = 0b100;

            /// 0b101: OCyREF is forced high / CombinedPwmMode2: OCyREF has the same behavior as in PWM mode 2. OCyREFC is the logical AND between OC1REF and OC2REF
            pub const ForceActive: u32 = 0b101;

            /// 0b110: In upcounting, channel is active as long as TIMx_CNT<TIMx_CCRy else inactive. In downcounting, channel is inactive as long as TIMx_CNT>TIMx_CCRy else active / AsymmetricPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC outputs OC1REF when the counter is counting up, OC2REF when it is counting down
            pub const PwmMode1: u32 = 0b110;

            /// 0b111: Inversely to PwmMode1 / AsymmetricPwmMode2: Inversely to AsymmetricPwmMode1
            pub const PwmMode2: u32 = 0b111;
        }
    }

    /// Output compare 3 clear enable Refer to OC1CE description.
    pub mod OC3CE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 selection This bit-field defines the direction of the channel (input/output) as well as the used input. Note: CC4S bits are writable only when the channel is OFF (CC4E = '0â in TIMx_CCER).
    pub mod CC4S {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: CC4 channel is configured as output
            pub const B_0x0: u32 = 0b00;

            /// 0b01: CC4 channel is configured as input, IC4 is mapped on TI4
            pub const B_0x1: u32 = 0b01;

            /// 0b10: CC4 channel is configured as input, IC4 is mapped on TI3
            pub const B_0x2: u32 = 0b10;

            /// 0b11: CC4 channel is configured as input, IC4 is mapped on TRC. This mode is working only if an internal trigger input is selected through TS bit (TIMx_SMCR register)
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Output compare 4 fast enable Refer to OC1FE description.
    pub mod OC4FE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 4 preload enable Refer to OC1PE description.
    pub mod OC4PE {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 4 mode Refer to OC3M\[3:0\] description.
    pub mod OC4M {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::OC3M::RW;
    }

    /// Output compare 4 clear enable Refer to OC1CE description.
    pub mod OC4CE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 3 mode Refer to OC1M\[3:0\] description.
    pub mod OC3M_3 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Normal output compare mode (modes 0-7)
            pub const Normal: u32 = 0b0;

            /// 0b1: Extended output compare mode (modes 7-15)
            pub const Extended: u32 = 0b1;
        }
    }

    /// Output compare 4 mode Refer to OC3M\[3:0\] description.
    pub mod OC4M_3 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::OC3M_3::RW;
    }

    /// Input capture 3 prescaler Refer to IC1PSC\[1:0\] description.
    pub mod IC3PSC {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (2 bits: 0b11 << 2)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Input capture 3 filter Refer to IC1F\[3:0\] description.
    pub mod IC3F {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (4 bits: 0b1111 << 4)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Input capture 4 prescaler Refer to IC1PSC\[1:0\] description.
    pub mod IC4PSC {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (2 bits: 0b11 << 10)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Input capture 4 filter Refer to IC1F\[3:0\] description.
    pub mod IC4F {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (4 bits: 0b1111 << 12)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare enable register
pub mod CCER {

    /// Capture/Compare 1 output enable When CC1 channel is configured as output, the OC1 level depends on MOE, OSSI, OSSR, OIS1, OIS1N and CC1NE bits, regardless of the CC1E bits state. Refer to for details. Note: On channels having a complementary output, this bit is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the CC1E active bit takes the new value from the preloaded bit only when a Commutation event is generated.
    pub mod CC1E {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Capture mode disabled / OC1 is not active (see below)
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Capture mode enabled / OC1 signal is output on the corresponding output pin
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 output polarity When CC1 channel is configured as input, both CC1NP/CC1P bits select the active polarity of TI1FP1 and TI2FP1 for trigger or capture operations. CC1NP=0, CC1P=0: non-inverted/rising edge. The circuit is sensitive to TIxFP1 rising edge (capture or trigger operations in reset, external clock or trigger mode), TIxFP1 is not inverted (trigger operation in gated mode or encoder mode). CC1NP=0, CC1P=1: inverted/falling edge. The circuit is sensitive to TIxFP1 falling edge (capture or trigger operations in reset, external clock or trigger mode), TIxFP1 is inverted (trigger operation in gated mode or encoder mode). CC1NP=1, CC1P=1: non-inverted/both edges/ The circuit is sensitive to both TIxFP1 rising and falling edges (capture or trigger operations in reset, external clock or trigger mode), TIxFP1is not inverted (trigger operation in gated mode). This configuration must not be used in encoder mode. CC1NP=1, CC1P=0: The configuration is reserved, it must not be used. Note: This bit is not writable as soon as LOCK level 2 or 3 has been programmed (LOCK bits in TIMx_BDTR register). On channels having a complementary output, this bit is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the CC1P active bit takes the new value from the preloaded bit only when a Commutation event is generated.
    pub mod CC1P {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1 active high (output mode) / Edge sensitivity selection (input mode, see below)
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1 active low (output mode) / Edge sensitivity selection (input mode, see below)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 complementary output enable On channels having a complementary output, this bit is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the CC1NE active bit takes the new value from the preloaded bit only when a Commutation event is generated.
    pub mod CC1NE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Off - OC1N is not active. OC1N level is then function of MOE, OSSI, OSSR, OIS1, OIS1N and CC1E bits.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: On - OC1N signal is output on the corresponding output pin depending on MOE, OSSI, OSSR, OIS1, OIS1N and CC1E bits.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 1 complementary output polarity CC1 channel configured as output: CC1 channel configured as input: This bit is used in conjunction with CC1P to define the polarity of TI1FP1 and TI2FP1. Refer to CC1P description. Note: This bit is not writable as soon as LOCK level 2 or 3 has been programmed (LOCK bits in TIMx_BDTR register) and CC1S=â00â (channel configured as output). On channels having a complementary output, this bit is preloaded. If the CCPC bit is set in the TIMx_CR2 register then the CC1NP active bit takes the new value from the preloaded bit only when a Commutation event is generated.
    pub mod CC1NP {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: OC1N active high.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1N active low.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Capture/Compare 2 output enable Refer to CC1E description
    pub mod CC2E {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 2 output polarity Refer to CC1P description
    pub mod CC2P {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 2 complementary output enable Refer to CC1NE description
    pub mod CC2NE {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 2 complementary output polarity Refer to CC1NP description
    pub mod CC2NP {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 output enable Refer to CC1E description
    pub mod CC3E {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 output polarity Refer to CC1P description
    pub mod CC3P {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 complementary output enable Refer to CC1NE description
    pub mod CC3NE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 3 complementary output polarity Refer to CC1NP description
    pub mod CC3NP {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 output enable Refer to CC1E description
    pub mod CC4E {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 output polarity Refer to CC1P description
    pub mod CC4P {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 4 complementary output polarity Refer to CC1NP description
    pub mod CC4NP {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 5 output enable Refer to CC1E description
    pub mod CC5E {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 5 output polarity Refer to CC1P description
    pub mod CC5P {
        /// Offset (17 bits)
        pub const offset: u32 = 17;
        /// Mask (1 bit: 1 << 17)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 6 output enable Refer to CC1E description
    pub mod CC6E {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (1 bit: 1 << 20)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Capture/Compare 6 output polarity Refer to CC1P description
    pub mod CC6P {
        /// Offset (21 bits)
        pub const offset: u32 = 21;
        /// Mask (1 bit: 1 << 21)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// counter
pub mod CNT {

    /// Counter value
    pub mod CNT {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// UIF copy This bit is a read-only copy of the UIF bit of the TIMx_ISR register. If the UIFREMAP bit in the TIMxCR1 is reset, bit 31 is reserved and read at 0.
    pub mod UIFCPY {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// prescaler
pub mod PSC {

    /// Prescaler value The counter clock frequency (CK_CNT) is equal to fCK_PSC / (PSC\[15:0\] + 1). PSC contains the value to be loaded in the active prescaler register at each update event (including when the counter is cleared through UG bit of TIMx_EGR register or through trigger controller when configured in âreset modeâ).
    pub mod PSC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// auto-reload register
pub mod ARR {

    /// Auto-reload value ARR is the value to be loaded in the actual auto-reload register. Refer to the for more details about ARR update and behavior. The counter is blocked while the auto-reload value is null.
    pub mod ARR {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// repetition counter register
pub mod RCR {

    /// Repetition counter value
    pub mod REP {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare register 1
pub mod CCR1 {

    /// Capture/Compare 1 value
    pub mod CCR1 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare register 2
pub mod CCR2 {

    /// Capture/Compare 2 value
    pub mod CCR2 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare register 3
pub mod CCR3 {

    /// Capture/Compare value
    pub mod CCR3 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare register 4
pub mod CCR4 {

    /// Capture/Compare value
    pub mod CCR4 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// break and dead-time register
pub mod BDTR {

    /// Dead-time generator setup This bit-field defines the duration of the dead-time inserted between the complementary outputs. DT correspond to this duration. DTG\[7:5\]=0xx => DT=DTG\[7:0\]x tDTG with tDTG=tDTS. DTG\[7:5\]=10x => DT=(64+DTG\[5:0\])xtDTG with tDTG=2xtDTS. DTG\[7:5\]=110 => DT=(32+DTG\[4:0\])xtDTG with tDTG=8xtDTS. DTG\[7:5\]=111 => DT=(32+DTG\[4:0\])xtDTG with tDTG=16xtDTS. Example if tDTS=125Â ns (8Â MHz), dead-time possible values are: 0 to 15875Â ns by 125Â ns steps, 16Â Î¼s to 31750Â nsÂ by 250Â ns steps, 32Â Î¼s to 63Â Î¼s by 1Â Î¼s steps, 64Â Î¼s to 126Â Î¼s by 2Â Î¼s steps Note: This bit-field can not be modified as long as LOCK level 1, 2 or 3 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod DTG {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (8 bits: 0xff << 0)
        pub const mask: u32 = 0xff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Lock configuration These bits offer a write protection against software errors. Note: The LOCK bits can be written only once after the reset. Once the TIMx_BDTR register has been written, their content is frozen until the next reset.
    pub mod LOCK {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (2 bits: 0b11 << 8)
        pub const mask: u32 = 0b11 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00: LOCK OFF - No bit is write protected.
            pub const B_0x0: u32 = 0b00;

            /// 0b01: LOCK Level 1 = DTG bits in TIMx_BDTR register, OISx and OISxN bits in TIMx_CR2 register and BK2BID, BKBID, BK2DSRM, BKDSRM, BK2P, BK2E, BK2F\[3:0\], BKF\[3:0\], AOE, BKP, BKE, OSSI, OSSR and DTG\[7:0\] bits in TIMx_BDTR register can no longer be written.
            pub const B_0x1: u32 = 0b01;

            /// 0b10: LOCK Level 2 = LOCK Level 1 + CC Polarity bits (CCxP/CCxNP bits in TIMx_CCER register, as long as the related channel is configured in output through the CCxS bits) as well as OSSR and OSSI bits can no longer be written.
            pub const B_0x2: u32 = 0b10;

            /// 0b11: LOCK Level 3 = LOCK Level 2 + CC Control bits (OCxM and OCxPE bits in TIMx_CCMRx registers, as long as the related channel is configured in output through the CCxS bits) can no longer be written.
            pub const B_0x3: u32 = 0b11;
        }
    }

    /// Off-state selection for Idle mode This bit is used when MOE=0 due to a break event or by a software write, on channels configured as outputs. See OC/OCN enable description for more details (enable register (TIM1_CCERTIMx_CCER)N/A). Note: This bit can not be modified as soon as the LOCK level 2 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod OSSI {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: When inactive, OC/OCN outputs are disabled (the timer releases the output control which is taken over by the GPIO logic and which imposes a Hi-Z state).
            pub const B_0x0: u32 = 0b0;

            /// 0b1: When inactive, OC/OCN outputs are first forced with their inactive level then forced to their idle level after the deadtime. The timer maintains its control over the output.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Off-state selection for Run mode This bit is used when MOE=1 on channels having a complementary output which are configured as outputs. OSSR is not implemented if no complementary output is implemented in the timer. See OC/OCN enable description for more details (enable register (TIM1_CCERTIMx_CCER)N/A). Note: This bit can not be modified as soon as the LOCK level 2 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod OSSR {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: When inactive, OC/OCN outputs are disabled (the timer releases the output control which is taken over by the GPIO logic, which forces a Hi-Z state).
            pub const B_0x0: u32 = 0b0;

            /// 0b1: When inactive, OC/OCN outputs are enabled with their inactive level as soon as CCxE=1 or CCxNE=1 (the output is still controlled by the timer).
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break enable This bit enables the complete break protection (including all sources connected to bk_acth and BKIN sources, as per ). Note: This bit cannot be modified when LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register). Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BKE {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break function disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break function enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break polarity Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register). Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BKP {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK is active low
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break input BRK is active high
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Automatic output enable Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod AOE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: MOE can be set only by software
            pub const B_0x0: u32 = 0b0;

            /// 0b1: MOE can be set by software or automatically at the next update event (if none of the break inputs BRK and BRK2 is active)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Main output enable This bit is cleared asynchronously by hardware as soon as one of the break inputs is active (BRK or BRK2). It is set by software or automatically depending on the AOE bit. It is acting only on the channels which are configured in output. In response to a break event or if MOE is written to 0: OC and OCN outputs are disabled or forced to idle state depending on the OSSI bit. See OC/OCN enable description for more details (enable register (TIM1_CCERTIMx_CCER)N/A).
    pub mod MOE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: In response to a break 2 event. OC and OCN outputs are disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC and OCN outputs are enabled if their respective enable bits are set (CCxE, CCxNE in TIMx_CCER register).
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break filter This bit-field defines the frequency used to sample BRK input and the length of the digital filter applied to BRK. The digital filter is made of an event counter in which N consecutive events are needed to validate a transition on the output: Note: This bit cannot be modified when LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BKF {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: No filter, BRK acts asynchronously
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: fSAMPLING=fCK_INT, N=2
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: fSAMPLING=fCK_INT, N=4
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: fSAMPLING=fCK_INT, N=8
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: fSAMPLING=fDTS/2, N=6
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: fSAMPLING=fDTS/2, N=8
            pub const B_0x5: u32 = 0b0101;

            /// 0b0110: fSAMPLING=fDTS/4, N=6
            pub const B_0x6: u32 = 0b0110;

            /// 0b0111: fSAMPLING=fDTS/4, N=8
            pub const B_0x7: u32 = 0b0111;

            /// 0b1000: fSAMPLING=fDTS/8, N=6
            pub const B_0x8: u32 = 0b1000;

            /// 0b1001: fSAMPLING=fDTS/8, N=8
            pub const B_0x9: u32 = 0b1001;

            /// 0b1010: fSAMPLING=fDTS/16, N=5
            pub const B_0xA: u32 = 0b1010;

            /// 0b1011: fSAMPLING=fDTS/16, N=6
            pub const B_0xB: u32 = 0b1011;

            /// 0b1100: fSAMPLING=fDTS/16, N=8
            pub const B_0xC: u32 = 0b1100;

            /// 0b1101: fSAMPLING=fDTS/32, N=5
            pub const B_0xD: u32 = 0b1101;

            /// 0b1110: fSAMPLING=fDTS/32, N=6
            pub const B_0xE: u32 = 0b1110;

            /// 0b1111: fSAMPLING=fDTS/32, N=8
            pub const B_0xF: u32 = 0b1111;
        }
    }

    /// Break 2 filter This bit-field defines the frequency used to sample BRK2 input and the length of the digital filter applied to BRK2. The digital filter is made of an event counter in which N consecutive events are needed to validate a transition on the output: Note: This bit cannot be modified when LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BK2F {
        /// Offset (20 bits)
        pub const offset: u32 = 20;
        /// Mask (4 bits: 0b1111 << 20)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: No filter, BRK2 acts asynchronously
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: fSAMPLING=fCK_INT, N=2
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: fSAMPLING=fCK_INT, N=4
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: fSAMPLING=fCK_INT, N=8
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: fSAMPLING=fDTS/2, N=6
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: fSAMPLING=fDTS/2, N=8
            pub const B_0x5: u32 = 0b0101;

            /// 0b0110: fSAMPLING=fDTS/4, N=6
            pub const B_0x6: u32 = 0b0110;

            /// 0b0111: fSAMPLING=fDTS/4, N=8
            pub const B_0x7: u32 = 0b0111;

            /// 0b1000: fSAMPLING=fDTS/8, N=6
            pub const B_0x8: u32 = 0b1000;

            /// 0b1001: fSAMPLING=fDTS/8, N=8
            pub const B_0x9: u32 = 0b1001;

            /// 0b1010: fSAMPLING=fDTS/16, N=5
            pub const B_0xA: u32 = 0b1010;

            /// 0b1011: fSAMPLING=fDTS/16, N=6
            pub const B_0xB: u32 = 0b1011;

            /// 0b1100: fSAMPLING=fDTS/16, N=8
            pub const B_0xC: u32 = 0b1100;

            /// 0b1101: fSAMPLING=fDTS/32, N=5
            pub const B_0xD: u32 = 0b1101;

            /// 0b1110: fSAMPLING=fDTS/32, N=6
            pub const B_0xE: u32 = 0b1110;

            /// 0b1111: fSAMPLING=fDTS/32, N=8
            pub const B_0xF: u32 = 0b1111;
        }
    }

    /// Break 2 enable Note: The BRK2 must only be used with OSSR = OSSI = 1. Note: This bit cannot be modified when LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register). Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BK2E {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK2 disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break input BRK2 enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break 2 polarity Note: This bit cannot be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register). Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BK2P {
        /// Offset (25 bits)
        pub const offset: u32 = 25;
        /// Mask (1 bit: 1 << 25)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK2 is active low
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break input BRK2 is active high
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break Disarm This bit is cleared by hardware when no break source is active. The BKDSRM bit must be set by software to release the bidirectional output control (open-drain output in Hi-Z state) and then be polled it until it is reset by hardware, indicating that the fault condition has disappeared. Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BKDSRM {
        /// Offset (26 bits)
        pub const offset: u32 = 26;
        /// Mask (1 bit: 1 << 26)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK is armed
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break input BRK is disarmed
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break2 Disarm Refer to BKDSRM description
    pub mod BK2DSRM {
        /// Offset (27 bits)
        pub const offset: u32 = 27;
        /// Mask (1 bit: 1 << 27)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Break Bidirectional In the bidirectional mode (BKBID bit set to 1), the break input is configured both in input mode and in open drain output mode. Any active break event asserts a low logic level on the Break input to indicate an internal break event to external devices. Note: This bit cannot be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register). Note: Any write operation to this bit takes a delay of 1 APB clock cycle to become effective.
    pub mod BKBID {
        /// Offset (28 bits)
        pub const offset: u32 = 28;
        /// Mask (1 bit: 1 << 28)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Break input BRK in input mode
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Break input BRK in bidirectional mode
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Break2 bidirectional Refer to BKBID description
    pub mod BK2BID {
        /// Offset (29 bits)
        pub const offset: u32 = 29;
        /// Mask (1 bit: 1 << 29)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// DMA control register
pub mod DCR {

    /// DMA base address This 5-bits vector defines the base-address for DMA transfers (when read/write access are done through the TIMx_DMAR address). DBA is defined as an offset starting from the address of the TIMx_CR1 register. Example: ...
    pub mod DBA {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (5 bits: 0b11111 << 0)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00000: TIMx_CR1,
            pub const B_0x0: u32 = 0b00000;

            /// 0b00001: TIMx_CR2,
            pub const B_0x1: u32 = 0b00001;

            /// 0b00010: TIMx_SMCR,
            pub const B_0x2: u32 = 0b00010;
        }
    }

    /// DMA burst length This 5-bit vector defines the length of DMA transfers (the timer recognizes a burst transfer when a read or a write access is done to the TIMx_DMAR address), i.e. the number of transfers. Transfers can be in half-words or in bytes (see example below). ... Example: Let us consider the following transfer: DBL = 7 bytes & DBA = TIMx_CR1. If DBL = 7 bytes and DBA = TIMx_CR1 represents the address of the byte to be transferred, the address of the transfer should be given by the following equation: (TIMx_CR1 address) + DBA + (DMA index), where DMA index = DBL In this example, 7 bytes are added to (TIMx_CR1 address) + DBA, which gives us the address from/to which the data is copied. In this case, the transfer is done to 7 registers starting from the following address: (TIMx_CR1 address) + DBA According to the configuration of the DMA Data Size, several cases may occur: If the DMA Data Size is configured in half-words, 16-bit data is transferred to each of the 7 registers. If the DMA Data Size is configured in bytes, the data is also transferred to 7 registers: the first register contains the first MSB byte, the second register, the first LSB byte and so on. So with the transfer Timer, one also has to specify the size of data transferred by DMA.
    pub mod DBL {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (5 bits: 0b11111 << 8)
        pub const mask: u32 = 0b11111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b00000: 1 transfer
            pub const B_0x0: u32 = 0b00000;

            /// 0b00001: 2 transfers
            pub const B_0x1: u32 = 0b00001;

            /// 0b00010: 3 transfers
            pub const B_0x2: u32 = 0b00010;

            /// 0b10001: 18 transfers
            pub const B_0x11: u32 = 0b10001;
        }
    }
}

/// DMA address for full transfer
pub mod DMAR {

    /// DMA register for burst accesses A read or write operation to the DMAR register accesses the register located at the address (TIMx_CR1 address) + (DBA + DMA index) x 4 where TIMx_CR1 address is the address of the control register 1, DBA is the DMA base address configured in TIMx_DCR register, DMA index is automatically controlled by the DMA transfer, and ranges from 0 to DBL (DBL configured in TIMx_DCR).
    pub mod DMAB {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (32 bits: 0xffffffff << 0)
        pub const mask: u32 = 0xffffffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare mode register 2 (output mode)
pub mod CCMR3_Output {

    /// Output Compare 6 mode bit 3
    pub mod OC6M_3 {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (1 bit: 1 << 24)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Normal output compare mode (modes 0-7)
            pub const Normal: u32 = 0b0;

            /// 0b1: Extended output compare mode (modes 7-15)
            pub const Extended: u32 = 0b1;
        }
    }

    /// Output Compare 5 mode bit 3
    pub mod OC5M_3 {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (1 bit: 1 << 16)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::OC6M_3::RW;
    }

    /// Output compare 6 clear enable
    pub mod OC6CE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 6 mode
    pub mod OC6M {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (3 bits: 0b111 << 12)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: The comparison between the output compare register TIMx_CCRy and the counter TIMx_CNT has no effect on the outputs / OpmMode1: Retriggerable OPM mode 1 - In up-counting mode, the channel is active until a trigger event is detected (on TRGI signal). In down-counting mode, the channel is inactive
            pub const Frozen: u32 = 0b000;

            /// 0b001: Set channel to active level on match. OCyREF signal is forced high when the counter matches the capture/compare register / OpmMode2: Inversely to OpmMode1
            pub const ActiveOnMatch: u32 = 0b001;

            /// 0b010: Set channel to inactive level on match. OCyREF signal is forced low when the counter matches the capture/compare register / Reserved
            pub const InactiveOnMatch: u32 = 0b010;

            /// 0b011: OCyREF toggles when TIMx_CNT=TIMx_CCRy / Reserved
            pub const Toggle: u32 = 0b011;

            /// 0b100: OCyREF is forced low / CombinedPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC is the logical OR between OC1REF and OC2REF
            pub const ForceInactive: u32 = 0b100;

            /// 0b101: OCyREF is forced high / CombinedPwmMode2: OCyREF has the same behavior as in PWM mode 2. OCyREFC is the logical AND between OC1REF and OC2REF
            pub const ForceActive: u32 = 0b101;

            /// 0b110: In upcounting, channel is active as long as TIMx_CNT<TIMx_CCRy else inactive. In downcounting, channel is inactive as long as TIMx_CNT>TIMx_CCRy else active / AsymmetricPwmMode1: OCyREF has the same behavior as in PWM mode 1. OCyREFC outputs OC1REF when the counter is counting up, OC2REF when it is counting down
            pub const PwmMode1: u32 = 0b110;

            /// 0b111: Inversely to PwmMode1 / AsymmetricPwmMode2: Inversely to AsymmetricPwmMode1
            pub const PwmMode2: u32 = 0b111;
        }
    }

    /// Output compare 6 preload enable
    pub mod OC6PE {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 6 fast enable
    pub mod OC6FE {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 5 clear enable
    pub mod OC5CE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 5 mode
    pub mod OC5M {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (3 bits: 0b111 << 4)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        pub use super::OC6M::RW;
    }

    /// Output compare 5 preload enable
    pub mod OC5PE {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Output compare 5 fast enable
    pub mod OC5FE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// capture/compare register 4
pub mod CCR5 {

    /// Capture/Compare 5 value CCR5 is the value to be loaded in the actual capture/compare 5 register (preload value). It is loaded permanently if the preload feature is not selected in the TIMx_CCMR3 register (bit OC5PE). Else the preload value is copied in the active capture/compare 5 register when an update event occurs. The active capture/compare register contains the value to be compared to the counter TIMx_CNT and signaled on OC5 output.
    pub mod CCR5 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Group Channel 5 and Channel 1 Distortion on Channel 1 output: This bit can either have immediate effect or be preloaded and taken into account after an update event (if preload feature is selected in TIMxCCMR1). Note: it is also possible to apply this distortion on combined PWM signals.
    pub mod GC5C1 {
        /// Offset (29 bits)
        pub const offset: u32 = 29;
        /// Mask (1 bit: 1 << 29)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No effect of OC5REF on OC1REFC5
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC1REFC is the logical AND of OC1REFC and OC5REF
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Group Channel 5 and Channel 2 Distortion on Channel 2 output: This bit can either have immediate effect or be preloaded and taken into account after an update event (if preload feature is selected in TIMxCCMR1). Note: it is also possible to apply this distortion on combined PWM signals.
    pub mod GC5C2 {
        /// Offset (30 bits)
        pub const offset: u32 = 30;
        /// Mask (1 bit: 1 << 30)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No effect of OC5REF on OC2REFC
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC2REFC is the logical AND of OC2REFC and OC5REF
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Group Channel 5 and Channel 3 Distortion on Channel 3 output: This bit can either have immediate effect or be preloaded and taken into account after an update event (if preload feature is selected in TIMxCCMR2). Note: it is also possible to apply this distortion on combined PWM signals.
    pub mod GC5C3 {
        /// Offset (31 bits)
        pub const offset: u32 = 31;
        /// Mask (1 bit: 1 << 31)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: No effect of OC5REF on OC3REFC
            pub const B_0x0: u32 = 0b0;

            /// 0b1: OC3REFC is the logical AND of OC3REFC and OC5REF
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// capture/compare register 6
pub mod CCR6 {

    /// Capture/Compare value
    pub mod CCR6 {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TIM1 alternate function option register 1
pub mod AF1 {

    /// BRK BKIN input enable This bit enables the BKIN alternate function input for the timerâs BRK input. BKIN input is 'ORedâ with the other BRK sources. Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BKINE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: BKIN input disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: BKIN input enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// BRK BKIN input polarity This bit selects the BKIN alternate function input sensitivity. It must be programmed together with the BKP polarity bit. Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BKINP {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: BKIN input polarity is not inverted (active low if BKP=0, active high if BKP=1)
            pub const B_0x0: u32 = 0b0;

            /// 0b1: BKIN input polarity is inverted (active high if BKP=0, active low if BKP=1)
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// ETR source selection These bits select the ETR input source. Others: Reserved Note: These bits can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod ETRSEL {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (4 bits: 0b1111 << 14)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: ETR legacy mode
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: COMP1 output
            pub const B_0x1: u32 = 0b0001;

            /// 0b0010: COMP2 output
            pub const B_0x2: u32 = 0b0010;

            /// 0b0011: ADC1 AWD1
            pub const B_0x3: u32 = 0b0011;

            /// 0b0100: ADC1 AWD2
            pub const B_0x4: u32 = 0b0100;

            /// 0b0101: ADC1 AWD3
            pub const B_0x5: u32 = 0b0101;
        }
    }
}

/// TIM1 alternate function option register 2
pub mod AF2 {

    /// BRK2 BKIN input enable This bit enables the BKIN2 alternate function input for the timerâs BRK2 input. BKIN2 input is 'ORedâ with the other BRK2 sources. Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BK2INE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: BKIN2 input disabled
            pub const B_0x0: u32 = 0b0;

            /// 0b1: BKIN2 input enabled
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// BRK2 BKIN2 input polarity This bit selects the BKIN2 alternate function input sensitivity. It must be programmed together with the BK2P polarity bit. Note: This bit can not be modified as long as LOCK level 1 has been programmed (LOCK bits in TIMx_BDTR register).
    pub mod BK2INP {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: BKIN2 input polarity is not inverted (active low if BK2P=0, active high if BK2P=1)
            pub const B_0x0: u32 = 0b0;

            /// 0b1: BKIN2 input polarity is inverted (active high if BK2P=0, active low if BK2P=1)
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// TIM1 timer input selection register
pub mod TISEL {

    /// selects TI1\[0\] to TI1\[15\] input Others: Reserved
    pub mod TI1SEL {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (4 bits: 0b1111 << 0)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: TIM1_CH1 input
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: COMP1 output
            pub const B_0x1: u32 = 0b0001;
        }
    }

    /// selects TI2\[0\] to TI2\[15\] input Others: Reserved
    pub mod TI2SEL {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (4 bits: 0b1111 << 8)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: TIM1_CH2 input
            pub const B_0x0: u32 = 0b0000;

            /// 0b0001: COMP2 output
            pub const B_0x1: u32 = 0b0001;
        }
    }

    /// selects TI3\[0\] to TI3\[15\] input Others: Reserved
    pub mod TI3SEL {
        /// Offset (16 bits)
        pub const offset: u32 = 16;
        /// Mask (4 bits: 0b1111 << 16)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: TIM1_CH3 input
            pub const B_0x0: u32 = 0b0000;
        }
    }

    /// selects TI4\[0\] to TI4\[15\] input Others: Reserved
    pub mod TI4SEL {
        /// Offset (24 bits)
        pub const offset: u32 = 24;
        /// Mask (4 bits: 0b1111 << 24)
        pub const mask: u32 = 0b1111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0000: TIM1_CH4 input
            pub const B_0x0: u32 = 0b0000;
        }
    }
}
#[repr(C)]
pub struct RegisterBlock {
    /// control register 1
    pub CR1: RWRegister<u32>,

    /// control register 2
    pub CR2: RWRegister<u32>,

    /// slave mode control register
    pub SMCR: RWRegister<u32>,

    /// DMA/Interrupt enable register
    pub DIER: RWRegister<u32>,

    /// status register
    pub SR: RWRegister<u32>,

    /// event generation register
    pub EGR: WORegister<u32>,

    /// CCMR1_Output and CCMR1_Input
    /// CCMR1_Output: capture/compare mode register 1 (output mode)
    /// CCMR1_Input: capture/compare mode register 1 (output mode)
    pub CCMR1: RWRegister<u32>,

    /// CCMR2_Output and CCMR2_Input
    /// CCMR2_Output: capture/compare mode register 2 (output mode)
    /// CCMR2_Input: capture/compare mode register 2 (output mode)
    pub CCMR2: RWRegister<u32>,

    /// capture/compare enable register
    pub CCER: RWRegister<u32>,

    /// counter
    pub CNT: RWRegister<u32>,

    /// prescaler
    pub PSC: RWRegister<u32>,

    /// auto-reload register
    pub ARR: RWRegister<u32>,

    /// repetition counter register
    pub RCR: RWRegister<u32>,

    /// capture/compare register 1
    pub CCR1: RWRegister<u32>,

    /// capture/compare register 2
    pub CCR2: RWRegister<u32>,

    /// capture/compare register 3
    pub CCR3: RWRegister<u32>,

    /// capture/compare register 4
    pub CCR4: RWRegister<u32>,

    /// break and dead-time register
    pub BDTR: RWRegister<u32>,

    /// DMA control register
    pub DCR: RWRegister<u32>,

    /// DMA address for full transfer
    pub DMAR: RWRegister<u32>,

    _reserved1: [u32; 1],

    /// capture/compare mode register 2 (output mode)
    pub CCMR3_Output: RWRegister<u32>,

    /// capture/compare register 4
    pub CCR5: RWRegister<u32>,

    /// capture/compare register 6
    pub CCR6: RWRegister<u32>,

    /// TIM1 alternate function option register 1
    pub AF1: RWRegister<u32>,

    /// TIM1 alternate function option register 2
    pub AF2: RWRegister<u32>,

    /// TIM1 timer input selection register
    pub TISEL: RWRegister<u32>,
}
pub struct ResetValues {
    pub CR1: u32,
    pub CR2: u32,
    pub SMCR: u32,
    pub DIER: u32,
    pub SR: u32,
    pub EGR: u32,
    pub CCMR1: u32,
    pub CCMR2: u32,
    pub CCER: u32,
    pub CNT: u32,
    pub PSC: u32,
    pub ARR: u32,
    pub RCR: u32,
    pub CCR1: u32,
    pub CCR2: u32,
    pub CCR3: u32,
    pub CCR4: u32,
    pub BDTR: u32,
    pub DCR: u32,
    pub DMAR: u32,
    pub CCMR3_Output: u32,
    pub CCR5: u32,
    pub CCR6: u32,
    pub AF1: u32,
    pub AF2: u32,
    pub TISEL: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}

/// Access functions for the TIM1 peripheral instance
pub mod TIM1 {
    use super::ResetValues;

    #[cfg(not(feature = "nosync"))]
    use super::Instance;

    #[cfg(not(feature = "nosync"))]
    const INSTANCE: Instance = Instance {
        addr: 0x40012c00,
        _marker: ::core::marker::PhantomData,
    };

    /// Reset values for each field in TIM1
    pub const reset: ResetValues = ResetValues {
        CR1: 0x00000000,
        CR2: 0x00000000,
        SMCR: 0x00000000,
        DIER: 0x00000000,
        SR: 0x00000000,
        EGR: 0x00000000,
        CCMR1: 0x00000000,
        CCMR2: 0x00000000,
        CCER: 0x00000000,
        CNT: 0x00000000,
        PSC: 0x00000000,
        ARR: 0x0000FFFF,
        RCR: 0x00000000,
        CCR1: 0x00000000,
        CCR2: 0x00000000,
        CCR3: 0x00000000,
        CCR4: 0x00000000,
        BDTR: 0x00000000,
        DCR: 0x00000000,
        DMAR: 0x00000000,
        CCMR3_Output: 0x00000000,
        CCR5: 0x00000000,
        CCR6: 0x00000000,
        AF1: 0x00000001,
        AF2: 0x00000001,
        TISEL: 0x00000000,
    };

    #[cfg(not(feature = "nosync"))]
    #[allow(renamed_and_removed_lints)]
    #[allow(private_no_mangle_statics)]
    #[no_mangle]
    static mut TIM1_TAKEN: bool = false;

    /// Safe access to TIM1
    ///
    /// This function returns `Some(Instance)` if this instance is not
    /// currently taken, and `None` if it is. This ensures that if you
    /// do get `Some(Instance)`, you are ensured unique access to
    /// the peripheral and there cannot be data races (unless other
    /// code uses `unsafe`, of course). You can then pass the
    /// `Instance` around to other functions as required. When you're
    /// done with it, you can call `release(instance)` to return it.
    ///
    /// `Instance` itself dereferences to a `RegisterBlock`, which
    /// provides access to the peripheral's registers.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn take() -> Option<Instance> {
        external_cortex_m::interrupt::free(|_| unsafe {
            if TIM1_TAKEN {
                None
            } else {
                TIM1_TAKEN = true;
                Some(INSTANCE)
            }
        })
    }

    /// Release exclusive access to TIM1
    ///
    /// This function allows you to return an `Instance` so that it
    /// is available to `take()` again. This function will panic if
    /// you return a different `Instance` or if this instance is not
    /// already taken.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn release(inst: Instance) {
        external_cortex_m::interrupt::free(|_| unsafe {
            if TIM1_TAKEN && inst.addr == INSTANCE.addr {
                TIM1_TAKEN = false;
            } else {
                panic!("Released a peripheral which was not taken");
            }
        });
    }

    /// Unsafely steal TIM1
    ///
    /// This function is similar to take() but forcibly takes the
    /// Instance, marking it as taken irregardless of its previous
    /// state.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub unsafe fn steal() -> Instance {
        TIM1_TAKEN = true;
        INSTANCE
    }
}

/// Raw pointer to TIM1
///
/// Dereferencing this is unsafe because you are not ensured unique
/// access to the peripheral, so you may encounter data races with
/// other users of this peripheral. It is up to you to ensure you
/// will not cause data races.
///
/// This constant is provided for ease of use in unsafe code: you can
/// simply call for example `write_reg!(gpio, GPIOA, ODR, 1);`.
pub const TIM1: *const RegisterBlock = 0x40012c00 as *const _;
