#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! System configuration controller
//!
//! Used by: stm32g051, stm32g061, stm32g0b0, stm32g0b1, stm32g0c1

use crate::RWRegister;
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// VREFBUF control and status register
pub mod VREFBUF_CSR {

    /// Voltage reference buffer mode enable This bit is used to enable the voltage reference buffer mode.
    pub mod ENVR {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Internal voltage reference mode disable (external voltage reference mode).
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Internal voltage reference mode (reference buffer enable or hold mode) enable.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// High impedance mode This bit controls the analog switch to connect or not the VREF+ pin. Refer to for the mode descriptions depending on ENVR bit configuration.
    pub mod HIZ {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: VREF+ pin is internally connected to the voltage reference buffer output.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: VREF+ pin is high impedance.
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Voltage reference scale This bit selects the value generated by the voltage reference buffer.
    pub mod VRS {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Voltage reference set to VREF_OUT1 (around 2.048 V).
            pub const B_0x0: u32 = 0b0;

            /// 0b1: Voltage reference set to VREF_OUT2 (around 2.5 V).
            pub const B_0x1: u32 = 0b1;
        }
    }

    /// Voltage reference buffer ready
    pub mod VRR {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (1 bit: 1 << 3)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: the voltage reference buffer output is not ready.
            pub const B_0x0: u32 = 0b0;

            /// 0b1: the voltage reference buffer output reached the requested level.
            pub const B_0x1: u32 = 0b1;
        }
    }
}

/// VREFBUF calibration control register
pub mod VREFBUF_CCR {

    /// Trimming code These bits are automatically initialized after reset with the trimming value stored in the Flash memory during the production test. Writing into these bits allows the tuning of the internal reference buffer voltage.
    pub mod TRIM {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (6 bits: 0x3f << 0)
        pub const mask: u32 = 0x3f << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}
#[repr(C)]
pub struct RegisterBlock {
    /// VREFBUF control and status register
    pub VREFBUF_CSR: RWRegister<u32>,

    /// VREFBUF calibration control register
    pub VREFBUF_CCR: RWRegister<u32>,
}
pub struct ResetValues {
    pub VREFBUF_CSR: u32,
    pub VREFBUF_CCR: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}
