#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! Serial peripheral interface
//!
//! Used by: stm32f100, stm32f102

use crate::{RORegister, RWRegister};
#[cfg(not(feature = "nosync"))]
use core::marker::PhantomData;

/// control register 1
pub mod CR1 {

    /// Bidirectional data mode enable
    pub mod BIDIMODE {
        /// Offset (15 bits)
        pub const offset: u32 = 15;
        /// Mask (1 bit: 1 << 15)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: 2-line unidirectional data mode selected
            pub const Unidirectional: u32 = 0b0;

            /// 0b1: 1-line bidirectional data mode selected
            pub const Bidirectional: u32 = 0b1;
        }
    }

    /// Output enable in bidirectional mode
    pub mod BIDIOE {
        /// Offset (14 bits)
        pub const offset: u32 = 14;
        /// Mask (1 bit: 1 << 14)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Output disabled (receive-only mode)
            pub const OutputDisabled: u32 = 0b0;

            /// 0b1: Output enabled (transmit-only mode)
            pub const OutputEnabled: u32 = 0b1;
        }
    }

    /// Hardware CRC calculation enable
    pub mod CRCEN {
        /// Offset (13 bits)
        pub const offset: u32 = 13;
        /// Mask (1 bit: 1 << 13)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CRC calculation disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: CRC calculation enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// CRC transfer next
    pub mod CRCNEXT {
        /// Offset (12 bits)
        pub const offset: u32 = 12;
        /// Mask (1 bit: 1 << 12)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Next transmit value is from Tx buffer
            pub const TxBuffer: u32 = 0b0;

            /// 0b1: Next transmit value is from Tx CRC register
            pub const CRC: u32 = 0b1;
        }
    }

    /// Data frame format
    pub mod DFF {
        /// Offset (11 bits)
        pub const offset: u32 = 11;
        /// Mask (1 bit: 1 << 11)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: 8-bit data frame format is selected for transmission/reception
            pub const EightBit: u32 = 0b0;

            /// 0b1: 16-bit data frame format is selected for transmission/reception
            pub const SixteenBit: u32 = 0b1;
        }
    }

    /// Receive only
    pub mod RXONLY {
        /// Offset (10 bits)
        pub const offset: u32 = 10;
        /// Mask (1 bit: 1 << 10)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Full duplex (Transmit and receive)
            pub const FullDuplex: u32 = 0b0;

            /// 0b1: Output disabled (Receive-only mode)
            pub const OutputDisabled: u32 = 0b1;
        }
    }

    /// Software slave management
    pub mod SSM {
        /// Offset (9 bits)
        pub const offset: u32 = 9;
        /// Mask (1 bit: 1 << 9)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Software slave management disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Software slave management enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Internal slave select
    pub mod SSI {
        /// Offset (8 bits)
        pub const offset: u32 = 8;
        /// Mask (1 bit: 1 << 8)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: 0 is forced onto the NSS pin and the I/O value of the NSS pin is ignored
            pub const SlaveSelected: u32 = 0b0;

            /// 0b1: 1 is forced onto the NSS pin and the I/O value of the NSS pin is ignored
            pub const SlaveNotSelected: u32 = 0b1;
        }
    }

    /// Frame format
    pub mod LSBFIRST {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Data is transmitted/received with the MSB first
            pub const MSBFirst: u32 = 0b0;

            /// 0b1: Data is transmitted/received with the LSB first
            pub const LSBFirst: u32 = 0b1;
        }
    }

    /// SPI enable
    pub mod SPE {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Peripheral disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Peripheral enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Baud rate control
    pub mod BR {
        /// Offset (3 bits)
        pub const offset: u32 = 3;
        /// Mask (3 bits: 0b111 << 3)
        pub const mask: u32 = 0b111 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b000: f_PCLK / 2
            pub const Div2: u32 = 0b000;

            /// 0b001: f_PCLK / 4
            pub const Div4: u32 = 0b001;

            /// 0b010: f_PCLK / 8
            pub const Div8: u32 = 0b010;

            /// 0b011: f_PCLK / 16
            pub const Div16: u32 = 0b011;

            /// 0b100: f_PCLK / 32
            pub const Div32: u32 = 0b100;

            /// 0b101: f_PCLK / 64
            pub const Div64: u32 = 0b101;

            /// 0b110: f_PCLK / 128
            pub const Div128: u32 = 0b110;

            /// 0b111: f_PCLK / 256
            pub const Div256: u32 = 0b111;
        }
    }

    /// Master selection
    pub mod MSTR {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Slave configuration
            pub const Slave: u32 = 0b0;

            /// 0b1: Master configuration
            pub const Master: u32 = 0b1;
        }
    }

    /// Clock polarity
    pub mod CPOL {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CK to 0 when idle
            pub const IdleLow: u32 = 0b0;

            /// 0b1: CK to 1 when idle
            pub const IdleHigh: u32 = 0b1;
        }
    }

    /// Clock phase
    pub mod CPHA {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: The first clock transition is the first data capture edge
            pub const FirstEdge: u32 = 0b0;

            /// 0b1: The second clock transition is the first data capture edge
            pub const SecondEdge: u32 = 0b1;
        }
    }
}

/// control register 2
pub mod CR2 {

    /// Tx buffer empty interrupt enable
    pub mod TXEIE {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: TXE interrupt masked
            pub const Masked: u32 = 0b0;

            /// 0b1: TXE interrupt not masked
            pub const NotMasked: u32 = 0b1;
        }
    }

    /// RX buffer not empty interrupt enable
    pub mod RXNEIE {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: RXE interrupt masked
            pub const Masked: u32 = 0b0;

            /// 0b1: RXE interrupt not masked
            pub const NotMasked: u32 = 0b1;
        }
    }

    /// Error interrupt enable
    pub mod ERRIE {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Error interrupt masked
            pub const Masked: u32 = 0b0;

            /// 0b1: Error interrupt not masked
            pub const NotMasked: u32 = 0b1;
        }
    }

    /// SS output enable
    pub mod SSOE {
        /// Offset (2 bits)
        pub const offset: u32 = 2;
        /// Mask (1 bit: 1 << 2)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: SS output is disabled in master mode
            pub const Disabled: u32 = 0b0;

            /// 0b1: SS output is enabled in master mode
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Tx buffer DMA enable
    pub mod TXDMAEN {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Tx buffer DMA disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Tx buffer DMA enabled
            pub const Enabled: u32 = 0b1;
        }
    }

    /// Rx buffer DMA enable
    pub mod RXDMAEN {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Rx buffer DMA disabled
            pub const Disabled: u32 = 0b0;

            /// 0b1: Rx buffer DMA enabled
            pub const Enabled: u32 = 0b1;
        }
    }
}

/// status register
pub mod SR {

    /// Busy flag
    pub mod BSY {
        /// Offset (7 bits)
        pub const offset: u32 = 7;
        /// Mask (1 bit: 1 << 7)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: SPI not busy
            pub const NotBusy: u32 = 0b0;

            /// 0b1: SPI busy
            pub const Busy: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Overrun flag
    pub mod OVR {
        /// Offset (6 bits)
        pub const offset: u32 = 6;
        /// Mask (1 bit: 1 << 6)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No overrun occurred
            pub const NoOverrun: u32 = 0b0;

            /// 0b1: Overrun occurred
            pub const Overrun: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// Mode fault
    pub mod MODF {
        /// Offset (5 bits)
        pub const offset: u32 = 5;
        /// Mask (1 bit: 1 << 5)
        pub const mask: u32 = 1 << offset;
        /// Read-only values
        pub mod R {

            /// 0b0: No mode fault occurred
            pub const NoFault: u32 = 0b0;

            /// 0b1: Mode fault occurred
            pub const Fault: u32 = 0b1;
        }
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }

    /// CRC error flag
    pub mod CRCERR {
        /// Offset (4 bits)
        pub const offset: u32 = 4;
        /// Mask (1 bit: 1 << 4)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: CRC value received matches the SPIx_RXCRCR value
            pub const Match: u32 = 0b0;

            /// 0b1: CRC value received does not match the SPIx_RXCRCR value
            pub const NoMatch: u32 = 0b1;
        }
    }

    /// Transmit buffer empty
    pub mod TXE {
        /// Offset (1 bits)
        pub const offset: u32 = 1;
        /// Mask (1 bit: 1 << 1)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Tx buffer not empty
            pub const NotEmpty: u32 = 0b0;

            /// 0b1: Tx buffer empty
            pub const Empty: u32 = 0b1;
        }
    }

    /// Receive buffer not empty
    pub mod RXNE {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (1 bit: 1 << 0)
        pub const mask: u32 = 1 << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values
        pub mod RW {

            /// 0b0: Rx buffer empty
            pub const Empty: u32 = 0b0;

            /// 0b1: Rx buffer not empty
            pub const NotEmpty: u32 = 0b1;
        }
    }
}

/// data register
pub mod DR {

    /// Data register
    pub mod DR {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// CRC polynomial register
pub mod CRCPR {

    /// CRC polynomial register
    pub mod CRCPOLY {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// RX CRC register
pub mod RXCRCR {

    /// Rx CRC register
    pub mod RxCRC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}

/// TX CRC register
pub mod TXCRCR {

    /// Tx CRC register
    pub mod TxCRC {
        /// Offset (0 bits)
        pub const offset: u32 = 0;
        /// Mask (16 bits: 0xffff << 0)
        pub const mask: u32 = 0xffff << offset;
        /// Read-only values (empty)
        pub mod R {}
        /// Write-only values (empty)
        pub mod W {}
        /// Read-write values (empty)
        pub mod RW {}
    }
}
#[repr(C)]
pub struct RegisterBlock {
    /// control register 1
    pub CR1: RWRegister<u32>,

    /// control register 2
    pub CR2: RWRegister<u32>,

    /// status register
    pub SR: RWRegister<u32>,

    /// data register
    pub DR: RWRegister<u32>,

    /// CRC polynomial register
    pub CRCPR: RWRegister<u32>,

    /// RX CRC register
    pub RXCRCR: RORegister<u32>,

    /// TX CRC register
    pub TXCRCR: RORegister<u32>,
}
pub struct ResetValues {
    pub CR1: u32,
    pub CR2: u32,
    pub SR: u32,
    pub DR: u32,
    pub CRCPR: u32,
    pub RXCRCR: u32,
    pub TXCRCR: u32,
}
#[cfg(not(feature = "nosync"))]
pub struct Instance {
    pub(crate) addr: u32,
    pub(crate) _marker: PhantomData<*const RegisterBlock>,
}
#[cfg(not(feature = "nosync"))]
impl ::core::ops::Deref for Instance {
    type Target = RegisterBlock;
    #[inline(always)]
    fn deref(&self) -> &RegisterBlock {
        unsafe { &*(self.addr as *const _) }
    }
}
#[cfg(feature = "rtic")]
unsafe impl Send for Instance {}
