#![allow(non_snake_case, non_upper_case_globals)]
#![allow(non_camel_case_types)]
//! Universal asynchronous receiver transmitter
//!
//! Used by: stm32f100, stm32f101, stm32f103, stm32f107

#[cfg(not(feature = "nosync"))]
pub use crate::stm32f1::peripherals::uart::Instance;
pub use crate::stm32f1::peripherals::uart::{RegisterBlock, ResetValues};
pub use crate::stm32f1::peripherals::uart::{BRR, CR1, CR2, CR3, DR, SR};

/// Access functions for the UART4 peripheral instance
pub mod UART4 {
    use super::ResetValues;

    #[cfg(not(feature = "nosync"))]
    use super::Instance;

    #[cfg(not(feature = "nosync"))]
    const INSTANCE: Instance = Instance {
        addr: 0x40004c00,
        _marker: ::core::marker::PhantomData,
    };

    /// Reset values for each field in UART4
    pub const reset: ResetValues = ResetValues {
        SR: 0x00000000,
        DR: 0x00000000,
        BRR: 0x00000000,
        CR1: 0x00000000,
        CR2: 0x00000000,
        CR3: 0x00000000,
    };

    #[cfg(not(feature = "nosync"))]
    #[allow(renamed_and_removed_lints)]
    #[allow(private_no_mangle_statics)]
    #[no_mangle]
    static mut UART4_TAKEN: bool = false;

    /// Safe access to UART4
    ///
    /// This function returns `Some(Instance)` if this instance is not
    /// currently taken, and `None` if it is. This ensures that if you
    /// do get `Some(Instance)`, you are ensured unique access to
    /// the peripheral and there cannot be data races (unless other
    /// code uses `unsafe`, of course). You can then pass the
    /// `Instance` around to other functions as required. When you're
    /// done with it, you can call `release(instance)` to return it.
    ///
    /// `Instance` itself dereferences to a `RegisterBlock`, which
    /// provides access to the peripheral's registers.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn take() -> Option<Instance> {
        external_cortex_m::interrupt::free(|_| unsafe {
            if UART4_TAKEN {
                None
            } else {
                UART4_TAKEN = true;
                Some(INSTANCE)
            }
        })
    }

    /// Release exclusive access to UART4
    ///
    /// This function allows you to return an `Instance` so that it
    /// is available to `take()` again. This function will panic if
    /// you return a different `Instance` or if this instance is not
    /// already taken.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn release(inst: Instance) {
        external_cortex_m::interrupt::free(|_| unsafe {
            if UART4_TAKEN && inst.addr == INSTANCE.addr {
                UART4_TAKEN = false;
            } else {
                panic!("Released a peripheral which was not taken");
            }
        });
    }

    /// Unsafely steal UART4
    ///
    /// This function is similar to take() but forcibly takes the
    /// Instance, marking it as taken irregardless of its previous
    /// state.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub unsafe fn steal() -> Instance {
        UART4_TAKEN = true;
        INSTANCE
    }
}

/// Raw pointer to UART4
///
/// Dereferencing this is unsafe because you are not ensured unique
/// access to the peripheral, so you may encounter data races with
/// other users of this peripheral. It is up to you to ensure you
/// will not cause data races.
///
/// This constant is provided for ease of use in unsafe code: you can
/// simply call for example `write_reg!(gpio, GPIOA, ODR, 1);`.
pub const UART4: *const RegisterBlock = 0x40004c00 as *const _;

/// Access functions for the UART5 peripheral instance
pub mod UART5 {
    use super::ResetValues;

    #[cfg(not(feature = "nosync"))]
    use super::Instance;

    #[cfg(not(feature = "nosync"))]
    const INSTANCE: Instance = Instance {
        addr: 0x40005000,
        _marker: ::core::marker::PhantomData,
    };

    /// Reset values for each field in UART5
    pub const reset: ResetValues = ResetValues {
        SR: 0x00000000,
        DR: 0x00000000,
        BRR: 0x00000000,
        CR1: 0x00000000,
        CR2: 0x00000000,
        CR3: 0x00000000,
    };

    #[cfg(not(feature = "nosync"))]
    #[allow(renamed_and_removed_lints)]
    #[allow(private_no_mangle_statics)]
    #[no_mangle]
    static mut UART5_TAKEN: bool = false;

    /// Safe access to UART5
    ///
    /// This function returns `Some(Instance)` if this instance is not
    /// currently taken, and `None` if it is. This ensures that if you
    /// do get `Some(Instance)`, you are ensured unique access to
    /// the peripheral and there cannot be data races (unless other
    /// code uses `unsafe`, of course). You can then pass the
    /// `Instance` around to other functions as required. When you're
    /// done with it, you can call `release(instance)` to return it.
    ///
    /// `Instance` itself dereferences to a `RegisterBlock`, which
    /// provides access to the peripheral's registers.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn take() -> Option<Instance> {
        external_cortex_m::interrupt::free(|_| unsafe {
            if UART5_TAKEN {
                None
            } else {
                UART5_TAKEN = true;
                Some(INSTANCE)
            }
        })
    }

    /// Release exclusive access to UART5
    ///
    /// This function allows you to return an `Instance` so that it
    /// is available to `take()` again. This function will panic if
    /// you return a different `Instance` or if this instance is not
    /// already taken.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub fn release(inst: Instance) {
        external_cortex_m::interrupt::free(|_| unsafe {
            if UART5_TAKEN && inst.addr == INSTANCE.addr {
                UART5_TAKEN = false;
            } else {
                panic!("Released a peripheral which was not taken");
            }
        });
    }

    /// Unsafely steal UART5
    ///
    /// This function is similar to take() but forcibly takes the
    /// Instance, marking it as taken irregardless of its previous
    /// state.
    #[cfg(not(feature = "nosync"))]
    #[inline]
    pub unsafe fn steal() -> Instance {
        UART5_TAKEN = true;
        INSTANCE
    }
}

/// Raw pointer to UART5
///
/// Dereferencing this is unsafe because you are not ensured unique
/// access to the peripheral, so you may encounter data races with
/// other users of this peripheral. It is up to you to ensure you
/// will not cause data races.
///
/// This constant is provided for ease of use in unsafe code: you can
/// simply call for example `write_reg!(gpio, GPIOA, ODR, 1);`.
pub const UART5: *const RegisterBlock = 0x40005000 as *const _;
