//! GATT commands and types needed for those commands.

use bluetooth_hci as hci;
use byteorder;
use nb;

use bbqueue::ArrayLength;
use byteorder::{ByteOrder, LittleEndian};

use crate::bluetooth::{bitflags::bitflags, event, opcode, RadioCoprocessor};

/// GATT-specific commands for the [`ActiveBlueNRG`](crate::ActiveBlueNRG).
pub trait Commands {
    /// Type of communication errors.
    type Error;

    /// Initialize the GATT server on the slave device. Initialize all the pools and active
    /// nodes. Also it adds GATT service with service changed characteristic. Until this command is
    /// issued the GATT channel will not process any commands even if the connection is opened. This
    /// command has to be given before using any of the GAP features.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [Command Complete](event::command::ReturnParameters::GattInit) event is generated.
    fn init(&mut self) -> nb::Result<(), Self::Error>;

    /// Initialize the GATT server on a slave device.
    ///
    /// This function exists to prevent name conflicts with other Commands traits' init methods.
    fn init_gatt(&mut self) -> nb::Result<(), Self::Error> {
        self.init()
    }

    /// Add a service to GATT Server.
    ///
    /// When a service is created in the server, the host needs to reserve the handle ranges for
    /// this service using [`max_attribute_records`](AddServiceParameters::max_attribute_records).
    /// This parameter specifies the maximum number of attribute records that can be added to this
    /// service (including the service attribute, include attribute, characteristic attribute,
    /// characteristic value attribute and characteristic descriptor attribute). Handle of the
    /// created service is returned in command complete event.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [Command complete](event::command::ReturnParameters::GattAddService) event is
    /// generated.
    fn add_service(&mut self, params: &AddServiceParameters) -> nb::Result<(), Self::Error>;

    /// Include a service to another service.
    ///
    /// Attribute server creates an INCLUDE definition attribute.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [Command complete](event::command::ReturnParameters::GattIncludeService) event is
    /// generated.
    fn include_service(&mut self, params: &IncludeServiceParameters)
        -> nb::Result<(), Self::Error>;

    /// Add a characteristic to a service.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// When the command is completed, a [command
    /// complete](event::command::ReturnParameters::GattAddCharacteristic) event will be
    /// generated by the controller which carries the status of the command and the handle of the
    /// characteristic as parameters.
    fn add_characteristic(
        &mut self,
        params: &AddCharacteristicParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Add a characteristic descriptor to a service.
    ///
    /// # Errors
    ///
    /// - [DescriptorTooLong](Error::DescriptorTooLong) if the [descriptor
    ///   value](AddDescriptorParameters::descriptor_value) is longer than the [maximum descriptor
    ///   value length](AddDescriptorParameters::descriptor_value_max_len).
    /// - [DescriptorBufferTooLong](Error::DescriptorBufferTooLong) if the [descriptor
    ///   value maximum length](AddDescriptorParameters::descriptor_value_max_len) is so large that
    ///   the serialized structure may be more than 255 bytes. The maximum size is 227.
    /// - Underlying communication errors.
    ///
    /// # Generated events
    ///
    /// When this command is completed, a [command
    /// complete](event::command::ReturnParameters::GattAddCharacteristicDescriptor) event
    /// will be generated by the controller which carries the status of the command and the handle
    /// of the characteristic descriptor.
    fn add_characteristic_descriptor<'a>(
        &mut self,
        params: &AddDescriptorParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Update a characteristic value in a service.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [characteristic
    ///   value](UpdateCharacteristicValueParameters::value] is so long that the command packet
    ///   would exceed 255 bytes. The maximum allowed length is 249 bytes.
    /// - Underlying communication errors.
    ///
    /// # Generated events
    ///
    /// When the command has completed, the controller will generate a [command
    /// complete](event::command::ReturnParameters::GattUpdateCharacteristicValue) event.
    fn update_characteristic_value<'a>(
        &mut self,
        params: &UpdateCharacteristicValueParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Delete the characteristic specified from the service.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// When the command has completed, the controller will generate a [command
    /// complete](event::command::ReturnParameters::GattDeleteCharacteristic) event.
    fn delete_characteristic(
        &mut self,
        service: ServiceHandle,
        characteristic: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Delete the service specified from the GATT server database.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// When the command has completed, the controller will generate a [command
    /// complete](event::command::ReturnParameters::GattDeleteService) event.
    fn delete_service(&mut self, service: ServiceHandle) -> nb::Result<(), Self::Error>;

    /// Delete the Include definition from the service.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// When the command has completed, the controller will generate a [command
    /// complete](event::command::ReturnParameters::GattDeleteIncludedService) event.
    fn delete_included_service(
        &mut self,
        params: &DeleteIncludedServiceParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Allows masking events from the GATT.
    ///
    /// The default configuration is all the events masked.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattSetEventMask) event is
    /// generated on the completion of the command.
    fn set_event_mask(&mut self, mask: Event) -> nb::Result<(), Self::Error>;

    /// Allows masking events from the GATT.
    ///
    /// This function exists to prevent name conflicts with other Commands traits' set_event_mask
    /// methods.
    fn set_gatt_event_mask(&mut self, mask: Event) -> nb::Result<(), Self::Error> {
        self.set_event_mask(mask)
    }

    /// Perform an ATT MTU exchange.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. When the ATT MTU exchange procedure is completed, an [Exchange MTU
    /// Response](event::BlueNRGEvent::AttExchangeMtuResponse) event is generated. Also, a
    /// [procedure complete](event::BlueNRGEvent::GattProcedureComplete) event is generated
    /// to indicate end of procedure.
    fn exchange_configuration(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Post the Find information request.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Find Information
    /// Response](event::BlueNRGEvent::AttFindInformationResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn find_information_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error>;

    /// Post the Find by type value request.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [attribute
    ///   value](FindByTypeValueParameters::value) to find is too long to fit in one command packet
    ///   (255 bytes). The maximum length is 246 bytes.
    /// - Underlying communication errors.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Find by Type Value
    /// Response](event::BlueNRGEvent::AttFindByTypeValueResponse) event. The end of the
    /// procedure is indicated by a [Gatt Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn find_by_type_value_request(
        &mut self,
        params: &FindByTypeValueParameters,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Send a Read By Type Request.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read by Type
    /// Response](event::BlueNRGEvent::AttReadByTypeResponse) event. The end of the procedure
    /// is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_by_type_request(
        &mut self,
        params: &ReadByTypeParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Sends a Read By Group Type request.
    ///
    /// The Read By Group Type Request is used to obtain the values of grouping attributes where the
    /// attribute type is known but the handle is not known. Grouping attributes are defined at GATT
    /// layer. The grouping attribute types are: Primary Service, Secondary Service and
    /// Characteristic.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read by Group Type
    /// Response](event::BlueNRGEvent::AttReadByGroupTypeResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_by_group_type_request(
        &mut self,
        params: &ReadByTypeParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Sends a Prepare Write request.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the attribute value is so long that the
    ///   serialized command would be longer than 255 bytes. The maximum length is 248 bytes.
    /// - Underlying comminication errors
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Prepare Write
    /// Response](event::BlueNRGEvent::AttPrepareWriteResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn prepare_write_request<'a>(
        &mut self,
        params: &WriteRequest<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Sends an Execute Write Request to write all pending prepared writes.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The result of the procedure is given through the [Execute Write
    /// Response](event::BlueNRGEvent::AttExecuteWriteResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn execute_write_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Sends an Execute Write Request to discard prepared writes.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The result of the procedure is given through the [Execute Write
    /// Response](event::BlueNRGEvent::AttExecuteWriteResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn cancel_write_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error>;

    /// This command will start the GATT client procedure to discover all primary services on the
    /// server.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read By Group
    /// Response](event::BlueNRGEvent::AttReadByGroupTypeResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn discover_all_primary_services(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error>;

    /// This command will start the procedure to discover the primary services of the specified
    /// UUID on the server.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Find By Type Value
    /// Response](event::BlueNRGEvent::AttFindByTypeValueResponse) event. The end of the
    /// procedure is indicated by a [Gatt Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn discover_primary_services_by_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to find all included services.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read By Type
    /// Response](event::BlueNRGEvent::AttReadByTypeResponse) event. The end of the procedure
    /// is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn find_included_services(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        service_handle_range: Range<ServiceHandle>,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to discover all the characteristics of a given service.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read By Type
    /// Response](event::BlueNRGEvent::AttReadByTypeResponse) event. The end of the procedure
    /// is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn discover_all_characteristics_of_service(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_handle_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to discover all the characteristics specified by the UUID.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Discover or Read
    /// Characteristic By
    /// UUID](event::BlueNRGEvent::GattDiscoverOrReadCharacteristicByUuidResponse) event. The
    /// end of the procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn discover_characteristics_by_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_handle_range: Range<CharacteristicHandle>,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to discover all characteristic descriptors on the server.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Find Information
    /// Response](event::BlueNRGEvent::AttFindInformationResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn discover_all_characteristic_descriptors(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to read the attribute value.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read
    /// Response](event::BlueNRGEvent::AttReadResponse) event. The end of the procedure is
    /// indicated by a [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete)
    /// event.
    fn read_characteristic_value(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to read all the characteristics specified by the UUID.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Discover or Read
    /// Characteristic by
    /// UUID](event::BlueNRGEvent::GattDiscoverOrReadCharacteristicByUuidResponse) event. The
    /// end of the procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_characteristic_using_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle_range: Range<CharacteristicHandle>,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to read a long characteristic value.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read Blob
    /// Response](event::BlueNRGEvent::AttReadBlobResponse) event. The end of the procedure
    /// is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_long_characteristic_value(
        &mut self,
        params: &LongCharacteristicReadParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Start a procedure to read multiple characteristic values from a server.
    ///
    /// This sub-procedure is used to read multiple Characteristic Values from a server when the
    /// client knows the Characteristic Value Handles.
    ///
    /// # Errors
    ///
    /// - [TooManyHandlesToRead](Error::TooManyHandlesToRead) if the number of handles to read would
    ///   cause the length of the serialized command to exceed 255 bytes. The maximum number of
    ///   handles is 126.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read Multiple
    /// Response](event::BlueNRGEvent::AttReadMultipleResponse) event. The end of the
    /// procedure is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_multiple_characteristic_values<'a>(
        &mut self,
        params: &MultipleCharacteristicReadParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to write a characteristic value.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [value](CharacteristicValue::value)
    ///   is too long to fit in one command packet. The maximum length is 250 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. When the procedure is completed, a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event is generated.
    fn write_characteristic_value<'a>(
        &mut self,
        params: &CharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to write a long characteristic value.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the
    ///   [value](LongCharacteristicValue::value) is too long to fit in one command packet. The
    ///   maximum length is 248 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Prepare Write
    /// Response](event::BlueNRGEvent::AttPrepareWriteResponse) and [Execute Write
    /// Response](event::BlueNRGEvent::AttExecuteWriteResponse) events. When the procedure is
    /// completed, a [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete)
    /// event is generated.
    fn write_long_characteristic_value<'a>(
        &mut self,
        params: &LongCharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to write a characteristic reliably.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the
    ///   [value](LongCharacteristicValue::value) is too long to fit in one command packet. The
    ///   maximum length is 248 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Prepare Write
    /// Response](event::BlueNRGEvent::AttPrepareWriteResponse) and [Execute Write
    /// Response](event::BlueNRGEvent::AttExecuteWriteResponse) events. When the procedure is
    /// completed, a [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete)
    /// event is generated.
    fn write_characteristic_value_reliably<'a>(
        &mut self,
        params: &LongCharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to write a long characteristic descriptor.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the
    ///   [value](LongCharacteristicValue::value) is too long to fit in one command packet. The
    ///   maximum length is 248 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Prepare Write
    /// Response](event::BlueNRGEvent::AttPrepareWriteResponse) and [Execute Write
    /// Response](event::BlueNRGEvent::AttExecuteWriteResponse) events. When the procedure is
    /// completed, a [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete)
    /// event is generated.
    fn write_long_characteristic_descriptor<'a>(
        &mut self,
        params: &LongCharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to read a long characteristic descriptor.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read Blob
    /// Response](event::BlueNRGEvent::AttReadBlobResponse) event. The end of the procedure
    /// is indicated by a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event.
    fn read_long_characteristic_descriptor(
        &mut self,
        params: &LongCharacteristicReadParameters,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to write a characteristic descriptor value.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [value](CharacteristicValue::value)
    ///   is too long to fit in one command packet. The maximum length is 250 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. When the procedure is completed, a [GATT Procedure
    /// Complete](event::BlueNRGEvent::GattProcedureComplete) event is generated.
    fn write_characteristic_descriptor<'a>(
        &mut self,
        params: &CharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to read a characteristic descriptor.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command status](hci::event::Event::CommandStatus) event is generated on the receipt of
    /// the command. The responses of the procedure are given through the [Read
    /// Response](event::BlueNRGEvent::AttReadResponse) event. The end of the procedure is
    /// indicated by a [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete)
    /// event.
    fn read_characteristic_descriptor(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Start the procedure to write a characteristic value without waiting for any response from
    /// the server.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [value](CharacteristicValue::value)
    ///   is too long to fit in one command packet. The maximum length is 250 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattWriteWithoutResponse)
    /// event is generated when this command is processed.
    fn write_without_response<'a>(
        &mut self,
        params: &CharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Start the procedure to write a characteristic value with an authentication signature without
    /// waiting for any response from the server. It cannot be used when the link is encrypted.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [value](CharacteristicValue::value)
    ///   is too long to fit in one command packet. The maximum length is 250 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command
    /// complete](event::command::ReturnParameters::GattSignedWriteWithoutResponse) event is
    /// generated when this command is processed.
    fn signed_write_without_response<'a>(
        &mut self,
        params: &CharacteristicValue<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Allow application to confirm indication. This command has to be sent when the application
    /// receives the [GATT Indication](event::BlueNRGEvent::GattIndication) event.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattConfirmIndication) event
    /// is generated when this command is processed.
    fn confirm_indication(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error>;

    /// Allows or rejects a write request from a client.
    ///
    /// This command has to be sent by the application when it receives the [GATT Write Permit
    /// Request](event::BlueNRGEvent::AttWritePermitRequest) event.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the [attribute
    ///   value](WriteResponseParameters::value) is so long that the serialized command cannot fit
    ///   in one packet. The maximum length is 250 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattWriteResponse) event is
    /// generated when this command is processed.
    fn write_response<'a>(
        &mut self,
        params: &WriteResponseParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Allows the GATT server to send a response to a read request from a client.
    ///
    /// The application has to send this command when it receives the [Read Permit
    /// Request](event::BlueNRGEvent::AttReadPermitRequest) or [Read Multiple Permit
    /// Request](event::BlueNRGEvent::AttReadMultiplePermitRequest). This command indicates
    /// to the stack that the response can be sent to the client. So if the application wishes to
    /// update any of the attributes before they are read by the client, it has to update the
    /// characteristic values using the
    /// [`update_characteristic_value`](Commands::update_characteristic_value) and then give this
    /// command. The application should perform the required operations within 30 seconds.
    /// Otherwise the GATT procedure will timeout.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattAllowRead) event is
    /// generated when this command is processed.
    fn allow_read(&mut self, conn_handle: hci::ConnectionHandle) -> nb::Result<(), Self::Error>;

    /// This command sets the security permission for the attribute handle specified. Currently the
    /// setting of security permission is allowed only for client configuration descriptor.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattSetSecurityPermission)
    /// event is generated when this command is processed.
    fn set_security_permission(
        &mut self,
        params: &SecurityPermissionParameters,
    ) -> nb::Result<(), Self::Error>;

    /// This command sets the value of a descriptor.
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the length of the descriptor value is
    ///   so long that the serialized command would not fit in one packet. The maximum length is 246
    ///   bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattSetDescriptorValue) event
    /// is generated when this command is processed.
    fn set_descriptor_value<'a>(
        &mut self,
        params: &DescriptorValueParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;

    /// Reads the value of the attribute handle specified from the local GATT database.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattReadHandleValue) event is
    /// generated when this command is processed.
    fn read_handle_value(&mut self, handle: CharacteristicHandle) -> nb::Result<(), Self::Error>;

    /// The command returns the value of the attribute handle from the specified offset.
    ///
    /// If the length to be returned is greater than 128, then only 128 bytes are
    /// [returned](event::command::ReturnParameters::GattReadHandleValueOffset). The
    /// application should send this command with incremented offsets until it gets an error with
    /// the offset it specified or the number of byes of attribute value returned is less than 128.
    ///
    /// # Errors
    ///
    /// Only underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// A [command complete](event::command::ReturnParameters::GattReadHandleValueOffset)
    /// event is generated when this command is processed.
    #[cfg(feature = "ms")]
    fn read_handle_value_offset(
        &mut self,
        handle: CharacteristicHandle,
        offset: usize,
    ) -> nb::Result<(), Self::Error>;

    /// Update the Attribute Value of a Characteristic belonging to a specified service.
    ///
    /// This command is an extension of the
    /// [`update_characteristic_value`](Commands::update_characteristic_value) command and supports
    /// updating of long attribute values (up to 512 bytes).
    ///
    /// # Errors
    ///
    /// - [ValueBufferTooLong](Error::ValueBufferTooLong) if the characteristic value is so long
    ///   that the command would not fit in one packet. The maximum length is 245 bytes.
    /// - Underlying communication errors are reported.
    ///
    /// # Generated events
    ///
    /// When the command has completed, the controller will generate a [command
    /// complete](event::command::ReturnParameters::GattUpdateLongCharacteristicValue) event.
    #[cfg(feature = "ms")]
    fn update_long_characteristic_value<'a>(
        &mut self,
        params: &UpdateLongCharacteristicValueParameters<'a>,
    ) -> nb::Result<(), Error<Self::Error>>;
}

impl<'buf, N: ArrayLength<u8>> Commands for RadioCoprocessor<'buf, N> {
    type Error = ();

    fn init(&mut self) -> nb::Result<(), Self::Error> {
        self.write_command(opcode::GATT_INIT, &[])
    }

    impl_variable_length_params!(add_service, AddServiceParameters, opcode::GATT_ADD_SERVICE);

    impl_variable_length_params!(
        include_service,
        IncludeServiceParameters,
        opcode::GATT_INCLUDE_SERVICE
    );

    impl_variable_length_params!(
        add_characteristic,
        AddCharacteristicParameters,
        opcode::GATT_ADD_CHARACTERISTIC
    );

    impl_validate_variable_length_params!(
        add_characteristic_descriptor<'a>,
        AddDescriptorParameters<'a>,
        opcode::GATT_ADD_CHARACTERISTIC_DESCRIPTOR
    );

    impl_validate_variable_length_params!(
        update_characteristic_value<'a>,
        UpdateCharacteristicValueParameters<'a>,
        opcode::GATT_UPDATE_CHARACTERISTIC_VALUE
    );

    fn delete_characteristic(
        &mut self,
        service: ServiceHandle,
        characteristic: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 4];
        LittleEndian::write_u16(&mut bytes[0..2], service.0);
        LittleEndian::write_u16(&mut bytes[2..4], characteristic.0);

        self.write_command(opcode::GATT_DELETE_CHARACTERISTIC, &bytes)
    }

    fn delete_service(&mut self, service: ServiceHandle) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes[0..2], service.0);

        self.write_command(opcode::GATT_DELETE_SERVICE, &bytes)
    }

    impl_params!(
        delete_included_service,
        DeleteIncludedServiceParameters,
        opcode::GATT_DELETE_INCLUDED_SERVICE
    );

    impl_value_params!(set_event_mask, Event, opcode::GATT_SET_EVENT_MASK);

    fn exchange_configuration(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);

        self.write_command(opcode::GATT_EXCHANGE_CONFIGURATION, &bytes)
    }

    fn find_information_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 6];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], attribute_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], attribute_range.to.0);

        self.write_command(opcode::GATT_FIND_INFORMATION_REQUEST, &bytes)
    }

    impl_validate_variable_length_params!(
        find_by_type_value_request<'a>,
        FindByTypeValueParameters<'a>,
        opcode::GATT_FIND_BY_TYPE_VALUE_REQUEST
    );

    impl_variable_length_params!(
        read_by_type_request,
        ReadByTypeParameters,
        opcode::GATT_READ_BY_TYPE_REQUEST
    );

    impl_variable_length_params!(
        read_by_group_type_request,
        ReadByTypeParameters,
        opcode::GATT_READ_BY_GROUP_TYPE_REQUEST
    );

    impl_validate_variable_length_params!(
        prepare_write_request<'a>,
        WriteRequest<'a>,
        opcode::GATT_PREPARE_WRITE_REQUEST
    );

    fn execute_write_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 3];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);
        bytes[2] = true as u8;

        self.write_command(opcode::GATT_EXECUTE_WRITE_REQUEST, &bytes)
    }

    fn cancel_write_request(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 3];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);
        bytes[2] = false as u8;

        self.write_command(opcode::GATT_EXECUTE_WRITE_REQUEST, &bytes)
    }

    fn discover_all_primary_services(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);

        self.write_command(opcode::GATT_DISCOVER_ALL_PRIMARY_SERVICES, &bytes)
    }

    fn discover_primary_services_by_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 19];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);
        let end = 2 + uuid.copy_into_slice(&mut bytes[2..]);

        self.write_command(
            opcode::GATT_DISCOVER_PRIMARY_SERVICES_BY_UUID,
            &bytes[..end],
        )
    }

    fn find_included_services(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        service_handle_range: Range<ServiceHandle>,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 6];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], service_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], service_handle_range.to.0);

        self.write_command(opcode::GATT_FIND_INCLUDED_SERVICES, &bytes)
    }

    fn discover_all_characteristics_of_service(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_handle_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 6];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], attribute_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], attribute_handle_range.to.0);

        self.write_command(opcode::GATT_DISCOVER_ALL_CHARACTERISTICS_OF_SERVICE, &bytes)
    }

    fn discover_characteristics_by_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        attribute_handle_range: Range<CharacteristicHandle>,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 23];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], attribute_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], attribute_handle_range.to.0);
        let uuid_len = uuid.copy_into_slice(&mut bytes[6..]);

        self.write_command(
            opcode::GATT_DISCOVER_CHARACTERISTICS_BY_UUID,
            &bytes[..6 + uuid_len],
        )
    }

    fn discover_all_characteristic_descriptors(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle_range: Range<CharacteristicHandle>,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 6];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], characteristic_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], characteristic_handle_range.to.0);

        self.write_command(opcode::GATT_DISCOVER_ALL_CHARACTERISTIC_DESCRIPTORS, &bytes)
    }

    fn read_characteristic_value(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 4];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], characteristic_handle.0);

        self.write_command(opcode::GATT_READ_CHARACTERISTIC_VALUE, &bytes)
    }

    fn read_characteristic_using_uuid(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle_range: Range<CharacteristicHandle>,
        uuid: Uuid,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 23];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], characteristic_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], characteristic_handle_range.to.0);
        let uuid_len = uuid.copy_into_slice(&mut bytes[6..]);

        self.write_command(
            opcode::GATT_READ_CHARACTERISTIC_BY_UUID,
            &bytes[..6 + uuid_len],
        )
    }

    impl_params!(
        read_long_characteristic_value,
        LongCharacteristicReadParameters,
        opcode::GATT_READ_LONG_CHARACTERISTIC_VALUE
    );

    impl_validate_variable_length_params!(
        read_multiple_characteristic_values<'a>,
        MultipleCharacteristicReadParameters<'a>,
        opcode::GATT_READ_MULTIPLE_CHARACTERISTIC_VALUES
    );

    impl_validate_variable_length_params!(
        write_characteristic_value<'a>,
        CharacteristicValue<'a>,
        opcode::GATT_WRITE_CHARACTERISTIC_VALUE
    );

    impl_validate_variable_length_params!(
        write_long_characteristic_value<'a>,
        LongCharacteristicValue<'a>,
        opcode::GATT_WRITE_LONG_CHARACTERISTIC_VALUE
    );

    impl_validate_variable_length_params!(
        write_characteristic_value_reliably<'a>,
        LongCharacteristicValue<'a>,
        opcode::GATT_WRITE_CHARACTERISTIC_VALUE_RELIABLY
    );

    impl_validate_variable_length_params!(
        write_long_characteristic_descriptor<'a>,
        LongCharacteristicValue<'a>,
        opcode::GATT_WRITE_LONG_CHARACTERISTIC_DESCRIPTOR
    );

    impl_params!(
        read_long_characteristic_descriptor,
        LongCharacteristicReadParameters,
        opcode::GATT_READ_LONG_CHARACTERISTIC_DESCRIPTOR
    );

    impl_validate_variable_length_params!(
        write_characteristic_descriptor<'a>,
        CharacteristicValue<'a>,
        opcode::GATT_WRITE_CHARACTERISTIC_DESCRIPTOR
    );

    fn read_characteristic_descriptor(
        &mut self,
        conn_handle: hci::ConnectionHandle,
        characteristic_handle: CharacteristicHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 4];
        LittleEndian::write_u16(&mut bytes[0..2], conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], characteristic_handle.0);

        self.write_command(opcode::GATT_READ_CHARACTERISTIC_DESCRIPTOR, &bytes)
    }

    impl_validate_variable_length_params!(
        write_without_response<'a>,
        CharacteristicValue<'a>,
        opcode::GATT_WRITE_WITHOUT_RESPONSE
    );

    impl_validate_variable_length_params!(
        signed_write_without_response<'a>,
        CharacteristicValue<'a>,
        opcode::GATT_SIGNED_WRITE_WITHOUT_RESPONSE
    );

    fn confirm_indication(
        &mut self,
        conn_handle: hci::ConnectionHandle,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);

        self.write_command(opcode::GATT_CONFIRM_INDICATION, &bytes)
    }

    impl_validate_variable_length_params!(
        write_response<'a>,
        WriteResponseParameters<'a>,
        opcode::GATT_WRITE_RESPONSE
    );

    fn allow_read(&mut self, conn_handle: hci::ConnectionHandle) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes, conn_handle.0);

        self.write_command(opcode::GATT_ALLOW_READ, &bytes)
    }

    impl_params!(
        set_security_permission,
        SecurityPermissionParameters,
        opcode::GATT_SET_SECURITY_PERMISSION
    );

    impl_validate_variable_length_params!(
        set_descriptor_value<'a>,
        DescriptorValueParameters<'a>,
        opcode::GATT_SET_DESCRIPTOR_VALUE
    );

    fn read_handle_value(&mut self, handle: CharacteristicHandle) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 2];
        LittleEndian::write_u16(&mut bytes, handle.0);

        self.write_command(opcode::GATT_READ_HANDLE_VALUE, &bytes)
    }

    #[cfg(feature = "ms")]
    fn read_handle_value_offset(
        &mut self,
        handle: CharacteristicHandle,
        offset: usize,
    ) -> nb::Result<(), Self::Error> {
        let mut bytes = [0; 3];
        LittleEndian::write_u16(&mut bytes, handle.0);
        bytes[2] = offset as u8;

        self.write_command(opcode::GATT_READ_HANDLE_VALUE_OFFSET, &bytes)
    }

    #[cfg(feature = "ms")]
    impl_validate_variable_length_params!(
        update_long_characteristic_value<'a>,
        UpdateLongCharacteristicValueParameters<'a>,
        opcode::GATT_UPDATE_LONG_CHARACTERISTIC_VALUE
    );
}

/// Potential errors from parameter validation.
///
/// Before some commands are sent to the controller, the parameters are validated. This type
/// enumerates the potential validation errors. Must be specialized on the types of communication
/// errors.
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Error<E> {
    /// For the [Add Characteristic Descriptor](Commands::add_characteristic_descriptor) command:
    /// the [descriptor value](AddDescriptorParameters::descriptor_value) is longer than the
    /// [maximum descriptor value length](AddDescriptorParameters::descriptor_value_max_len).
    DescriptorTooLong,

    /// For the [Add Characteristic Descriptor](Commands::add_characteristic_descriptor) command:
    /// the [descriptor value maximum length](AddDescriptorParameters::descriptor_value_max_len) is
    /// so large that the serialized structure may be more than 255 bytes. The maximum size is 227.
    DescriptorBufferTooLong,

    /// For the [Update Characteristir Value](Commands::update_characteristic_value) command: the
    /// length of the [characteristic value](UpdateCharacteristicValueParameters::value) is so large
    /// that the serialized structure would be more than 255 bytes. The maximum size is 249.
    ValueBufferTooLong,

    /// For the [Read Multiple Characteristic Values](Commands::read_multiple_characteristic_values)
    /// command: the number of [handles](MultipleCharacteristicReadParameters::handles) would cause
    /// the serialized command to be more than 255 bytes. The maximum length is 126 handles.
    TooManyHandlesToRead,

    /// Underlying communication error.
    Comm(E),
}

fn rewrap_error<E>(e: nb::Error<E>) -> nb::Error<Error<E>> {
    match e {
        nb::Error::WouldBlock => nb::Error::WouldBlock,
        nb::Error::Other(c) => nb::Error::Other(Error::Comm(c)),
    }
}

/// Parameters for the [GATT Add Service](Commands::add_service) command.
pub struct AddServiceParameters {
    /// UUID of the service
    pub uuid: Uuid,

    /// Type of service
    pub service_type: ServiceType,

    /// The maximum number of attribute records that can be added to this service (including the
    /// service attribute, include attribute, characteristic attribute, characteristic value
    /// attribute and characteristic descriptor attribute).
    pub max_attribute_records: u8,
}

impl AddServiceParameters {
    const MAX_LENGTH: usize = 19;

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= Self::MAX_LENGTH);

        let next = self.uuid.copy_into_slice(bytes);
        bytes[next] = self.service_type as u8;
        bytes[next + 1] = self.max_attribute_records as u8;

        next + 2
    }
}

/// Types of UUID
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Uuid {
    /// 16-bit UUID
    Uuid16(u16),

    /// 128-bit UUID
    Uuid128([u8; 16]),
}

impl Uuid {
    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        match *self {
            Uuid::Uuid16(uuid) => {
                assert!(bytes.len() >= 3);

                bytes[0] = 0x01;
                LittleEndian::write_u16(&mut bytes[1..3], uuid);

                3
            }
            Uuid::Uuid128(uuid) => {
                assert!(bytes.len() >= 17);

                bytes[0] = 0x02;
                bytes[1..17].copy_from_slice(&uuid);

                17
            }
        }
    }
}

/// Types of GATT services
#[derive(Copy, Clone, Debug, PartialEq)]
#[repr(u8)]
pub enum ServiceType {
    /// Primary service
    Primary = 0x01,
    /// Secondary service
    Secondary = 0x02,
}

/// Parameters for the [GATT Include Service](Commands::include_service) command.
pub struct IncludeServiceParameters {
    /// Handle of the service to which another service has to be included
    pub service_handle: ServiceHandle,

    /// Range of handles of the service which has to be included in the service.
    pub include_handle_range: Range<ServiceHandle>,

    /// UUID of the included service
    pub include_uuid: Uuid,
}

impl IncludeServiceParameters {
    const MAX_LENGTH: usize = 23;

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= Self::MAX_LENGTH);

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.include_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.include_handle_range.to.0);
        let uuid_len = self.include_uuid.copy_into_slice(&mut bytes[6..]);

        6 + uuid_len
    }
}

/// Handle for GATT Services.
#[derive(Copy, Clone, Debug, PartialEq, PartialOrd)]
pub struct ServiceHandle(pub u16);

/// Two ordered points that represent a range. The points may be identical to represent a range with
/// only one value.
pub struct Range<T> {
    from: T,
    to: T,
}

impl<T: PartialOrd> Range<T> {
    /// Create and return a new [Range].
    ///
    /// # Errors
    ///
    /// - [Inverted](RangeError::Inverted) if the beginning value is greater than the ending value.
    pub fn new(from: T, to: T) -> Result<Self, RangeError> {
        if to < from {
            return Err(RangeError::Inverted);
        }

        Ok(Self { from, to })
    }
}

/// Potential errors that can occer when creating a [Range].
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum RangeError {
    /// The beginning of the range came after the end.
    Inverted,
}

/// Parameters for the [GATT Add Characteristic](Commands::add_characteristic) command.
pub struct AddCharacteristicParameters {
    /// Handle of the service to which the characteristic has to be added
    pub service_handle: ServiceHandle,

    /// UUID of the characteristic
    pub characteristic_uuid: Uuid,

    /// Maximum length of the characteristic value
    pub characteristic_value_len: usize,

    /// Properties of the characteristic (defined in Volume 3, Part G, Section 3.3.3.1 of Bluetooth
    /// Specification 4.1)
    pub characteristic_properties: CharacteristicProperty,

    /// Security requirements of the characteristic
    pub security_permissions: CharacteristicPermission,

    /// Which types of events will be generated when the attribute is accessed.
    pub gatt_event_mask: CharacteristicEvent,

    /// The minimum encryption key size requirement for this attribute.
    pub encryption_key_size: EncryptionKeySize,

    /// If true, the attribute has a variable length value field. Otherwise, the value field length
    /// is fixed.
    pub is_variable: bool,

    /// If true, the
    /// [`characteristic_value_len`](AddCharacteristicParameters::characteristic_value_len)
    /// parameter only takes 1 byte.
    pub fw_version_before_v72: bool,
}

impl AddCharacteristicParameters {
    const MAX_LENGTH: usize = 26;

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= Self::MAX_LENGTH);

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        let uuid_len = self.characteristic_uuid.copy_into_slice(&mut bytes[2..19]);
        let mut next = 2 + uuid_len;
        if self.fw_version_before_v72 {
            bytes[next] = self.characteristic_value_len as u8;
            next += 1;
        } else {
            LittleEndian::write_u16(
                &mut bytes[next..next + 2],
                self.characteristic_value_len as u16,
            );
            next += 2;
        }
        let next = next;
        bytes[next] = self.characteristic_properties.bits();
        bytes[next + 1] = self.security_permissions.bits();
        bytes[next + 2] = self.gatt_event_mask.bits();
        bytes[next + 3] = self.encryption_key_size.0;
        bytes[next + 4] = self.is_variable as u8;

        next + 5
    }
}

bitflags! {
    /// Available [properties](AddCharacteristicParameters::characteristic_properties) for
    /// characteristics. Defined in Volume 3, Part G, Section 3.3.3.1 of Bluetooth Specification
    /// 4.1.
    pub struct CharacteristicProperty: u8 {
        /// If set, permits broadcasts of the Characteristic Value using Server Characteristic
        /// Configuration Descriptor. If set, the Server Characteristic Configuration Descriptor
        /// shall exist.
        const BROADCAST = 0x01;

        /// If set, permits reads of the Characteristic Value using procedures defined in Volume 3,
        /// Part G, Section 4.8 of the Bluetooth specification 4.1.
        const READ = 0x02;

        /// If set, permit writes of the Characteristic Value without response using procedures
        /// defined in Volume 3, Part G, Section 4.9.1 of the Bluetooth specification 4.1.
        const WRITE_WITHOUT_RESPONSE = 0x04;

        /// If set, permits writes of the Characteristic Value with response using procedures
        /// defined in Volume 3, Part Section 4.9.3 or Section 4.9.4 of the Bluetooth
        /// specification 4.1.
        const WRITE = 0x08;

        /// If set, permits notifications of a Characteristic Value without acknowledgement using
        /// the procedure defined in Volume 3, Part G, Section 4.10 of the Bluetooth specification
        /// 4.1. If set, the Client Characteristic Configuration Descriptor shall exist.
        const NOTIFY = 0x10;

        /// If set, permits indications of a Characteristic Value with acknowledgement using the
        /// procedure defined in Volume 3, Part G, Section 4.11 of the Bluetooth specification
        /// 4.1. If set, the Client Characteristic Configuration Descriptor shall exist.
        const INDICATE = 0x20;

        /// If set, permits signed writes to the Characteristic Value using the Signed Writes
        /// procedure defined in Volume 3, Part G, Section 4.9.2 of the Bluetooth specification
        /// 4.1.
        const AUTHENTICATED = 0x40;

        /// If set, additional characteristic properties are defined in the Characteristic Extended
        /// Properties Descriptor defined in Volume 3, Part G, Section 3.3.3.1 of the Bluetooth
        /// specification 4.1. If set, the Characteristic Extended Properties Descriptor shall
        /// exist.
        const EXTENDED_PROPERTIES = 0x80;
    }
}

bitflags! {
    /// [Permissions](AddCharacteristicParameter::security_permissions) available for
    /// characteristics.
    pub struct CharacteristicPermission: u8 {
        /// Need authentication to read.
        const AUTHENTICATED_READ = 0x01;

        /// Need authorization to read.
        const AUTHORIZED_READ = 0x02;

        /// Link should be encrypted to read.
        const ENCRYPTED_READ = 0x04;

        /// Need authentication to write.
        const AUTHENTICATED_WRITE = 0x08;

        /// Need authorization to write.
        const AUTHORIZED_WRITE = 0x10;

        /// Link should be encrypted for write.
        const ENCRYPTED_WRITE = 0x20;
    }
}

bitflags! {
    /// Which events may be generated when a characteristic is accessed.
    pub struct CharacteristicEvent: u8 {
        /// The application will be notified when a client writes to this attribute.
        const ATTRIBUTE_WRITE = 0x01;

        /// The application will be notified when a write request/write command/signed write command
        /// is received by the server for this attribute.
        const CONFIRM_WRITE = 0x02;

        /// The application will be notified when a read request of any type is got for this
        /// attribute.
        const CONFIRM_READ = 0x04;
    }
}

/// Encryption key size, in bytes.
#[derive(Copy, Clone, Debug, PartialEq)]
pub struct EncryptionKeySize(u8);

impl EncryptionKeySize {
    /// Validate the size as a valid encryption key size. Valid range is 7 to 16, inclusive.
    ///
    /// # Errors
    ///
    /// - [TooShort](EncryptionKeySizeError::TooShort) if the provided size is less than 7.
    /// - [TooLong](EncryptionKeySizeError::TooLong) if the provided size is greater than 16.
    pub fn with_value(sz: usize) -> Result<Self, EncryptionKeySizeError> {
        const MIN: usize = 7;
        const MAX: usize = 16;

        if sz < MIN {
            return Err(EncryptionKeySizeError::TooShort);
        }

        if sz > MAX {
            return Err(EncryptionKeySizeError::TooLong);
        }

        Ok(Self(sz as u8))
    }

    /// Retrieve the key size.
    pub fn value(&self) -> usize {
        self.0 as usize
    }
}

/// Errors that can occur when creating an [`EncryptionKeySize`].
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum EncryptionKeySizeError {
    /// The provided size was less than the minimum allowed size.
    TooShort,
    /// The provided size was greater than the maximum allowed size.
    TooLong,
}

/// Handle for GATT characteristics.
#[derive(Copy, Clone, Debug, PartialEq, PartialOrd)]
pub struct CharacteristicHandle(pub u16);

/// Parameters for the [GATT Add Characteristic Descriptor](Commands::add_characteristic_descriptor)
/// command.
pub struct AddDescriptorParameters<'a> {
    /// Handle of the service to which characteristic belongs.
    pub service_handle: ServiceHandle,

    /// Handle of the characteristic to which description is to be added.
    pub characteristic_handle: CharacteristicHandle,

    /// UUID of the characteristic descriptor.
    ///
    /// See [KnownDescriptor] for some useful descriptors. This value is not restricted to the known
    /// descriptors, however.
    pub descriptor_uuid: Uuid,

    /// The maximum length of the descriptor value.
    pub descriptor_value_max_len: usize,

    /// Current Length of the characteristic descriptor value.
    pub descriptor_value: &'a [u8],

    /// What security requirements must be met before the descriptor can be accessed.
    pub security_permissions: DescriptorPermission,

    /// What types of access are allowed for the descriptor.
    pub access_permissions: AccessPermission,

    /// Which types of events will be generated when the attribute is accessed.
    pub gatt_event_mask: CharacteristicEvent,

    /// The minimum encryption key size requirement for this attribute.
    pub encryption_key_size: EncryptionKeySize,

    /// If true, the attribute has a variable length value field. Otherwise, the value field length
    /// is fixed.
    pub is_variable: bool,
}

impl<'a> AddDescriptorParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.descriptor_value.len() > self.descriptor_value_max_len {
            return Err(Error::DescriptorTooLong);
        }

        if 28 + self.descriptor_value_max_len > Self::MAX_LENGTH {
            return Err(Error::DescriptorBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        // The buffer should be big enough to hold this descriptor, assuming a 128-bit UUID.
        assert!(bytes.len() >= 28 + self.descriptor_value.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        let uuid_len = self.descriptor_uuid.copy_into_slice(&mut bytes[4..]);
        bytes[4 + uuid_len] = self.descriptor_value_max_len as u8;
        bytes[5 + uuid_len] = self.descriptor_value.len() as u8;
        bytes[6 + uuid_len..6 + uuid_len + self.descriptor_value.len()]
            .copy_from_slice(self.descriptor_value);
        let next = 6 + uuid_len + self.descriptor_value.len();
        bytes[next] = self.security_permissions.bits();
        bytes[1 + next] = self.access_permissions.bits();
        bytes[2 + next] = self.gatt_event_mask.bits();
        bytes[3 + next] = self.encryption_key_size.0;
        bytes[4 + next] = self.is_variable as u8;

        5 + next
    }
}

/// Common characteristic descriptor UUIDs.
#[repr(u16)]
pub enum KnownDescriptor {
    /// Characteristic Extended Properties Descriptor
    CharacteristicExtendedProperties = 0x2900,
    /// Characteristic User Descriptor
    CharacteristicUser = 0x2901,
    /// Client configuration descriptor
    ClientConfiguration = 0x2902,
    /// Server configuration descriptor
    ServerConfiguration = 0x2903,
    /// Characteristic presentation format
    CharacteristicPresentationFormat = 0x2904,
    /// Characteristic aggregated format
    CharacteristicAggregatedFormat = 0x2905,
}

impl From<KnownDescriptor> for Uuid {
    fn from(value: KnownDescriptor) -> Self {
        Uuid::Uuid16(value as u16)
    }
}

bitflags! {
    /// Permissions available for characteristic descriptors.
    pub struct DescriptorPermission: u8 {
        /// Authentication required.
        const AUTHENTICATED = 0x01;

        /// Authorization required.
        const AUTHORIZED = 0x02;

        /// Encryption required.
        const ENCRYPTED = 0x04;
    }
}

bitflags! {
    /// Types of access for characteristic descriptors
    pub struct AccessPermission: u8 {
        /// Readable
        const READ = 0x01;

        /// Writable
        const WRITE = 0x02;

        /// Readable and writeable
        const READ_WRITE = Self::READ.bits | Self::WRITE.bits;
    }
}

/// Handle for GATT characteristic descriptors.
#[derive(Copy, Clone, Debug, PartialEq)]
pub struct DescriptorHandle(pub u16);

/// Parameters for the [Update Characteristic Value](Commands::update_characteristic_value)
/// command.
pub struct UpdateCharacteristicValueParameters<'a> {
    /// Handle of the service to which characteristic belongs.
    pub service_handle: ServiceHandle,

    /// Handle of the characteristic.
    pub characteristic_handle: CharacteristicHandle,

    /// The offset from which the attribute value has to be updated. If this is set to 0, and the
    /// attribute value is of [variable length](AddCharacteristicParameters::is_variable), then the
    /// length of the attribute will be set to the length of
    /// [value](UpdateCharacteristicValueParameters::value). If the offset is set to a value greater
    /// than 0, then the length of the attribute will be set to the [maximum
    /// length](AddCharacteristicParameters::characteristic_value_len) as specified for the
    /// attribute while adding the characteristic.
    pub offset: usize,

    /// The new characteristic value.
    pub value: &'a [u8],
}

impl<'a> UpdateCharacteristicValueParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        const MAX_VALUE_LEN: usize = 249;
        if self.value.len() > MAX_VALUE_LEN {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= 6 + self.value.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        bytes[4] = self.offset as u8;
        bytes[5] = self.value.len() as u8;
        bytes[6..6 + self.value.len()].copy_from_slice(self.value);

        6 + self.value.len()
    }
}

/// Parameters for the [GATT Delete Included Service](Commands::delete_included_service) command.
pub struct DeleteIncludedServiceParameters {
    /// Handle of the service to which Include definition belongs
    pub service: ServiceHandle,

    /// Handle of the Included definition to be deleted.
    pub included_service: ServiceHandle,
}

impl DeleteIncludedServiceParameters {
    const LENGTH: usize = 4;

    fn copy_into_slice(&self, bytes: &mut [u8]) {
        assert!(bytes.len() >= Self::LENGTH);

        LittleEndian::write_u16(&mut bytes[0..2], self.service.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.included_service.0);
    }
}

bitflags! {
    /// Flags for individual events that can be masked by the [GATT Set Event
    /// Mask](Commands::set_event_mask) command.
    pub struct Event: u32 {
        /// [GATT Attribute Modified](event::BlueNRGEvent::GattAttributeModified).
        const ATTRIBUTE_MODIFIED = 0x0000_0001;
        /// [GATT Procedure Timeout](event::BlueNRGEvent::GattProcedureTimeout).
        const PROCEDURE_TIMEOUT = 0x0000_0002;
        /// [ATT Exchange MTU Response](event::BlueNRGEvent::AttExchangeMtuResponse).
        const EXCHANGE_MTU_RESPONSE = 0x0000_0004;
        /// [ATT Find Information Response](event::BlueNRGEvent::AttFindInformationResponse).
        const FIND_INFORMATION_RESPONSE = 0x0000_0008;
        /// [ATT Find By Type Value
        /// Response](event::BlueNRGEvent::AttFindByTypeValueResponse).
        const FIND_BY_TYPE_VALUE_RESPONSE = 0x0000_0010;
        /// [ATT Find By Type Response](event::BlueNRGEvent::AttFindByTypeResponse).
        const READ_BY_TYPE_RESPONSE = 0x0000_0020;
        /// [ATT Read Response](event::BlueNRGEvent::AttReadResponse).
        const READ_RESPONSE = 0x0000_0040;
        /// [ATT Read Blob Response](event::BlueNRGEvent::AttReadBlobResponse).
        const READ_BLOB_RESPONSE = 0x0000_0080;
        /// [ATT Read Multiple Response](event::BlueNRGEvent::AttReadMultipleResponse).
        const READ_MULTIPLE_RESPONSE = 0x0000_0100;
        /// [ATT Read By Group](event::BlueNRGEvent::AttReadByGroupTypeResponse).
        const READ_BY_GROUP_RESPONSE = 0x0000_0200;
        /// [ATT Prepare Write Response](event::BlueNRGEvent::AttPrepareWriteResponse).
        const PREPARE_WRITE_RESPONSE = 0x0000_0800;
        /// [ATT Execute Write Response](event::BlueNRGEvent::AttExecuteWriteResponse).
        const EXECUTE_WRITE_RESPONSE = 0x0000_1000;
        /// [GATT Indication](event::BlueNRGEvent::GattIndication).
        const INDICATION = 0x0000_2000;
        /// [GATT Notification](event::BlueNRGEvent::GattNotification).
        const NOTIFICATION = 0x0000_4000;
        /// [GATT Error Response](event::BlueNRGEvent::AttErrorResponse).
        const ERROR_RESPONSE = 0x0000_8000;
        /// [GATT Procedure Complete](event::BlueNRGEvent::GattProcedureComplete).
        const PROCEDURE_COMPLETE = 0x0001_0000;
        /// [GATT Discover Characteristic by UUID or Read Using Characteristic
        /// UUID](event::BlueNRGEvent::GattDiscoverOrReadCharacteristicByUuidResponse).
        const DISCOVER_OR_READ_CHARACTERISTIC_BY_UUID_RESPONSE = 0x0002_0000;
        #[cfg(feature = "ms")]
        /// [GATT Tx Pool Available](event::BlueNRGEvent::GattTxPoolAvailable)
        const TX_POOL_AVAILABLE = 0x0004_0000;
    }
}

impl Event {
    const LENGTH: usize = 4;

    fn copy_into_slice(&self, bytes: &mut [u8]) {
        assert!(bytes.len() >= Self::LENGTH);

        LittleEndian::write_u32(bytes, self.bits());
    }
}

/// Parameters for the [GATT Find by Type Value Request](Commands::find_by_type_value_request)
/// command.
pub struct FindByTypeValueParameters<'a> {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Range of attributes to be discovered on the server.
    pub attribute_handle_range: Range<CharacteristicHandle>,

    /// UUID to find.
    pub uuid: Uuid16,

    /// Attribute value to find.
    ///
    /// Note: Though the max attribute value that is allowed according to the spec is 512 octets,
    /// due to the limitation of the transport layer (command packet max length is 255 bytes) the
    /// value is limited to 246 bytes.
    pub value: &'a [u8],
}

impl<'a> FindByTypeValueParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if 9 + self.value.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= 9 + self.value.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.attribute_handle_range.to.0);
        LittleEndian::write_u16(&mut bytes[6..8], self.uuid.0);
        bytes[8] = self.value.len() as u8;
        bytes[9..9 + self.value.len()].copy_from_slice(&self.value);

        9 + self.value.len()
    }
}

/// 16-bit UUID
pub struct Uuid16(pub u16);

/// Parameters for the [Read by Group Type Request](Commands::read_by_group_type_request) command.
pub struct ReadByTypeParameters {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Range of values to be read on the server.
    pub attribute_handle_range: Range<CharacteristicHandle>,

    /// UUID of the attribute.
    pub uuid: Uuid,
}

impl ReadByTypeParameters {
    const MAX_LENGTH: usize = 23;

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= Self::MAX_LENGTH);

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute_handle_range.from.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.attribute_handle_range.to.0);
        6 + self.uuid.copy_into_slice(&mut bytes[6..])
    }
}

/// Parameters for the [Prepare Write Request](Commands::prepare_write_request) command.
pub struct WriteRequest<'a> {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Handle of the attribute whose value has to be written
    pub attribute_handle: CharacteristicHandle,

    /// The offset at which value has to be written
    pub offset: usize,

    /// Value of the attribute to be written
    pub value: &'a [u8],
}

impl<'a> WriteRequest<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if 9 + self.value.len() > 255 {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= 9 + self.value.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute_handle.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.offset as u16);
        bytes[6] = self.value.len() as u8;
        bytes[7..7 + self.value.len()].copy_from_slice(&self.value);

        7 + self.value.len()
    }
}

/// Parameters for the [Read long characteristic value](Commands::read_long_characteristic_value)
/// command.
pub struct LongCharacteristicReadParameters {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Handle of the characteristic to be read
    pub attribute: CharacteristicHandle,

    /// Offset from which the value needs to be read.
    pub offset: usize,
}

impl LongCharacteristicReadParameters {
    const LENGTH: usize = 6;

    fn copy_into_slice(&self, bytes: &mut [u8]) {
        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.offset as u16);
    }
}

/// Parameters for the [Read Multiple Characteristic
/// Values](Commands::read_multiple_characteristic_values) command.
pub struct MultipleCharacteristicReadParameters<'a> {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// The handles for which the attribute value has to be read.
    ///
    /// The maximum length is 126 handles.
    pub handles: &'a [CharacteristicHandle],
}

impl<'a> MultipleCharacteristicReadParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        const MAX_HANDLE_COUNT: usize = 126;
        if self.handles.len() > MAX_HANDLE_COUNT {
            return Err(Error::TooManyHandlesToRead);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= 3 + 2 * self.handles.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        bytes[2] = self.handles.len() as u8;
        let mut next = 3;
        for handle in self.handles.iter() {
            LittleEndian::write_u16(&mut bytes[next..next + 2], handle.0);
            next += 2
        }

        next
    }
}

/// Parameters for the [Write Characteristic Value](Commands::write_characteristic_value) command.
pub struct CharacteristicValue<'a> {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Handle of the characteristic to be written.
    pub characteristic_handle: CharacteristicHandle,

    /// Value to be written. The maximum length is 250 bytes.
    pub value: &'a [u8],
}

impl<'a> CharacteristicValue<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= self.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        bytes[4] = self.value.len() as u8;
        bytes[5..self.len()].copy_from_slice(self.value);

        self.len()
    }

    fn len(&self) -> usize {
        5 + self.value.len()
    }
}

/// Parameters for the [Write Long Characteristic Value](Commands::write_long_characteristic_value)
/// command.
pub struct LongCharacteristicValue<'a> {
    /// Connection handle for which the command is given.
    pub conn_handle: hci::ConnectionHandle,

    /// Handle of the characteristic to be written.
    pub characteristic_handle: CharacteristicHandle,

    /// Offset at which the attribute has to be written.
    pub offset: usize,

    /// Value to be written. The maximum length is 248 bytes.
    pub value: &'a [u8],
}

impl<'a> LongCharacteristicValue<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= self.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.offset as u16);
        bytes[6] = self.value.len() as u8;
        bytes[7..self.len()].copy_from_slice(self.value);

        self.len()
    }

    fn len(&self) -> usize {
        7 + self.value.len()
    }
}

/// Parameters for the [Write Response](Commands::write_response) command.
pub struct WriteResponseParameters<'a> {
    /// Connection handle for which the command is given
    pub conn_handle: hci::ConnectionHandle,

    /// Handle of the attribute that was passed in the [Write Permit
    /// Request](event::BlueNRGEvent::AttWritePermitRequest) event.
    pub attribute_handle: CharacteristicHandle,

    /// Is the command rejected, and if so, why?
    pub status: Result<(), hci::Status<event::Status>>,

    /// Value as passed in the [Write Permit
    /// Request](event::BlueNRGEvent::AttWritePermitRequest) event.
    pub value: &'a [u8],
}

impl<'a> WriteResponseParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= self.len());
        LittleEndian::write_u16(&mut bytes[0..2], self.conn_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute_handle.0);
        match self.status {
            Ok(_) => {
                bytes[4] = 0;
                bytes[5] = 0;
            }
            Err(code) => {
                bytes[4] = 1;
                bytes[5] = code.into();
            }
        }
        bytes[6] = self.value.len() as u8;
        bytes[7..self.len()].copy_from_slice(self.value);

        self.len()
    }

    fn len(&self) -> usize {
        7 + self.value.len()
    }
}

/// Parameters for the [Set Security Permission](Commands::set_security_permission) command.
pub struct SecurityPermissionParameters {
    /// Handle of the service which contains the attribute whose security permission has to be
    /// modified.
    pub service_handle: ServiceHandle,

    /// Handle of the attribute whose security permission has to be modified.
    pub attribute_handle: CharacteristicHandle,

    /// Security requirements for the attribute.
    pub permission: CharacteristicPermission,
}

impl SecurityPermissionParameters {
    const LENGTH: usize = 5;

    fn copy_into_slice(&self, bytes: &mut [u8]) {
        assert!(bytes.len() >= Self::LENGTH);

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.attribute_handle.0);
        bytes[4] = self.permission.bits();
    }
}

/// Parameters for the [Set Descriptor Value](Commands::set_descriptor_value) command.
pub struct DescriptorValueParameters<'a> {
    /// Handle of the service which contains the descriptor.
    pub service_handle: ServiceHandle,

    /// Handle of the characteristic which contains the descriptor.
    pub characteristic_handle: CharacteristicHandle,

    /// Handle of the descriptor whose value has to be set.
    pub descriptor_handle: DescriptorHandle,

    /// Offset from which the descriptor value has to be updated.
    pub offset: usize,

    /// Descriptor value
    pub value: &'a [u8],
}

impl<'a> DescriptorValueParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= self.len());
        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        LittleEndian::write_u16(&mut bytes[4..6], self.descriptor_handle.0);
        LittleEndian::write_u16(&mut bytes[6..8], self.offset as u16);
        bytes[8] = self.value.len() as u8;
        bytes[9..self.len()].copy_from_slice(self.value);

        self.len()
    }

    fn len(&self) -> usize {
        9 + self.value.len()
    }
}

/// Parameters for the [Update Long Characteristic
/// Value](Commands::update_long_characteristic_value) command.
#[cfg(feature = "ms")]
pub struct UpdateLongCharacteristicValueParameters<'a> {
    /// Handle of the service to which characteristic belongs.
    pub service_handle: ServiceHandle,

    /// Handle of the characteristic.
    pub characteristic_handle: CharacteristicHandle,

    /// Controls whether an indication, notification, both, or neither is generated by the attribute
    /// update.
    pub update_type: UpdateType,

    /// Total length of the Attribute value after the update. In case of a [variable
    /// size](AddCharacteristicParameters::is_variable) characteristic, this field specifies the new
    /// length of the characteristic value after the update; in case of fixed length characteristic
    /// this field is ignored.
    pub total_len: usize,

    /// The offset from which the Attribute value has to be updated.
    pub offset: usize,

    /// Updated value of the characteristic.
    pub value: &'a [u8],
}

#[cfg(feature = "ms")]
impl<'a> UpdateLongCharacteristicValueParameters<'a> {
    const MAX_LENGTH: usize = 255;

    fn validate<E>(&self) -> Result<(), Error<E>> {
        if self.len() > Self::MAX_LENGTH {
            return Err(Error::ValueBufferTooLong);
        }

        Ok(())
    }

    fn copy_into_slice(&self, bytes: &mut [u8]) -> usize {
        assert!(bytes.len() >= self.len());

        LittleEndian::write_u16(&mut bytes[0..2], self.service_handle.0);
        LittleEndian::write_u16(&mut bytes[2..4], self.characteristic_handle.0);
        bytes[4] = self.update_type.bits();
        LittleEndian::write_u16(&mut bytes[5..7], self.total_len as u16);
        LittleEndian::write_u16(&mut bytes[7..9], self.offset as u16);
        bytes[9] = self.value.len() as u8;
        bytes[10..self.len()].copy_from_slice(self.value);

        self.len()
    }

    fn len(&self) -> usize {
        10 + self.value.len()
    }
}

#[cfg(feature = "ms")]
bitflags! {
    /// Flags for types of updates that the controller should signal when a characteristic value is
    /// [updated](Commands::update_long_characteristic_value).
    pub struct UpdateType: u8 {
        /// A notification can be sent if enabled in the client characteristic configuration
        /// descriptor.
        const NOTIFICATION = 0x01;
        /// An indication can be sent if enabled in the client characteristic configuration
        /// descriptor.
        const INDICATION = 0x02;
    }
}
