#![no_std]

//!
//! This module contains the definitions for stivale2 boot protocol.
//!
//! This Documentation contains miminal information about the protocol and provides brief
//! documentation for helper functions and methods.
//!
//! For detailed documentation, Visit the official [docs](https://github.com/stivale/stivale/blob/master/STIVALE2.md)

#[repr(C, packed)]
pub struct Stivale2Tag {
    pub identifier: u64,
    pub next: u64,
}

/// The kernel executable shall have a section .stivale2hdr which will contain the header that the bootloader will parse.
/// The following header should be initalized as static and should be linked as section
/// `.stivale2hdr`
#[repr(C, packed)]
pub struct Stivale2Header {
    pub entry_point: u64,

    /// The stack pointer should be casted to `*const ()`.
    ///
    /// # Example
    /// ```
    /// static STACK: [u8;1024] = [0;1024];
    /// let ptr = &STACK as *const u8 as *const ();
    ///
    /// ```
    pub stack: *const (),
    pub flags: u64,
    pub tags: *const (),
}

unsafe impl Sync for Stivale2Header {}
unsafe impl Send for Stivale2Header {}

impl Stivale2Header {
    /// Create a stivale2 header structure which should be linked to the ELF Section
    /// ".stivale2hdr".
    ///
    /// # Example
    ///
    /// ```
    ///static STACK : [u8;4096] = [0;4096];
    ///
    ///#[used]
    ///#[link_section = ".stivale2hdr"]
    ///static hdr : Stivale2Header = Stivale2Header::new(0,&stack, 1 << 0 , core::mem::null());
    ///
    /// ```
    pub const fn new<const SIZE: usize>(
        entry_point: u64,
        stack: &[u8; SIZE],
        flags: u64,
        tags: *const (),
    ) -> Self {
        Stivale2Header {
            entry_point,
            flags,
            tags,
            stack: stack as *const u8 as *const (),
        }
    }
}

pub const STIVALE2_HEADER_TAG_FRAMEBUFFER_ID: u64 = 0x3ecc1bc43d0f7971;

#[repr(C, packed)]
pub struct Stivale2HeaderTagFramebuffer {
    pub identifier: u64,
    pub next: *const (),
    pub framebuffer_width: u16,
    pub framebuffer_height: u16,
    pub framebuffer_bpp: u16,
}

unsafe impl Sync for Stivale2HeaderTagFramebuffer {}
unsafe impl Send for Stivale2HeaderTagFramebuffer {}

pub const STIVALE2_HEADER_TAG_FB_MTRR_ID: u64 = 0x4c7bb07731282e00;

pub const STIVALE2_HEADER_TAG_TERMINAL_ID: u64 = 0xa85d499b1823be72;

#[repr(C, packed)]
pub struct Stivale2HeaderTagTerminal {
    pub identifier: u64,
    pub next: *const (),
    pub flags: u64,
}

unsafe impl Sync for Stivale2HeaderTagTerminal {}
unsafe impl Send for Stivale2HeaderTagTerminal {}

pub const STIVALE2_HEADER_TAG_SMP_ID: u64 = 0x1ab015085f3273df;

#[repr(C, packed)]
pub struct Stivale2HeaderTagSmp {
    pub identifier: u64,
    pub next: *const (),
    pub flags: u64,
}

unsafe impl Sync for Stivale2HeaderTagSmp {}
unsafe impl Send for Stivale2HeaderTagSmp {}

pub const STIVALE2_HEADER_TAG_5LV_PAGING_ID: u64 = 0x932f477032007e8f;

pub const STIVALE2_HEADER_TAG_UNMAP_NULL_ID: u64 = 0x92919432b16fe7e7;

pub const STIVALE2_BOOTLOADER_BRAND_SIZE: usize = 64;
pub const STIVALE2_BOOTLOADER_VERSION_SIZE: usize = 64;

/// The info passed on by the bootloader.
///
/// `tags` should be iterated to get the desired tags.
#[repr(C, packed)]
pub struct Stivale2Struct {
    pub bootloader_brand: [u8; STIVALE2_BOOTLOADER_BRAND_SIZE],
    pub bootloader_version: [u8; STIVALE2_BOOTLOADER_VERSION_SIZE],
    pub tags: u64,
}

impl Stivale2Struct {
    /// Get a tag from the info passed on by the bootloader.
    ///
    /// Returned pointer should be checked for validity and cast into desired structure if valid.
    ///
    /// # Examples
    ///
    /// ```
    ///let tag = _info.get_tag(0xc2b3f4c3233b0974);
    ///
    ///if tag.is_null() {
    ///    // handle case
    ///} else {
    ///    let term = tag as *const Stivale2StructTagTerminal;
    ///}
    /// ```
    pub fn get_tag(&self, id: u64) -> *const () {
        let mut current = id as *const ();

        loop {
            if current.is_null() {
                return core::ptr::null();
            }

            let _c = current as *const Stivale2Tag;
            unsafe {
                if (*_c).identifier == id {
                    return current;
                }

                current = (*_c).next as *const ();
            }
        }
    }
}

pub const STIVALE2_STRUCT_TAG_CMDLINE_ID: u64 = 0xe5e76a1b4597a781;

#[repr(C, packed)]
pub struct Stivale2StructTagCmdline {
    pub identifier: u64,
    pub next: *const (),
    pub cmdline: u64,
}

unsafe impl Sync for Stivale2StructTagCmdline {}
unsafe impl Send for Stivale2StructTagCmdline {}

pub const STIVALE2_STRUCT_TAG_MEMMAP_ID: u64 = 0x2187f79e8612de07;

pub const STIVALE2_MMAP_USABLE: u64 = 1;
pub const STIVALE2_MMAP_RESERVED: u64 = 2;
pub const STIVALE2_MMAP_ACPI_RECLAIMABLE: u64 = 3;
pub const STIVALE2_MMAP_ACPI_NVS: u64 = 4;
pub const STIVALE2_MMAP_BAD_MEMORY: u64 = 5;
pub const STIVALE2_MMAP_BOOTLOADER_RECLAIMABLE: u64 = 0x100;
pub const STIVALE2_MMAP_KERNEL_AND_MODULES: u64 = 0x100;
pub const STIVALE2_MMAP_FRAMEBUFFER: u64 = 0x100;

pub const STIVALE2_STRUCT_TAG_TERMINAL_ID: u64 = 0xc2b3f4c3233b0974;

#[repr(C, packed)]
pub struct Stivale2StructTagTerminal {
    pub identifier: u64,
    pub next: *const (),
    pub flags: u32,
    pub cols: u16,
    pub rows: u16,
    pub term_write: u64,
}

unsafe impl Sync for Stivale2StructTagTerminal {}
unsafe impl Send for Stivale2StructTagTerminal {}

impl Stivale2StructTagTerminal {
    /// Get a `terminal_write` function from the function pointer provided by the bootloader.
    ///
    /// Returns a closure that can be used to print to the terminal by passing a `&str`.
    ///
    ///# Example
    ///
    /// ```
    /// let print = terminal.get_term_write();
    /// print("Hello world");
    ///
    /// ```
    pub fn get_term_write(&mut self) -> impl Fn(&str) {
        type Stivale2TermWrite = extern "C" fn(*const i8, u64);

        let _func = unsafe {
            core::mem::transmute::<*const (), Stivale2TermWrite>(self.term_write as *const ())
        };

        move |msg| {
            _func(msg.as_ptr() as *const i8, msg.len() as u64);
        }
    }
}
