use std::{ffi::OsStr, fmt::Display, io, process, sync::Arc, time::Duration};

use async_trait::async_trait;
use tokio::{process::Command, signal, time};

use crate::{Done, LocationInterface, Result};

pub(crate) enum Stdio {
    Inherit,
    Ignore,
    Collect,
}

pub(crate) enum ExitReason {
    CtrlC,
    ProcessFinished(io::Result<process::Output>),
}

pub(crate) enum CtrlCResult {
    ProcessExited,
    Timeout,
}

#[async_trait]
pub trait CommandInterface<'a, S, E, L, K, V>
where
    S: Display + Send + Sync,
    E: IntoIterator<Item = (K, V)> + Clone + Send + Sync,
    L: LocationInterface + Send + Sync,
    K: AsRef<OsStr>,
    V: AsRef<OsStr>,
{
    fn exe(&self) -> &S;
    fn env(&self) -> &E;
    fn pwd(&self) -> &L;
    fn msg(&self) -> &S;

    #[cfg(unix)]
    const SHELL: &'static str = "/bin/sh";

    #[cfg(windows)]
    const SHELL: &'static str = "cmd";

    #[cfg(unix)]
    fn shelled(cmd: &S) -> Vec<String> {
        vec!["-c".to_string(), cmd.to_string()]
    }

    #[cfg(windows)]
    fn shelled(cmd: &S) -> Vec<String> {
        vec!["/c".to_string(), cmd.to_string()]
    }

    fn timeout() -> Duration {
        Duration::from_secs(10)
    }

    async fn run(&self) -> Result {
        self.spawn(Stdio::Inherit).await
    }

    async fn output(&self) -> Result {
        self.spawn(Stdio::Collect).await
    }

    async fn silent(&self) -> Result {
        self.spawn(Stdio::Ignore).await
    }

    async fn spawn(&self, stdio: Stdio) -> Result {
        let cmd = self;

        eprintln!("{}", crate::headline!(cmd));

        let (stdout, stderr) = match stdio {
            Stdio::Inherit => (process::Stdio::inherit(), process::Stdio::inherit()),
            Stdio::Ignore => (process::Stdio::null(), process::Stdio::null()),
            Stdio::Collect => (process::Stdio::piped(), process::Stdio::piped()),
        };

        let child = Arc::new(
            Command::new(Self::SHELL)
                .args(Self::shelled(cmd.exe()))
                .envs(cmd.env().to_owned())
                .current_dir(cmd.pwd().as_path())
                .stdout(stdout)
                .stderr(stderr)
                .spawn()?,
        );

        let exit_reason = {
            let child = child.clone();
            tokio::select! {
                result = child.wait_with_output() => ExitReason::ProcessFinished(result),
                _ = signal::ctrl_c() => ExitReason::CtrlC,
            }
        };

        match exit_reason {
            ExitReason::ProcessFinished(result) => {
                let output = result?;
                if output.status.success() {
                    match stdio {
                        Stdio::Inherit | Stdio::Ignore => Ok(Done::Bye),
                        Stdio::Collect => Ok(Done::Output(output.into())),
                    }
                } else {
                    Err(output.into())
                }
            }
            ExitReason::CtrlC => {
                let res = {
                    let child = child.clone();
                    tokio::select! {
                        _ = child.wait() => CtrlCResult::ProcessExited,
                        _ = time::sleep(Self::timeout()) => CtrlCResult::Timeout,
                    }
                };

                match res {
                    CtrlCResult::ProcessExited => Ok(Done::Bye),
                    CtrlCResult::Timeout => {
                        if let Ok(()) = child.kill().await {
                            child.wait().await?;
                        }
                        Ok(Done::Bye)
                    }
                }
            }
        }
    }
}
