use std::{ffi::OsStr, fmt::Display, io, process::Stdio, sync::Arc, time::Duration};

use async_trait::async_trait;
#[cfg(unix)]
use nix::{errno::Errno, sys::signal::Signal, unistd::Pid, Error as NixError};
use tokio::{
    process::{Child, Command},
    sync::Mutex,
    time,
};

use crate::{CommandInterface, LocationInterface, Result};

pub enum ProcessStatus {
    Ready {
        tag: &'static str,
    },
    Running {
        tag: &'static str,
        process: Arc<Mutex<Child>>,
    },
}

#[async_trait]
pub trait ProcessInterface<'a, S, E, L, K, V, Cmd>: Send + Sync
where
    S: Display + Send + Sync,
    E: IntoIterator<Item = (K, V)> + Clone + Send + Sync,
    L: LocationInterface + Send + Sync,
    K: AsRef<OsStr>,
    V: AsRef<OsStr>,
    Cmd: CommandInterface<'a, S, E, L, K, V>,
{
    const TIMEOUT: u64 = 10; // sec

    fn tag(&self) -> &'static str;
    fn command(&self) -> &Cmd;
    fn status(&self) -> &ProcessStatus;
    fn set_status(&mut self, status: ProcessStatus);

    async fn up(&mut self) -> Result<()> {
        let cmd = self.command();

        eprintln!("{}", crate::headline!(cmd));

        let child = Command::new(Cmd::SHELL)
            .args(Cmd::shelled(cmd.exe()))
            .envs(cmd.env().to_owned())
            .current_dir(cmd.pwd().as_path())
            .stdout(Stdio::inherit())
            .stderr(Stdio::inherit())
            .spawn()?;

        self.set_status(ProcessStatus::Running {
            tag: self.tag(),
            process: Arc::new(Mutex::new(child)),
        });

        Ok(())
    }

    #[cfg(unix)]
    async fn down(&mut self) -> io::Result<()> {
        match self.status() {
            ProcessStatus::Ready { tag } => Err(io::Error::new(
                io::ErrorKind::InvalidInput,
                format!("[{process}] Process is not running", process = tag),
            )),
            ProcessStatus::Running { tag, process } => {
                let mut process = process.lock().await;
                match process.id() {
                    Some(process_id) => {
                        let pid = Pid::from_raw(process_id as i32);
                        match nix::sys::signal::kill(pid, Signal::SIGINT) {
                            Ok(()) => {
                                let res = tokio::select! {
                                    res = process.wait() => Some(res),
                                    _ = time::sleep(Self::timeout()) => None,
                                };
                                match res {
                                    Some(Ok(_)) => Ok(()),
                                    Some(Err(error)) => {
                                        eprintln!(
                                            "[{process} [{pid}]] IO error on SIGINT: {error}",
                                            process = tag,
                                            pid = pid,
                                            error = error,
                                        );
                                        self.kill().await
                                    }
                                    None => {
                                        eprintln!(
                                            "[{process} [{pid}]] Killing process after timeout",
                                            process = tag,
                                            pid = pid,
                                        );
                                        self.kill().await
                                    }
                                }
                            }
                            Err(NixError::Sys(Errno::EINVAL)) => {
                                eprintln!(
                                    "[{process} [{pid}]] Invalid signal. Killing process.",
                                    process = tag,
                                    pid = pid,
                                );
                                self.kill().await
                            }
                            Err(NixError::Sys(Errno::EPERM)) => Err(io::Error::new(
                                io::ErrorKind::InvalidInput,
                                format!(
                                    "[{process} [{pid}]] Insufficient permissions to signal process.",
                                    process = tag,
                                    pid = pid,
                                ),
                            )),
                            Err(NixError::Sys(Errno::ESRCH)) => Err(io::Error::new(
                                io::ErrorKind::InvalidInput,
                                format!(
                                    "[{process} [{pid}]] Process does not exist",
                                    process = tag,
                                    pid = pid,
                                ),
                            )),
                            Err(error) => Err(io::Error::new(
                                io::ErrorKind::InvalidInput,
                                format!(
                                    "[{process} [{pid}]] Unexpected error: {error}",
                                    process = tag,
                                    pid = pid,
                                    error = error,
                                ),
                            )),
                        }
                    }
                    None => Err(io::Error::new(
                        io::ErrorKind::InvalidInput,
                        format!("[{process}] Process does not have an id", process = tag),
                    )),
                }
            }
        }
    }

    #[cfg(windows)]
    async fn down(&mut self) -> io::Result<()> {
        match self.status {
            ProcessStatus::Ready { tag } => Err(io::Error::new(
                io::ErrorKind::InvalidInput,
                format!("[{process}] Process is not running", process = tag),
            )),
            ProcessStatus::Running { tag: _, process: _ } => self.kill(),
        }
    }

    async fn kill(&self) -> io::Result<()> {
        match self.status() {
            ProcessStatus::Ready { tag } => Ok(()),
            ProcessStatus::Running { tag: _, process } => {
                let mut process = process.lock().await;
                process.kill().await?;
                process.wait().await?;
                Ok(())
            }
        }
    }

    fn timeout() -> Duration {
        Duration::from_secs(Self::TIMEOUT)
    }
}

pub mod pool {
    use std::{
        ffi::OsStr,
        fmt::Display,
        process::Stdio,
        sync::{
            atomic::{AtomicUsize, Ordering},
            Arc,
        },
        time::{Duration, Instant},
    };

    use console::Color;
    use tokio::{
        io::{AsyncBufReadExt, BufReader},
        process::Command,
        signal,
        sync::Mutex,
        task, time,
    };

    use crate::{
        CommandInterface, Done, LocationInterface, ProcessInterface, ProcessStatus, Result,
    };

    enum TerdownReason {
        CtrlC,
        ProcessExited,
    }

    pub struct ProcessPool;

    impl ProcessPool {
        pub async fn run<'a, S, E, L, K, V, Cmd, P>(pool: Vec<P>) -> Result
        where
            S: Display + Send + Sync,
            E: IntoIterator<Item = (K, V)> + Clone + Send + Sync,
            L: LocationInterface + Send + Sync,
            K: AsRef<OsStr>,
            V: AsRef<OsStr>,
            Cmd: CommandInterface<'a, S, E, L, K, V> + Send + Sync,
            P: ProcessInterface<'a, S, E, L, K, V, Cmd> + 'static,
        {
            let pool_size = pool.len();
            let exited_processes = Arc::new(AtomicUsize::new(0));

            let tag_col_length = pool.iter().fold(0, |acc, process| {
                let len = process.tag().len();
                if len > acc {
                    len
                } else {
                    acc
                }
            });

            let colors = colors::make(pool_size as u8);

            let processes: Vec<(P, Color)> = pool.into_iter().zip(colors).collect();

            let processes_list = processes
                .iter()
                .fold(String::new(), |acc, (process, color)| {
                    let styled = console::style(process.tag().to_string()).fg(*color).bold();
                    if acc == "" {
                        styled.to_string()
                    } else {
                        format!("{}, {}", acc, styled)
                    }
                });

            eprintln!("❯ {} {}", console::style("Running:").bold(), processes_list);

            for (mut process, color) in processes {
                let exited_processes = exited_processes.clone();

                task::spawn(async move {
                    let tag = process.tag();
                    let cmd = process.command();
                    let colored_tag = console::style(tag.to_owned()).fg(color).bold();
                    let colored_tag_col = {
                        let len = tag.len();
                        let pad = " ".repeat(if len < tag_col_length {
                            tag_col_length - len + 2
                        } else {
                            2
                        });
                        console::style(format!(
                            "{tag}{pad}{pipe}",
                            tag = colored_tag,
                            pad = pad,
                            pipe = console::style("|").fg(color).bold()
                        ))
                    };

                    eprintln!(
                        "{tag} {headline}",
                        tag = colored_tag_col,
                        headline = crate::headline!(cmd),
                    );

                    let mut child = Command::new(Cmd::SHELL)
                        .args(Cmd::shelled(cmd.exe()))
                        .envs(cmd.env().to_owned())
                        .current_dir(cmd.pwd().as_path())
                        .stdout(Stdio::piped())
                        .stderr(Stdio::piped())
                        .spawn()
                        .expect(&format!("Failed to spawn the process: {}", cmd.exe()));

                    let child_stdout = child.stdout.take().expect(&format!(
                        "Failed to get a handle to stdout of {} ({})",
                        tag,
                        cmd.exe()
                    ));

                    let child_stderr = child.stderr.take().expect(&format!(
                        "Failed to get a handle to stderr of {} ({})",
                        tag,
                        cmd.exe()
                    ));

                    let mut child_stdout_reader = BufReader::new(child_stdout).lines();
                    let mut child_stderr_reader = BufReader::new(child_stderr).lines();

                    let stdout_printer = task::spawn({
                        let tag = colored_tag_col.clone();
                        async move {
                            while let Some(line) = child_stdout_reader.next_line().await.unwrap() {
                                eprintln!("{} {}", tag, line);
                            }
                        }
                    });

                    let stderr_printer = task::spawn({
                        let tag = colored_tag_col.clone();
                        async move {
                            while let Some(line) = child_stderr_reader.next_line().await.unwrap() {
                                eprintln!("{} {}", tag, line);
                            }
                        }
                    });

                    process.set_status(ProcessStatus::Running {
                        tag,
                        process: Arc::new(Mutex::new(child)),
                    });

                    let ctrlc = task::spawn({
                        let tag = colored_tag.clone();
                        async move {
                            signal::ctrl_c()
                                .await
                                .expect("Error setting process Ctrl-C handler");
                            eprintln!("Exiting: {}", tag);
                        }
                    });

                    let res = child.wait_with_output().await;

                    // for process in pool {
                    //     if let Err(err) = process.kill().await {
                    //         eprintln!("⚠️  Failed to kill the process `{}`: {}", process.tag(), err);
                    //     }
                    // }

                    stdout_printer.abort();
                    stderr_printer.abort();
                    ctrlc.abort();

                    match res {
                        Ok(output) => match output.status.code() {
                            Some(0) => eprintln!(
                                "{} Process {} exited with code 0.",
                                colored_tag_col, colored_tag
                            ),
                            Some(code) => eprintln!(
                                "{} Process {} exited with non-zero code: {}",
                                colored_tag_col, colored_tag, code
                            ),
                            None => eprintln!(
                                "{} Process {} exited without code.",
                                colored_tag_col, colored_tag
                            ),
                        },
                        Err(error) => eprintln!(
                            "{} Process {} exited with error: {}",
                            colored_tag_col, colored_tag, error
                        ),
                    }
                    exited_processes.fetch_add(1, Ordering::Relaxed);
                });
            }

            signal::ctrl_c().await.unwrap();

            let expire = Instant::now() + P::timeout();
            while exited_processes.load(Ordering::Relaxed) < pool_size {
                if Instant::now() > expire {
                    eprintln!("⚠️  Timeout. Exiting.");
                    break;
                }
                time::sleep(Duration::from_millis(500)).await;
            }

            Ok(Done::Bye)
        }
    }

    mod colors {
        use console::Color;
        use rand::{seq::SliceRandom, thread_rng};

        pub fn make(n: u8) -> Vec<Color> {
            // Preferred colors
            let mut primaries = vec![
                // Color::Red, // Red is for errors
                Color::Green,
                Color::Yellow,
                Color::Blue,
                Color::Magenta,
                Color::Cyan,
            ];
            // Not as good as primaries, but good enough to distinct processes
            let secondaries = vec![
                Color::Color256(24),
                Color::Color256(172),
                Color::Color256(142),
            ];

            // Let's check first if we can get away with just primary colors
            if n <= primaries.len() as u8 {
                shuffle(primaries, n)
            }
            // Otherwise, let's check if primary + secondary combined would work
            else if n <= (primaries.len() + primaries.len()) as u8 {
                primaries.extend(secondaries);
                shuffle(primaries, n)
            } else {
                // TODO: Duplicate primary + secondary colors vec as many is needed, then shuffle
                todo!()
            }
        }

        fn shuffle<T>(mut items: Vec<T>, n: u8) -> Vec<T> {
            items.truncate(n as usize);
            items.shuffle(&mut thread_rng());
            items
        }
    }
}
