
use std::str;
use std::convert::TryInto;

use crate::{StellarValue, StellarObject};

pub fn object(serialized_object: &Vec<u8>) -> Result<StellarObject, String> {

    let value_type: u8 = u8::from_le_bytes([serialized_object[0]; 1]);

    let key_length: usize = u8::from_le_bytes([serialized_object[1]; 1]) as usize;

    let value_index: usize = key_length + 2;
    
    let key_slice = &serialized_object[2..value_index];
    
    let value_slice = &serialized_object[value_index..serialized_object.len()];
    
    let key: String = str::from_utf8(key_slice).unwrap().to_string();

    match value_type {
        
        1 => {
            let value = str::from_utf8(value_slice).unwrap().to_string();
            return Ok(StellarObject(key, StellarValue::Str(value)))
        },
        
        2 => {
            let value: u8 = u8::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt8(value)))
        },

        3 => {
            let value: u16 = u16::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt16(value)))
        },

        4 => {
            let value: u32 = u32::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt32(value)))
        },

        5 => {
            let value: u64 = u64::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt64(value)))
        },

        6 => {
            let value: u128 = u128::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt128(value)))
        },
        
        15 => {
            return Ok(StellarObject(key, StellarValue::Bytes(value_slice.to_vec())))
        }
        _ => Err(String::from("no value type found"))
    }

}

pub fn list(serialized_objects: &Vec<u8>) -> Vec<StellarObject> {

    let index_1_vec = &serialized_objects[0..8];

    let index_1_value = usize::from_le_bytes(index_1_vec.try_into().unwrap());

    let full_index_vec = &serialized_objects[0..index_1_value];

    let index_vecs = full_index_vec.len();

    let indices = index_vecs/8;

    let mut object_start = index_vecs;

    let mut objects: Vec<Vec<u8>> = Vec::new();

    for i in 0..indices {

        if i == (indices - 1) {
            
            let object_end = serialized_objects.len();

            let object = &serialized_objects[object_start..object_end];

            objects.push(object.to_vec());
            
        
        } else {

            let index_start = (i + 1) * 8;

            let index_end = index_start + 8;

            let index_vec = &serialized_objects[index_start..index_end];

            let object_end = usize::from_le_bytes(index_vec.try_into().unwrap());

            let object = &serialized_objects[object_start..object_end];

            objects.push(object.to_vec());

            object_start = object_end;

        }

    }

    let deserialize_objects: Vec<StellarObject> = objects.iter()
        .map(|x| object(x).unwrap())
        .collect();

    return deserialize_objects

}
