
use std::str;
use std::convert::TryInto;

#[derive(Debug, Clone)]
pub enum StellarValue {
    String(String),
    UInt8(u8),
    UInt16(u16),
    UInt32(u32),
    UInt64(u64),
    UInt128(u128),
    Bytes(Vec<u8>)
}

#[derive(Debug, Clone)]
pub struct StellarObject(pub String, pub StellarValue);

pub fn serialize_stellar_object(object: &StellarObject) -> Vec<u8> {

    let key: String = object.0.to_string();
    
    let key_bytes: Vec<u8> = key.into_bytes();

    let key_length: u8 = key_bytes.len() as u8;

    let key_length_bytes: Vec<u8> = key_length.to_le_bytes().to_vec();

    let value: StellarValue = object.1.clone();

    match value {

        StellarValue::String(res) => {
            return [1_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.into_bytes()].concat()
        },

        StellarValue::UInt8(res) => {
            return [2_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.to_le_bytes().to_vec()].concat()
        },

        StellarValue::UInt16(res) => {
            return [3_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.to_le_bytes().to_vec()].concat()
        },

        StellarValue::UInt32(res) => {
            return [4_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.to_le_bytes().to_vec()].concat()
        },

        StellarValue::UInt64(res) => {
            return [5_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.to_le_bytes().to_vec()].concat()
        },

        StellarValue::UInt128(res) => {
            return [6_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res.to_le_bytes().to_vec()].concat()
        },

        StellarValue::Bytes(res) => {
            return [15_u8.to_le_bytes().to_vec(), key_length_bytes, key_bytes, res].concat()
        }

    };

}

pub fn deserialize_stellar_object(serialized_object: &Vec<u8>) -> Result<StellarObject, String> {

    let value_type: u8 = u8::from_le_bytes([serialized_object[0]; 1]);

    let key_length: usize = u8::from_le_bytes([serialized_object[1]; 1]) as usize;

    let value_index: usize = key_length + 2;
    
    let key_slice = &serialized_object[2..value_index];
    
    let value_slice = &serialized_object[value_index..serialized_object.len()];
    
    let key: String = str::from_utf8(key_slice).unwrap().to_string();

    match value_type {
        
        1 => {
            let value = str::from_utf8(value_slice).unwrap().to_string();
            return Ok(StellarObject(key, StellarValue::String(value)))
        },
        
        2 => {
            let value: u8 = u8::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt8(value)))
        },

        3 => {
            let value: u16 = u16::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt16(value)))
        },

        4 => {
            let value: u32 = u32::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt32(value)))
        },

        5 => {
            let value: u64 = u64::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt64(value)))
        },

        6 => {
            let value: u128 = u128::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::UInt128(value)))
        },
        
        15 => {
            return Ok(StellarObject(key, StellarValue::Bytes(value_slice.to_vec())))
        }
        _ => Err(String::from("no value type found"))
    }

}

pub fn serialize_stellar_objects(mut objects: Vec<StellarObject>) -> Vec<u8> {

    objects.sort_by_key(|x| x.0.to_string());

    objects.reverse();

    objects.dedup_by_key(|x| x.0.to_string());

    objects.sort_by_key(|x| x.0.to_string());

    let serialized_objects: Vec<Vec<u8>> = objects.iter()
        .map(|x| serialize_stellar_object(x))
        .collect();

    let serialized_objects_concatenated = serialized_objects.concat();

    let object_lengths: Vec<u64> = serialized_objects.iter()
        .map(|x| x.len() as u64)
        .collect();

    let mut current_object_location: u64 = object_lengths.len() as u64 * 8;

    let mut index_vec: Vec<Vec<u8>> = Vec::new();

    for object_length in object_lengths {

        index_vec.push(current_object_location.to_le_bytes().to_vec());
        current_object_location += object_length;

    }

    let result = [index_vec.concat(), serialized_objects_concatenated].concat();

    return result;

}

pub fn deserialize_stellar_objects(serialized_objects: &Vec<u8>) -> Vec<StellarObject> {

    let index_1_vec = &serialized_objects[0..8];

    let index_1_value = usize::from_le_bytes(index_1_vec.try_into().unwrap());

    let full_index_vec = &serialized_objects[0..index_1_value];

    let index_vecs = full_index_vec.len();

    let indices = index_vecs/8;

    let mut object_start = index_vecs;

    let mut objects: Vec<Vec<u8>> = Vec::new();

    for i in 0..indices {

        if i == (indices - 1) {
            
            let object_end = serialized_objects.len();

            let object = &serialized_objects[object_start..object_end];

            objects.push(object.to_vec());
            
        
        } else {

            let index_start = (i + 1) * 8;

            let index_end = index_start + 8;

            let index_vec = &serialized_objects[index_start..index_end];

            let object_end = usize::from_le_bytes(index_vec.try_into().unwrap());

            let object = &serialized_objects[object_start..object_end];

            objects.push(object.to_vec());

            object_start = object_end;

        }

    }

    let deserialize_objects: Vec<StellarObject> = objects.iter()
        .map(|x| deserialize_stellar_object(x).unwrap())
        .collect();

    return deserialize_objects

}
