# Stellar Notation Rust

The Rust Implementation of Stellar Notation, a serialization library of a data interchange format for transmission and storage.

## Usage

```
stellar-notation = "0.9.3"
```

```
use stellar_notation::{
    StellarObject, StellarValue,
    serialize_stellar_object, deserialize_stellar_object,
    serialize_stellar_objects, deserialize_stellar_objects
}
```

## Object

An Object is a Key-Value Tuple. Keys are UTF-8 String and Values can be a UTF-8 String, Unsigned Integer and Bytes.

### Value Types

| Enum | Type | Code | Support |
|---|---|---|---|
| String | UTF-8 | 0x01 | ✔ |
| UInt8 | u8 | 0x02 | ✔ |
| UInt16 | u16 | 0x03 | ✔ |
| UInt32 | u32 | 0x04 | ✔ |
| UInt64 | u64 | 0x05 | ✔ |
| UInt128 | u128 | 0x06 | ✔ |
| SInt8 | i8 | 0x07 | |
| SInt16 | i16 | 0x08 | |
| SInt32 | i32 | 0x09 | |
| SInt16 | i16 | 0x08 | |
| SInt32 | i32 | 0x09 | |
| SInt64 | i64 | 0x0A | |
| SInt128 | i128 | 0x0B | |
| FP32 | fp32 | 0x0C | |
| FP64 | fp64 | 0x0D | |
| Bool | bool | 0x0E | |
| Bytes | Vec*<*u8> | 0x0F | ✔ |

### Object Creation

```
let key: String = String::from("key_1");
let value: StellarValue = StellarValue::StringType(String::from("value_1"));
let object: StellarObject = StellarObject(key, value);
```

### Object Serialization

Converts a StellarObject into bytes.

```
let serialized: Vec<u8> = serialize_stellar_object(object);
```

### Serialized Object Structure

| Value Type | Key Length | Key Data | Value Data |
|---|---|---|---|
| 1 byte | 1 byte | max 255 bytes | max ~18 exabytes |

Value data is also limited by the system file size limit.

### Object Deserialization

Reconverts bytes into a StellarObject

```

let object: StellarObject = deserialize_stellar_object(serialized);

```

## Objects

### Objects Serialization

Converts a collection of StellarObjects into bytes.

```

let objects: Vec<StellarObject> = vec![
    object1, object2, object3
]

let serialized_objects: Vec<u8> = serialize_stellar_objects(objects);

```

### Serialized Objects Structure

| Index | Objects |
|---|---|
| 8 bytes each | concatenated serialized objects |

### Objects Deserialization

Reconverts bytes into a collection of StellarObjects.

```

let deserialized_objects: Vec<StellarObject> = deserialize_stellar_objects(serialized_objects);

```

## Future Topics
- Signed Integers
- Floating Point Numbers
- Boolean

