
use std::str;
use std::convert::TryInto;

#[derive(Debug, Clone)]
pub enum StellarValue {
    StringType(String),
    IntegerType(u128),
    ByteType(Vec<u8>)
}

#[derive(Clone, Debug)]
pub struct StellarObject(pub String, pub StellarValue);

pub fn serialize(object: StellarObject) -> Vec<u8> {
    
    let key_bytes: Vec<u8> = object.0.into_bytes();

    let key_length: u8 = key_bytes.len() as u8;

    let key_length_bytes: Vec<u8> = key_length.to_le_bytes().to_vec();

    match object.1 {

        StellarValue::StringType(r) => {
            let value_type = 1_u8.to_le_bytes().to_vec();
            let value_bytes = r.into_bytes();
            let result: Vec<u8> = [value_type, key_length_bytes, key_bytes, value_bytes].concat();
            return result
        },

        StellarValue::IntegerType(r) => {
            let value_type = 2_u8.to_le_bytes().to_vec();
            let value_bytes = r.to_le_bytes().to_vec();
            let result: Vec<u8> = [value_type, key_length_bytes, key_bytes, value_bytes].concat();
            return result
        },

        StellarValue::ByteType(r) => {
            let value_type = 3_u8.to_le_bytes().to_vec();
            let value_bytes = r;
            let result: Vec<u8> = [value_type, key_length_bytes, key_bytes, value_bytes].concat();
            return result
        }

    };

}

pub fn deserialize(vec: &Vec<u8>) -> Result<StellarObject, String> {

    let value_type = usize::from_le(vec[0].into());

    let key_length = usize::from_le(vec[1].into());
    
    let key_slice = &vec[2..key_length + 2];
    
    let value_slice = &vec[key_length + 2..vec.len()];
    
    let key = str::from_utf8(key_slice).unwrap().to_string();

    match value_type {
        1 => {
            let value = str::from_utf8(value_slice).unwrap().to_string();
            return Ok(StellarObject(key, StellarValue::StringType(value)))
        },
        2 => {
            let value = u128::from_le_bytes(value_slice.try_into().unwrap());
            return Ok(StellarObject(key, StellarValue::IntegerType(value)))
        },
        3 => {
            return Ok(StellarObject(key, StellarValue::ByteType(value_slice.to_vec())))
        }
        _ => Err(String::from("no value type found"))
    }

}

pub fn encode(objects: &Vec<Vec<u8>>) -> Vec<u8> {

    let concat_objects = objects.concat();

    let obj_sizes: Vec<u64> = objects.into_iter().map(|x| x.len() as u64).collect();

    let mut object_loc: u64 = obj_sizes.len() as u64 * 8;

    let mut index_vec: Vec<Vec<u8>> = Vec::new();

    for obj_size in obj_sizes {
        index_vec.push(object_loc.to_le_bytes().to_vec());
        object_loc += obj_size;
    }

    let result = [index_vec.concat(), concat_objects].concat();

    return result;

}

pub fn decode(list: &Vec<u8>) -> Vec<Vec<u8>> {

    let index_1_vec = &list[0..8];

    let index_1_value = usize::from_le_bytes(index_1_vec.try_into().unwrap());

    let full_index_vec = &list[0..index_1_value];

    let index_vecs = full_index_vec.len();

    let indices = index_vecs/8;

    let mut object_start = index_vecs;

    let mut objects: Vec<Vec<u8>> = Vec::new();

    for i in 0..indices {

        if i == (indices - 1) {
            
            let object_end = list.len();

            let object = &list[object_start..object_end];

            objects.push(object.to_vec());
            
        
        } else {

            let index_start = (i + 1) * 8;

            let index_end = index_start + 8;

            let index_vec = &list[index_start..index_end];

            let object_end = usize::from_le_bytes(index_vec.try_into().unwrap());

            let object = &list[object_start..object_end];

            objects.push(object.to_vec());

            object_start = object_end;

        }

    }

    return objects

}

pub fn push(list: &Vec<u8>, object: StellarObject) -> Vec<u8> {

    let mut objects = decode(&list);

    let serialized_object = serialize(object);

    objects.push(serialized_object);

    let result = encode(&objects);

    return result

}

pub fn find(list: &Vec<u8>, key: &str) -> Option<StellarObject> {

    let decoded: Vec<Vec<u8>> = decode(&list);

    let objects: Vec<StellarObject> = decoded.iter()
        .map(|x| deserialize(x).unwrap())
        .collect();

    let query = objects.iter()
        .find(|x| x.0 == key);

    match query {
        Some(r) => Some(r.to_owned()),
        None => None
    }

}

pub fn remove(list: &Vec<u8>, key: &str) -> Vec<u8> {

    let objects = decode(list);

    let mut deserialized_objects: Vec<StellarObject> = objects.iter()
        .map(|x| deserialize(x).unwrap())
        .collect();

    deserialized_objects.retain(|x| x.0 != key);

    let serialized_objects: Vec<Vec<u8>> = deserialized_objects.iter()
        .map(|x| serialize(x.to_owned()))
        .collect();

    let result = encode(&serialized_objects);

    return result

}
