
use std::str;
use std::convert::TryFrom;
use std::convert::TryInto;

#[derive(Debug)]
pub enum ValueType {
    StringType(String),
    IntegerType(u128),
    VecType(Vec<u8>)
}

pub fn serialize(key: &str, value: ValueType) -> Result<Vec<u8>, String> {
    
    let key_bytes: Vec<u8> = String::from(key).into_bytes();

    let key_length = u8::try_from(key_bytes.len());

    match key_length {
        
        Ok(len) => {

            match value {

                ValueType::StringType(r) => {
                    
                    let value_type = 1_u8.to_le_bytes().to_vec();
                    let value_bytes = r.into_bytes();
                    let result: Vec<u8> = [value_type, len.to_le_bytes().to_vec(), key_bytes, value_bytes].concat();
                    return Ok(result)
                
                },

                ValueType::IntegerType(r) => {
                    
                    let value_type = 2_u8.to_le_bytes().to_vec();
                    let value_bytes = r.to_le_bytes().to_vec();
                    let result: Vec<u8> = [value_type, len.to_le_bytes().to_vec(), key_bytes, value_bytes].concat();
                    return Ok(result)

                },

                ValueType::VecType(r) => {
                    let value_type = 3_u8.to_le_bytes().to_vec();
                    let value_bytes = r;
                    let result: Vec<u8> = [value_type, len.to_le_bytes().to_vec(), key_bytes, value_bytes].concat();
                    return Ok(result)
                }

            };

        },

        Err(_) => Err(String::from("key is too long"))

    }

}

pub fn deserialize(vec: Vec<u8>) -> Result<(String, ValueType), String> {

    let value_type = usize::from_le(vec[0].into());

    let key_length = usize::from_le(vec[1].into());
    
    let key_slice = &vec[2..key_length + 2];
    
    let value_slice = &vec[key_length + 2..vec.len()];
    
    let key = str::from_utf8(key_slice).unwrap().to_string();

    match value_type {
        1 => {
            let value = str::from_utf8(value_slice).unwrap().to_string();
            return Ok((key, ValueType::StringType(value)))
        },
        2 => {
            let value = u128::from_le_bytes(value_slice.try_into().unwrap());
            return Ok((key, ValueType::IntegerType(value)))
        },
        3 => {
            return Ok((key, ValueType::VecType(value_slice.to_vec())))
        }
        _ => Err(String::from("no value type found"))
    }

}

pub fn encode(objects: Vec<Vec<u8>>) -> Vec<u8> {

    let concat_objects = objects.concat();

    let obj_sizes: Vec<u64> = objects.into_iter().map(|x| x.len() as u64).collect();

    let mut object_loc: u64 = obj_sizes.len() as u64 * 8;

    let mut index_vec: Vec<Vec<u8>> = Vec::new();

    for obj_size in obj_sizes {
        index_vec.push(object_loc.to_le_bytes().to_vec());
        object_loc += obj_size;
    }

    let result = [index_vec.concat(), concat_objects].concat();

    return result;

}

pub fn decode(list: &Vec<u8>) -> Vec<Vec<u8>> {

    let index_1_vec = &list[0..8];

    let index_1_value = usize::from_le_bytes(index_1_vec.try_into().unwrap());

    let full_index_vec = &list[0..index_1_value];

    let index_vecs = full_index_vec.len();

    let indices = index_vecs/8;

    let mut object_start = index_vecs;

    let mut objects: Vec<Vec<u8>> = Vec::new();

    for i in 0..indices {

        if i == (indices - 1) {
            
            let object_end = list.len();

            let object = &list[object_start..object_end];

            objects.push(object.to_vec());
            
        
        } else {

            let index_start = (i + 1) * 8;

            let index_end = index_start + 8;

            let index_vec = &list[index_start..index_end];

            let object_end = usize::from_le_bytes(index_vec.try_into().unwrap());

            let object = &list[object_start..object_end];

            objects.push(object.to_vec());

            object_start = object_end;

        }

    }

    return objects

}

pub fn push(list: &Vec<u8>, object: &Vec<u8>) -> Vec<u8> {

    let mut objects = decode(&list);

    objects.retain(|x| get_object_key(&x) != get_object_key(object));

    objects.push(object.to_vec());

    let new_list = encode(objects);

    return new_list

}

pub fn find(list: &Vec<u8>, key: &str) -> Option<Vec<u8>> {

    let mut objects = decode(&list);

    objects.retain(|x| get_object_key(x) == String::from(key));

    let retention_result = &objects[0];

    match objects.is_empty() {
        true => None,
        false => Some(retention_result.to_vec())
    }

}

pub fn remove(list: &Vec<u8>, key: &str) -> Vec<u8> {

    let mut objects = decode(list);

    objects.retain(|x| get_object_key(x) != String::from(key));

    let new_list = encode(objects);

    return new_list

}

pub fn get_object_key(object: &Vec<u8>) -> String {

    let key_length = usize::from_le(object[1].into());
    
    let key_slice = &object[2..(key_length + 2)];

    let key = str::from_utf8(key_slice).unwrap().to_string();

    return key

}
