
use std::str;
use std::convert::TryFrom;
use std::convert::TryInto;

pub fn string_object(key: &str, value: String) -> Vec<u8> {

    let value_type: Vec<u8> = 1_u8.to_le_bytes().to_vec();
    
    let key_bytes: Vec<u8> = String::from(key).into_bytes();
            
    let value_bytes: Vec<u8> = value.into_bytes();

    let key_bytes_length = u8::try_from(key_bytes.len());

    match key_bytes_length {
        
        Ok(len) => {

            let result: Vec<u8> = [
                value_type,
                len.to_le_bytes().to_vec(),
                key_bytes,
                value_bytes
            ].concat();

            return result

        },

        Err(_) => return vec![]

    }

}

pub fn integer_object(key: &str, value: u128) -> Vec<u8> {

    let value_type: Vec<u8> = 2_u8.to_le_bytes().to_vec();
    
    let key_bytes: Vec<u8> = String::from(key).into_bytes();
            
    let value_bytes: Vec<u8> = value.to_le_bytes().to_vec();

    let key_bytes_length = u8::try_from(key_bytes.len());

    match key_bytes_length {
        
        Ok(len) => {

            let result: Vec<u8> = [
                value_type,
                len.to_le_bytes().to_vec(),
                key_bytes,
                value_bytes
            ].concat();

            return result

        },

        Err(_) => return vec![]

    }

}

pub fn deserialize(vec: Vec<u8>) -> (String, String) {

    let key_length = usize::from_le(vec[1].into());
    
    let key_slice = &vec[2..key_length + 2];
    
    let value_slice = &vec[key_length + 2..vec.len()];
    
    let key = str::from_utf8(key_slice).unwrap().to_string();
    
    let value = str::from_utf8(value_slice).unwrap().to_string();
    
    return (key, value)

}

pub fn encode(objects: Vec<Vec<u8>>) -> Vec<u8> {

    let concat_objects = objects.concat();

    let obj_sizes: Vec<usize> = objects.into_iter().map(|x| x.len()).collect();

    let mut current_index = obj_sizes.len() * 8;

    let mut result: Vec<u8> = Vec::new();

    for x in &obj_sizes[0..obj_sizes.len()] {

        let index = current_index.to_le_bytes().to_vec();
        result = [result, index].concat();
        current_index += x;

    }

    result = [result, concat_objects].concat();

    return result;

}

pub fn decode(list: Vec<u8>) -> Vec<Vec<u8>> {
    // get index
    let index_1_vec = &list[0..8];

    let index_1_value = usize::from_le_bytes(index_1_vec.try_into().unwrap());

    let full_index_vec = &list[0..index_1_value];

    let index_vecs = full_index_vec.len();

    let mut objects: Vec<Vec<u8>> = Vec::new();
    // iter through index

    let mut object_start = index_vecs;

    for i in 0..(index_vecs/8) {

        let index_start = i * 8;
        let index_end = index_start + 8;

        let index_vec = &list[index_start..index_end];

        let index_value = usize::from_le_bytes(index_vec.try_into().unwrap());

        let object_end = object_start + index_value;

        let object = &list[object_start..object_end];

        objects.push(object.to_vec());

        object_start = object_end;

    }

    return objects

}

pub fn push(list: Vec<u8>, object: Vec<u8>) -> Vec<u8> {

    let mut decoded_list = decode(list);

    decoded_list.push(object);

    decoded_list.sort_by_key(|x| get_key(x.to_vec()));

    let encoded_objects = encode(decoded_list);

    return encoded_objects

}

pub fn find(list: Vec<u8>, key: &str) -> Vec<u8> {

    let decoded_list = decode(list);

    let result = decoded_list.iter().find(|&x| get_key(x.to_vec()) == String::from(key));

    match result {
        
        Some(res) => {
            return res.to_vec()
        },

        None => {
            return vec![]
        }

    }

}

pub fn remove(list: Vec<u8>, key: &str) -> Vec<u8> {

    let objects = decode(list);

    let mut new_objects: Vec<Vec<u8>> = Vec::new();

    for object in objects {

        let object_key = get_key(object.to_vec());

        if object_key != String::from(key) {

            new_objects.push(object)

        }

    }

    let encoded_objects = encode(new_objects);

    return encoded_objects

}

fn get_key(object: Vec<u8>) -> String {

    let key_length = usize::from_le(object[1].into());
    
    let key_slice = &object[2..(key_length + 2)];

    let key = str::from_utf8(key_slice).unwrap().to_string();

    return key

}

pub fn sort(list: Vec<u8>) -> Vec<u8> {

    let mut decoded_list = decode(list);

    decoded_list.sort_by_key(|x| get_key(x.to_vec()));

    let encoded_objects = encode(decoded_list);

    return encoded_objects

}