use serde::{Deserialize, Serialize};

pub mod bans {
    use super::{Deserialize, Serialize};

    #[derive(Debug, Deserialize, Serialize, Default)]
    #[allow(non_snake_case)]
    pub struct User {
        /// Returns Steam ID of specified account
        pub SteamId: String,
        /// Whether or not the account is community banned.
        pub CommunityBanned: bool,
        /// Whether or not the account is VAC banned.
        pub VACBanned: bool,
        /// Amount of VAC bans the account has, Returns 0 if the account has none.
        pub NumberOfVACBans: i32,
        /// Amount of days since the last ban
        pub DaysSinceLastBan: i32,
        /// Amount of Game bans the account has, Returns 0 if the account has none.
        pub NumberOfGameBans: i32,
        /// Whether or not the account has an economy ban
        pub EconomyBan: String,
    }

    #[derive(Debug, Deserialize, Default)]
    pub struct Response {
        /// A vector of all the users
        pub players: Vec<User>,
    }
}

pub mod level {
    use super::{Deserialize, Serialize};

    #[derive(Debug, Serialize, Deserialize, Default)]
    #[allow(non_snake_case)]
    pub struct User {
        /// The steam level of the profile
        pub player_level: Option<i32>,
    }

    #[derive(Debug, Serialize, Deserialize, Default)]
    #[allow(non_snake_case)]
    pub struct Response {
        pub response: User,
    }
}

pub mod friends {
    use super::{Deserialize, Serialize};

    #[derive(Debug, Deserialize, Serialize, Default)]
    #[allow(non_snake_case)]
    pub struct User {
        /// Returns Steam ID of friend
        pub steamid: String,
        /// Friends relationship to the specified Steam ID
        pub relationship: String,
        /// Amount of time the users has been friends with the specified Steam ID
        pub friend_since: i32,
    }

    #[derive(Debug, Deserialize, Default)]
    #[allow(non_snake_case)]
    pub struct Users {
        pub friends: Vec<User>,
    }

    #[derive(Debug, Deserialize, Default)]
    pub struct Response {
        pub friendslist: Users,
    }
}

pub mod summaries {
    use super::{Deserialize, Serialize};

    /// `communityvisibilitystate`
    /// Returns `3` if the profile is public
    /// Otherwise the profile isn't visible to you
    /// # Example output
    /// `3`
    ///
    /// `relationship`
    /// Friends relationship to the specified Steam ID
    ///
    /// `friend_since`
    /// Amount of time the users has been friends with the specified Steam ID
    #[derive(Debug, Deserialize, Serialize, Default)]
    pub struct User {
        /// Returns Steam ID of specified account
        pub steamid: String,

        /// The current visibility state of the profile
        pub communityvisibilitystate: i32,

        /// Profile state
        ///
        /// 0 if the profile doesn't have a community profile
        /// 1 if it does
        pub profilestate: i32,

        /// Profile name (not the actual account login name)
        pub personaname: String,

        /// Comment section permissions
        /// TODO: look into this further, docs online are lacking
        /// TODO: add tests
        pub commentpermissions: Option<i32>,

        // The profiles custom steam url
        pub profileurl: String,
        /// Url of the players avatar (32x32px)
        pub avatar: String,

        /// Url of the players avatar (64x64px)
        pub avatarmedium: String,

        /// Url of the players avatar (184x184px)
        pub avatarfull: String,

        /// Hash of the users avatar
        pub avatarhash: String,

        /// Time since user was last online (unix timestamp)
        pub lastlogoff: Option<i32>,

        /// The Users current status:
        ///
        /// 0 = Offline
        ///
        /// 1 = Online
        ///
        /// 2 = Busy
        ///
        /// 3 = Away
        ///
        /// 4 = Snooze
        ///
        /// 5 = looking to trade
        ///
        /// 6 = looking to play
        ///
        /// If the player's profile is private, this will always be 0.
        pub personastate: i32,

        // private data
        pub realname: Option<String>,
        pub primaryclanid: Option<String>,
        pub timecreated: Option<i32>,
        pub gameid: Option<String>,
        pub gameserverip: Option<String>,
        pub gameextrainfo: Option<String>,
        pub loccountrycode: Option<String>,
        pub locstatecode: Option<String>,
        pub loccityid: Option<i32>,
    }

    #[derive(Debug, Deserialize, Default)]
    pub struct Users {
        pub players: Vec<User>,
    }

    #[derive(Debug, Deserialize, Default)]
    pub struct Response {
        pub response: Users,
    }
}

pub mod profile {
    use super::Serialize;
    // This will hold the info for each player from all the above structs
    #[derive(Default, Debug, Serialize)]
    #[allow(non_snake_case)] // blame valve for the random variable naming schemes
    pub struct User {
        // GetPlayerSummaries
        // lowercase
        pub steamid: String,
        pub communityvisibilitystate: i32,
        pub profilestate: i32,
        pub personaname: String,
        pub commentpermissions: i32,
        pub profileurl: String,
        pub avatar: String,
        pub avatarmedium: String,
        pub avatarfull: String,
        pub avatarhash: String,
        pub lastlogoff: i32,
        pub personastate: i32,

        // private data
        pub realname: String,
        pub primaryclanid: String,
        pub timecreated: i32,
        pub gameid: String,
        pub gameserverip: String,
        pub gameextrainfo: String,
        pub loccountrycode: String,
        pub locstatecode: String,
        pub loccityid: i32,

        //GetPlayerBans
        //CamelCase
        pub CommunityBanned: bool,
        pub VACBanned: bool,
        pub NumberOfVACBans: i32,
        pub DaysSinceLastBan: i32,
        pub NumberOfGameBans: i32,
        pub EconomyBan: String,

        //GetSteamLevel
        //snake_case
        pub player_level: i32,
    }

    // this will store all the above structs if multiple id's are requested
    #[derive(Debug, Serialize, Default)]
    pub struct Users {
        pub api_call_count: i32,
        pub user: Vec<User>,
    }
}

pub mod api {
    #[derive(Default)]
    pub struct Api<'a> {
        pub interface: &'a str,
        pub version: &'a str,
        pub accepts_multiple_ids: bool,
    }
}
