/// Generates a Steam API url
///
/// # Returns
/// A string containing the URL
///
/// # Arguments
/// * `method` - A string slice containing the desired method
/// * `steamids` - A string slice containing the steamids
/// * `api_key` - A string slice containing the API Key to use with the API.
///
/// # Panics
/// `manage_api_url` will panic if an invalid/unsupported method is passed
///
/// # Example
/// Single Steam ID
/// ```
/// let method = "GetSteamLevel";
/// let steamids = "76561198421169032";
/// let api_key = "XXXXXXXXXXX";
///
/// let url = steam_api::functions::manage_api_url(method, steamids, api_key);
/// assert_eq!(url, "https://api.steampowered.com/IPlayerService/GetSteamLevel/v1/?key=XXXXXXXXXXX&steamid=76561198421169032");
/// ```
#[must_use]
pub fn manage_api_url(method: &str, steamids: &str, api_key: &str) -> String {
    let mut api: crate::structs::api::Api = crate::structs::api::Api::default();

    // GetFriendList and GetSteamLevel don't work with multiple steamids
    // Thanks valve
    match method {
        "GetPlayerSummaries" => {
            api.interface = "ISteamUser";
            api.version = "v2";
            api.accepts_multiple_ids = true;
        }
        "GetFriendList" => {
            api.interface = "ISteamUser";
            api.version = "v1";
            api.accepts_multiple_ids = false;
        }
        "GetPlayerBans" => {
            api.interface = "ISteamUser";
            api.version = "v1";
            api.accepts_multiple_ids = true;
        }
        "GetSteamLevel" => {
            api.interface = "IPlayerService";
            api.version = "v1";
            api.accepts_multiple_ids = false;
        }
        _ => panic!("Method not found!"),
    }

    if api.accepts_multiple_ids {
        format!(
            "https://api.steampowered.com/{}/{}/{}/?key={}&steamids={}",
            api.interface, method, api.version, api_key, steamids
        )
    } else {
        format!(
            "https://api.steampowered.com/{}/{}/{}/?key={}&steamid={}",
            api.interface, method, api.version, api_key, steamids
        )
    }
}
