use crate::{
    elem::{AssociatedCollectionTrait, CollectionTrait, Owned},
    exact_size::ExactSized,
};
use std::hint::unreachable_unchecked;

/// Collections that can construct `Self` without data.
pub trait EmptyConstructible {
    /// Construct `Self` without data.
    fn new() -> Self
    where
        Self: Owned + Sized,
    {
        // SAFETY: Constrained by trait requirements
        unsafe { unreachable_unchecked() }
    }
}

/// Collections that can clear itself contents
pub trait Erasable {
    /// Clear `self`, removing all values
    fn clear(&mut self)
    where
        Self: Owned,
    {
        // SAFETY: Constrained by trait requirements
        unsafe { unreachable_unchecked() }
    }
}

/// Collections that can retain a specific element
pub trait Retainable: CollectionTrait {
    /// Retains only the elements specified by the predicate.
    fn retain<F: FnMut(&Self::ElemType) -> bool>(&mut self, pred: F);
}

/// Collections that can retain a specific element
pub trait AssociatedRetainable: AssociatedCollectionTrait {
    /// Retains only the elements specified by the predicate.
    fn retain<F: FnMut(&Self::KeyType, &mut Self::ValueType) -> bool>(&mut self, pred: F);
}

/// Collections that can search elements
pub trait Contains: CollectionTrait {
    /// Returns true if `self` contains an element equal to the given value.
    fn contains(&self, value: &Self::ElemType) -> bool;
}

/// Collections that can search elements
pub trait AssociatedContains: AssociatedCollectionTrait {
    /// Returns true if `self` contains an value for the specified key.
    ///
    /// Note: `{BTreeMap, HashMap}::contains_key` are more general as they have different bounds on `K: Borrow<Q>`
    fn contains_key(&self, value: &Self::KeyType) -> bool;
}

/// Collections that can resize self
pub trait Resizable: CollectionTrait + ExactSized {
    /// Modifies `self` in-place so that len() is equal to new_len,
    /// either by removing excess elements from the back or by appending clones of `value` to the back.
    fn resize(&mut self, new_len: Self::SizeType, value: Self::ElemType);

    /// Modifies `self` in-place so that len() is equal to new_len,
    /// either by removing excess elements from the back or by appending elements generated by calling generator to the back.
    fn resize_with(&mut self, new_len: Self::SizeType, generator: impl FnMut() -> Self::ElemType);

    /// Shortens `self`, keeping the first `len` elements and dropping the rest.
    fn truncate(&mut self, len: Self::SizeType);
}

mod impls {
    use super::*;

    macro_rules! misc_impls {
        () => {};
        ([@EmptyCtor $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> EmptyConstructible for $t {
                fn new() -> Self {
                    <$t>::new()
                }
            }
            misc_impls!($($tail)*);
        };
        ([@Erase $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> Erasable for $t {
                fn clear(&mut self) {
                    <$t>::clear(self)
                }
            }
            misc_impls!($($tail)*);
        };
        ([@Contain $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> Contains for $t {
                fn contains(&self, value: &T) -> bool {
                    <$t>::contains(self, value)
                }
            }
            misc_impls!($($tail)*);
        };
        ([@MapContain $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> AssociatedContains for $t {
                fn contains_key(&self, value: &K) -> bool{
                    <$t>::contains_key(self, value)
                }
            }
            misc_impls!($($tail)*);
        };
        ([@Retain $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> Retainable for $t {
                fn retain<F: FnMut(&T) -> bool>(&mut self, pred: F) {
                    <$t>::retain(self, pred)
                }
            }
            misc_impls!($($tail)*);
        };
        ([@MapRetain $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> AssociatedRetainable for $t {
                fn retain<F: FnMut(&K, &mut V) -> bool>(&mut self, pred: F) {
                    <$t>::retain(self, pred)
                }
            }
            misc_impls!($($tail)*);
        };
        ([@Resize $($args: ident $(: $bound: path $(| $others:path )*)?),* => $t: ty]; $($tail:tt)*) => {
            impl<$($args $(: $bound $(+ $others)*)?),*> Resizable for $t {
                fn resize(&mut self, new_len: Self::SizeType, value: T) {
                    <$t>::resize(self, new_len, value)
                }

                fn resize_with(&mut self, new_len: Self::SizeType, generator: impl FnMut() -> T) {
                    <$t>::resize_with(self, new_len, generator)
                }

                fn truncate(&mut self, len: Self::SizeType) {
                    <$t>::truncate(self, len)
                }
            }
            misc_impls!($($tail)*);
        };
    }

    /*
    // FIXME: [T; N] doesn't implement Default when T: Default
    impl<T: std::default::Default, const N: usize> EmptyConstructible for [T; N] {
        fn new() -> Self {
            std::default::Default::default()
        }
    }
     */

    impl<T: PartialEq<T>> Contains for &[T] {
        fn contains(&self, value: &T) -> bool {
            <[T]>::contains(self, value)
        }
    }

    impl<T: PartialEq<T>> Contains for &mut [T] {
        fn contains(&self, value: &T) -> bool {
            <[T]>::contains(self, value)
        }
    }

    impl<T: PartialEq<T>, const N: usize> Contains for [T; N] {
        fn contains(&self, value: &T) -> bool {
            self.as_slice().contains(value)
        }
    }

    impl<T: PartialEq<T>> Contains for Vec<T> {
        fn contains(&self, value: &T) -> bool {
            self.as_slice().contains(value)
        }
    }

    impl Erasable for &str {}
    impl<T> Erasable for &[T] {}
    impl<T> Erasable for &mut [T] {}
    impl<T, const N: usize> Erasable for [T; N] {}

    impl EmptyConstructible for &str {}
    impl<T> EmptyConstructible for &[T] {}
    impl<T> EmptyConstructible for &mut [T] {}
    impl<T, const N: usize> EmptyConstructible for [T; N] {}

    misc_impls!(
        [@EmptyCtor T => Vec<T>];
        [@EmptyCtor T => std::collections::VecDeque<T>];
        [@EmptyCtor T: std::cmp::Ord => std::collections::BinaryHeap<T>];
        [@EmptyCtor T => std::collections::HashSet<T>];
        [@EmptyCtor K, V => std::collections::HashMap<K, V>];
        [@EmptyCtor T => std::collections::LinkedList<T>];
        [@EmptyCtor T: std::cmp::Ord => std::collections::BTreeSet<T>];
        [@EmptyCtor K: std::cmp::Ord, V => std::collections::BTreeMap<K, V>];

        [@Erase T, A: std::alloc::Allocator => Vec<T, A>];
        [@Erase T => std::collections::VecDeque<T>];
        [@Erase T: std::cmp::Ord => std::collections::BinaryHeap<T>];
        [@Erase T, S => std::collections::HashSet<T, S>];
        [@Erase K, V, S => std::collections::HashMap<K, V, S>];
        [@Erase T => std::collections::LinkedList<T>];
        [@Erase T: std::cmp::Ord => std::collections::BTreeSet<T>];
        [@Erase K: std::cmp::Ord, V => std::collections::BTreeMap<K, V>];

        [@Contain T: std::cmp::PartialEq<T> => std::collections::VecDeque<T>];
        [@Contain T: std::cmp::PartialEq<T> => std::collections::LinkedList<T>];
        [@Contain T: std::hash::Hash | std::cmp::Eq, S: std::hash::BuildHasher => std::collections::HashSet<T, S>];
        [@Contain T: std::cmp::Ord => std::collections::BTreeSet<T>];
        [@MapContain K: std::hash::Hash | std::cmp::Eq, V, S: std::hash::BuildHasher => std::collections::HashMap<K, V, S>];
        [@MapContain K: std::cmp::Ord, V => std::collections::BTreeMap<K, V>];

        [@Retain T, A: std::alloc::Allocator => Vec<T, A>];
        [@Retain T: std::cmp::PartialEq<T> => std::collections::VecDeque<T>];
        [@Retain T: std::hash::Hash | std::cmp::Eq, S: std::hash::BuildHasher => std::collections::HashSet<T, S>];
        [@Retain T: std::cmp::Ord => std::collections::BTreeSet<T>];
        [@MapRetain K: std::hash::Hash | std::cmp::Eq, V, S: std::hash::BuildHasher => std::collections::HashMap<K, V, S>];
        [@MapRetain K: std::cmp::Ord, V => std::collections::BTreeMap<K, V>];

        [@Resize T: Clone, A: std::alloc::Allocator => Vec<T, A>];
        [@Resize T: Clone => std::collections::VecDeque<T>];
    );
}
