//! Implementations of [StaticReflect] for core types (for `#![no_std]`)
use crate::StaticReflect;
use crate::types::{TypeInfo};
use std::mem::{self, ManuallyDrop};
use core::ptr::NonNull;

macro_rules! impl_primitive {
    ($target:ty => $info:expr) => {
        unsafe impl StaticReflect for $target {
            const TYPE_INFO: TypeInfo<'static> = $info;
        }
    }
}
macro_rules! impl_ints {
    ($($target:ty),*) => {
        $(unsafe impl StaticReflect for $target {
            #[allow(unused_comparisons)]
            const TYPE_INFO: TypeInfo<'static> = {
                let size = std::mem::size_of::<$target>();
                let signed = <$target>::MIN < 0;
                TypeInfo::integer(size, signed)
            };
        })*
    }
}
// NOTE: Pointer sized integers have machine-dependent implementation :(
impl_ints!(u8, u16, u32, u64, i8, i16, i32, i64, usize, isize);

#[cfg(feature = "builtins")]
impl_primitive!(str => TypeInfo::Str);
impl_primitive!(() => TypeInfo::Unit);
impl_primitive!(bool => TypeInfo::Bool);
impl_primitive!(f32 => TypeInfo::F32);
impl_primitive!(f64 => TypeInfo::F64);

// Builtin support for the never type
impl_primitive!(! => TypeInfo::Never);


/// Support [StaticReflect] for [ManuallyDrop] by just representing the inner type
unsafe impl<T: StaticReflect> StaticReflect for ManuallyDrop<T> {
    const TYPE_INFO: TypeInfo<'static> = {
        assert!(mem::size_of::<Self>() == mem::size_of::<T>());
        T::TYPE_INFO
    };
}

/// A pointer
///
/// NOTE: The pointed-to value can be anything,
/// even if it doesn't implement [StaticReflect].
///
/// This is fine since the static reflection system
/// doesn't maintain
/// information about pointers (to avoid cycles).
unsafe impl <T> StaticReflect for *mut T {
    const TYPE_INFO: TypeInfo<'static> = TypeInfo::Pointer;
}
/// An immutable pointer
///
/// The static reflection system makes no distinction between
/// mutable and immutable pointers.
unsafe impl <T> StaticReflect for *const T {
    const TYPE_INFO: TypeInfo<'static> = TypeInfo::Pointer;
}

/// A non-zero pointer type, where optional types
/// are guaranteed to use the nullable representation
///
/// If `T: SimpleNonZeroPointer` -> `sizeof(Option<T>) == sizeof(T) && repr(Option<T>) == repr(T)`
pub unsafe trait SimpleNonZeroPointer: StaticReflect {}

unsafe impl <T> SimpleNonZeroPointer for NonNull<T> {}
unsafe impl <T> StaticReflect for NonNull<T> {
    const TYPE_INFO: TypeInfo<'static> = TypeInfo::Pointer;
}

unsafe impl <T: SimpleNonZeroPointer> StaticReflect for Option<T> {
    const TYPE_INFO: TypeInfo<'static> = TypeInfo::Pointer;
}