//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::borrow::Cow;
use std::collections::HashMap;
use std::convert::TryInto;

use anyhow::Context;
use k8s_openapi::api::core::v1::{ConfigMap, Namespace, Node, Pod, Secret};
use kube::api::{self, Api};
use kube::config::{Config, KubeConfigOptions, Kubeconfig};
use kube::{Client, Resource, ResourceExt};
use serde_json as json;

use crate::v0;
use crate::Location;

pub(crate) use helm::Helm;
use helper::{group_nodes_by_region, group_nodes_by_zone, is_aks, is_eks};
use kubeconfig::KubeconfigExt;

mod helm;
mod helper;
mod kubeconfig;
mod secret;
mod show;

pub(crate) const KUBE_SYSTEM_NS: &str = "kube-system";
const STATEHUB_CLUSTER_TOKEN_SECRET_TYPE: &str = "statehub.io/cluster-token";
const STATEHUB_CLUSTER_TOKEN_SECRET_NAME: &str = "statehub-cluster-token";
const STATEHUB_CLOUD_CREDS_SECRET_TYPE: &str = "statehub.io/cloud-credential";
const STATEHUB_AZURE_CREDS_SECRET_NAME: &str = "statehub-azure-creds";
const STATEHUB_AWS_CREDS_SECRET_NAME: &str = "statehub-aws-creds";
const STATEHUB_CLUSTER_CONFIGMAP_NAME: &str = "statehub";

#[derive(Debug)]
pub(crate) struct Kubectl {
    kubeconfig: Kubeconfig,
    context: String,
}

impl Kubectl {
    pub(crate) fn new() -> anyhow::Result<Self> {
        let kubeconfig = Kubeconfig::read()?;
        let context = kubeconfig
            .current_context()
            .ok_or_else(|| anyhow::anyhow!("No current context in kubeconfig"))?
            .to_string();
        Ok(Self {
            kubeconfig,
            context,
        })
    }

    pub(crate) fn with_context(context: &str) -> anyhow::Result<Self> {
        let kubeconfig = Kubeconfig::read()?;
        let context = context.to_string();

        Ok(Self {
            kubeconfig,
            context,
        })
    }

    pub(crate) fn clone_for_context(&self, context: &str) -> Self {
        let kubeconfig = self.kubeconfig.clone();
        let context = context.to_string();
        Self {
            kubeconfig,
            context,
        }
    }

    pub(crate) fn has_context(&self, name: &str) -> bool {
        self.kubeconfig
            .contexts
            .iter()
            .any(|context| context.name == name)
    }

    pub(crate) fn _default_context(&self) -> Option<&str> {
        None
    }

    pub(crate) fn cluster_name(&self) -> v0::ClusterName {
        normalize_name(&self.context)
    }

    async fn get_client(&self) -> anyhow::Result<Client> {
        let context = Some(self.context.clone());
        let options = KubeConfigOptions {
            context,
            ..KubeConfigOptions::default()
        };
        let kubeconfig = self.kubeconfig.clone();
        Config::from_custom_kubeconfig(kubeconfig, &options)
            .await
            .context("Loading context from kubeconfig")?
            .try_into()
            .context("Connecting to Kubernetes cluster")
    }

    async fn all_namespaces(&self) -> anyhow::Result<impl IntoIterator<Item = Namespace>> {
        let namespaces = self.api().await?;
        let lp = self.list_params();
        Ok(namespaces.list(&lp).await?)
    }

    async fn all_nodes(&self) -> anyhow::Result<impl IntoIterator<Item = Node>> {
        let nodes = self.api().await?;
        let lp = self.list_params();
        Ok(nodes.list(&lp).await?)
    }

    async fn all_pods(&self, namespace: &str) -> anyhow::Result<impl IntoIterator<Item = Pod>> {
        let pods = self.namespaced_api(namespace).await?;
        let lp = self.list_params();
        Ok(pods.list(&lp).await?)
    }

    async fn create_namespace(&self, namespace: &str) -> anyhow::Result<Namespace> {
        let namespaces = self.api().await?;
        let namespace = json::from_value(json::json!({
            "apiVerion": "v1",
            "kind": "Namespace",
            "metadata": {
                "name": namespace,
            }
        }))?;
        let pp = self.post_params();
        let namespace = namespaces.create(&pp, &namespace).await?;
        Ok(namespace)
    }

    async fn create_configmap(
        &self,
        name: &str,
        namespace: &str,
        cluster_name: &v0::ClusterName,
        default_state: &str,
        api: &str,
        cleanup_grace: &str,
    ) -> anyhow::Result<ConfigMap> {
        let configmaps = self.namespaced_api(namespace).await?;
        let configmap = json::from_value(json::json!({
            "apiVerion": "v1",
            "kind": "ConfigMap",
            "metadata": {
                "name": name,
                "namespace": namespace,
            },
            "data": {
                "cluster-name": cluster_name,
                "default-state": default_state,
                "api-url": api,
                "cleanup-grace": cleanup_grace,
            }
        }))?;
        let pp = self.post_params();
        let configmap = configmaps.create(&pp, &configmap).await?;

        Ok(configmap)
    }

    pub(crate) async fn store_azure_credentials(
        &self,
        namespace: &str,
        azure: json::Value,
    ) -> anyhow::Result<Secret> {
        self.set_secret(
            namespace,
            STATEHUB_CLOUD_CREDS_SECRET_TYPE,
            STATEHUB_AZURE_CREDS_SECRET_NAME,
            azure,
        )
        .await
    }

    pub(crate) async fn store_aws_credentials(
        &self,
        namespace: &str,
        aws: json::Value,
    ) -> anyhow::Result<Secret> {
        self.set_secret(
            namespace,
            STATEHUB_CLOUD_CREDS_SECRET_TYPE,
            STATEHUB_AWS_CREDS_SECRET_NAME,
            aws,
        )
        .await
    }

    async fn delete_configmap(&self, namespace: &str, configmap: &str) -> anyhow::Result<()> {
        log::info!("Deleting configmap {}", configmap);
        let configmaps = self.namespaced_api::<ConfigMap>(namespace).await?;
        let dp = self.delete_params();
        configmaps
            .delete(configmap, &dp)
            .await?
            .map_left(|cm| log::info!("Delete in progress {:#?}", cm))
            .map_right(|cm| log::info!("Delete succeeded: {:#?}", cm));

        Ok(())
    }

    fn delete_params(&self) -> api::DeleteParams {
        api::DeleteParams::default()
    }

    fn list_params(&self) -> api::ListParams {
        api::ListParams::default()
    }

    fn post_params(&self) -> api::PostParams {
        api::PostParams::default()
    }

    async fn api<T>(&self) -> anyhow::Result<Api<T>>
    where
        T: Resource,
        <T as Resource>::DynamicType: Default,
    {
        self.get_client().await.map(Api::all)
    }

    async fn namespaced_api<T>(&self, namespace: &str) -> anyhow::Result<Api<T>>
    where
        T: Resource,
        <T as Resource>::DynamicType: Default,
    {
        self.get_client()
            .await
            .map(|client| Api::namespaced(client, namespace))
    }

    pub(crate) async fn get_regions(
        &self,
        zone: bool,
    ) -> anyhow::Result<HashMap<Option<String>, Vec<String>>> {
        let group_nodes = |nodes| {
            if zone {
                group_nodes_by_zone(nodes)
            } else {
                group_nodes_by_region(nodes)
            }
        };

        self.all_nodes().await.map(group_nodes)
    }

    pub(crate) async fn collect_node_locations(&self) -> anyhow::Result<Vec<Location>> {
        self.get_regions(false)
            .await?
            .into_iter()
            .map(|(region, nodes)| {
                region.ok_or_else(|| {
                    anyhow::anyhow!("Cannot determine location for nodes {:?}", nodes)
                })
            })
            .collect::<Result<Vec<_>, _>>()?
            .into_iter()
            .map(|region| region.parse())
            .collect::<Result<Vec<Location>, _>>()
            .map_err(anyhow::Error::msg)
    }

    pub(crate) async fn store_configmap(
        &self,
        namespace: &str,
        cluster_name: &v0::ClusterName,
        default_state: &str,
        api: &str,
        cleanup_grace: &str,
    ) -> anyhow::Result<ConfigMap> {
        if self
            .delete_configmap(namespace, STATEHUB_CLUSTER_CONFIGMAP_NAME)
            .await
            .is_ok()
        {
            log::trace!("Removing previous configmap");
        }

        self.create_configmap(
            STATEHUB_CLUSTER_CONFIGMAP_NAME,
            namespace,
            cluster_name,
            default_state,
            api,
            cleanup_grace,
        )
        .await
    }

    // TODO: implement provider fetch out of cluster
    pub(crate) async fn get_cluster_provider(&self) -> anyhow::Result<v0::Provider> {
        let nodes = self.all_nodes().await?;
        if is_aks(nodes) {
            Ok(v0::Provider::Aks)
        } else if is_eks() {
            Ok(v0::Provider::Eks)
        } else {
            Ok(v0::Provider::Generic)
        }
    }
}

pub(crate) async fn validate_namespace(namespace: &str) -> anyhow::Result<Namespace> {
    let kube = Kubectl::new()?;
    let existing = kube
        .all_namespaces()
        .await?
        .into_iter()
        .find(|ns| ns.name() == namespace);

    if let Some(existing) = existing {
        log::info!("Using existing namespace {}", existing.name());
        Ok(existing)
    } else {
        log::info!("Creating new namespace {}", namespace);
        kube.create_namespace(namespace).await
    }
}

pub(crate) async fn list_namespaces() -> anyhow::Result<impl IntoIterator<Item = Namespace>> {
    Kubectl::new()?.all_namespaces().await
}

pub(crate) async fn list_nodes() -> anyhow::Result<impl IntoIterator<Item = Node>> {
    Kubectl::new()?.all_nodes().await
}

pub(crate) async fn list_pods() -> anyhow::Result<impl IntoIterator<Item = Pod>> {
    Kubectl::new()?.all_pods(KUBE_SYSTEM_NS).await
}

pub(crate) async fn store_cluster_token(namespace: &str, token: &str) -> anyhow::Result<Secret> {
    let kubectl = Kubectl::new()?;

    let token = json::json!({ "cluster-token": token });

    kubectl
        .set_secret(
            namespace,
            STATEHUB_CLUSTER_TOKEN_SECRET_TYPE,
            STATEHUB_CLUSTER_TOKEN_SECRET_NAME,
            token,
        )
        .await
}

pub(crate) fn extract_cluster_token(secret: &Secret) -> Option<Cow<'_, str>> {
    secret
        .data
        .get("cluster-token")
        .map(|bytes| bytes.0.as_slice())
        .map(String::from_utf8_lossy)
}

pub(crate) fn get_current_cluster_name() -> Option<v0::ClusterName> {
    Kubeconfig::read().ok()?.current_cluster_name()
}

const CLUSTER_DELIMITER: char = '/';

pub(super) fn normalize_name(name: &str) -> v0::ClusterName {
    if let Some((_, tail)) = name.rsplit_once(CLUSTER_DELIMITER) {
        tail.into()
    } else {
        name.into()
    }
}
