//
// Copyright (c) 2022 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::fmt;

use async_stream::stream;
use console::Emoji;
use futures::stream::Stream;

const CHECK: Emoji<'static, 'static> = Emoji("\u{2705}", "√");
const CROSS: Emoji<'static, 'static> = Emoji("\u{274c}", "x");

use super::*;

trait Diagnose {
    fn diagnose(&self, text: &str) -> Result<String, String>;
}

impl<T: fmt::Display> Diagnose for anyhow::Result<T> {
    fn diagnose(&self, label: &str) -> Result<String, String> {
        self.as_ref()
            .map(|text| format!("{:<60} - {} ({})", label, CHECK, text))
            .map_err(|err| format!("{:<60} - {} ({})", label, CROSS, err))
    }
}

impl<T: fmt::Display> Diagnose for kube::Result<T> {
    fn diagnose(&self, label: &str) -> Result<String, String> {
        self.as_ref()
            .map(|text| format!("{:<60} - {} ({})", label, CHECK, text))
            .map_err(|err| format!("{:<60} - {} ({})", label, CROSS, err))
    }
}

impl Kubectl {
    pub(crate) async fn check_cluster(&self) -> impl Stream<Item = Result<String, String>> + '_ {
        let config = self.get_config().await.unwrap();
        let statehub_kubectl =
            statehub_cluster_operator::cluster::Operator::with_config(config).unwrap();

        stream!(
            let crd = self
                .check_statehub_cluster_crd()
                .await
                .diagnose("StatehubCluster CRD is installed correctly");
            yield crd;

            let object = self
                .check_statehub_cluster_object()
                .await
                .map(|cluster| cluster.name())
                .diagnose("StatehubCluster Object is present");
            yield object;

            if let Ok(cluster) = self.check_statehub_cluster_object().await {
                let namespace = statehub_kubectl
                    .check_namespace(&cluster)
                    .await
                    .map(|namespace| namespace.finalizers().join(","))
                    .diagnose("Namespace exists");
                yield namespace;
            }
        )
    }
}
