//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use k8s_openapi::apiextensions_apiserver::pkg::apis::apiextensions::v1::CustomResourceDefinition;
use kube::CustomResourceExt;
use kube::{Resource, ResourceExt};
use statehub_cluster_operator::{ClusterSpec, StatehubCluster};

use super::*;

impl Kubectl {
    pub(super) async fn create_statehub_cluster_crd(
        &self,
    ) -> anyhow::Result<CustomResourceDefinition> {
        let crd = StatehubCluster::crd();
        self.create_crd_impl(crd).await
    }

    pub(super) async fn create_statehub_cluster(
        &self,
        namespace: &str,
    ) -> anyhow::Result<StatehubCluster> {
        let namespace = namespace.to_string();
        let name = self.cluster_name();
        let spec = ClusterSpec { namespace };
        let mut cluster = StatehubCluster::new(name, spec);
        let clusters = self.api::<StatehubCluster>().await?;
        if let Ok(existing) = clusters.get(name).await {
            cluster.meta_mut().resource_version = existing.resource_version();
        }

        let pp = self.post_params();

        let cluster = if cluster.resource_version().is_some() {
            clusters.replace(name, &pp, &cluster).await?
        } else {
            clusters.create(&pp, &cluster).await?
        };
        Ok(cluster)
    }

    pub(super) async fn delete_statehub_cluster(&self, name: &str) -> anyhow::Result<()> {
        let clusters = self.api::<StatehubCluster>().await?;
        let dp = self.delete_params();
        clusters.delete(name, &dp).await?;
        Ok(())
    }

    pub(super) async fn delete_statehub_cluster_crd(&self) -> anyhow::Result<()> {
        let crd = StatehubCluster::crd();
        let name = crd.name();
        let crds = self.api::<CustomResourceDefinition>().await?;
        let dp = self.delete_params();
        crds.delete(&name, &dp).await?;
        Ok(())
    }

    pub(super) async fn check_statehub_cluster_crd(&self) -> anyhow::Result<String> {
        let name = StatehubCluster::crd().name();
        let version = StatehubCluster::api_resource().version;
        self.api::<CustomResourceDefinition>()
            .await?
            .get(&name)
            .await
            .context("StatehubCluster CRD is not installed")
            .map(|crd| format!("{}/{}", crd.name(), version))
    }

    pub(super) async fn check_statehub_cluster_object(&self) -> anyhow::Result<StatehubCluster> {
        let clusters = self.api::<StatehubCluster>().await?;
        let cluster_name = self.cluster_name();
        clusters.get(cluster_name).await.context(format!(
            "StatehubCluster({}) object not found",
            cluster_name
        ))
    }

    async fn create_crd_impl(
        &self,
        mut crd: CustomResourceDefinition,
    ) -> anyhow::Result<CustomResourceDefinition> {
        let name = crd.name();
        let crds = self.api::<CustomResourceDefinition>().await?;
        if let Ok(existing) = crds.get(&name).await {
            crd.meta_mut().resource_version = existing.resource_version();
        }

        let pp = self.post_params();

        let crd = if crd.resource_version().is_some() {
            crds.replace(&name, &pp, &crd).await?
        } else {
            crds.create(&pp, &crd).await?
        };
        Ok(crd)
    }

    async fn _delete_crd_impl(&self, name: &str) -> anyhow::Result<()> {
        let crds = self.api::<CustomResourceDefinition>().await?;
        let dp = self.delete_params();
        crds.delete(name, &dp).await?;
        Ok(())
    }
}
