//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::fmt;

use async_stream::try_stream;
use console::Emoji;
use futures::stream::Stream;

const CHECK: Emoji<'static, 'static> = Emoji("\u{2705}", "√");
const CROSS: Emoji<'static, 'static> = Emoji("\u{274c}", "x");

use super::*;

trait Diagnose {
    fn diagnose(&self, text: &str) -> Result<String, String>;
}

impl<T: fmt::Display> Diagnose for anyhow::Result<T> {
    fn diagnose(&self, label: &str) -> Result<String, String> {
        self.as_ref()
            .map(|text| format!("{:<60} - {} ({})", label, CHECK, text))
            .map_err(|err| format!("{:<60} - {} ({})", label, CROSS, err))
    }
}

impl Kubectl {
    pub(crate) fn check_cluster(&self) -> impl Stream<Item = Result<String, String>> + '_ {
        try_stream!(
            let crd = self
                .check_statehub_cluster_crd()
                .await
                .diagnose("StatehubCluster CRD is installed correcly")?;
            yield crd;

            let cluster = self.check_statehub_cluster_object().await;
            let namespace = cluster.as_ref().ok().map(|cluster| cluster.spec.namespace.clone());
            let name = cluster.map(|cluster| format!("{} [namespace: {}]", cluster.name(), cluster.spec.namespace))
                .diagnose("StatehubCluster Object is present")?;
            yield name;

            // Should be safe to unwrap, as we only get here if cluster is Ok
            // and hence namespace is Some
            let namespace = namespace.unwrap();
            let namespace = Ok(namespace).diagnose("namespace")?;
            yield namespace;

        )
    }
}
