//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::fmt;
use std::str;

use serde::{Deserialize, Serialize};
use thiserror::Error;

use crate::aws;
use crate::azure;

pub trait CloudLocation {
    const VENDOR: &'static str;
    const VENDOR_PREFIX: &'static str;
    fn as_str(&self) -> &'static str;
}

#[derive(Clone, Copy, Debug, PartialEq, Eq, PartialOrd, Ord, Hash, Serialize, Deserialize)]
#[cfg_attr(feature = "jsonschema", derive(schemars::JsonSchema))]
pub enum Location {
    Aws(aws::AwsRegion),
    Azure(azure::AzureRegion),
}

impl Location {
    pub fn is_aws(&self) -> bool {
        matches!(self, Location::Aws(_))
    }

    pub fn is_azure(&self) -> bool {
        matches!(self, Location::Azure(_))
    }

    pub fn region(&self) -> &str {
        match self {
            Location::Aws(region) => region.as_str(),
            Location::Azure(region) => region.as_str(),
            // Location::Gcp(region) => region.as_str(),
        }
    }
}

impl fmt::Display for Location {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Aws(region) => region.fmt(f),
            Self::Azure(region) => region.fmt(f),
        }
    }
}

impl From<aws::AwsRegion> for Location {
    fn from(region: aws::AwsRegion) -> Self {
        Self::Aws(region)
    }
}

impl From<azure::AzureRegion> for Location {
    fn from(region: azure::AzureRegion) -> Self {
        Self::Azure(region)
    }
}

impl str::FromStr for Location {
    type Err = ParseError;

    fn from_str(text: &str) -> Result<Self, Self::Err> {
        let aws = text.parse::<aws::AwsRegion>();
        let azure = text.parse::<azure::AzureRegion>();
        // let gcp = text.parse::<v1::GcpRegion>();

        match (aws, azure) {
            (Ok(aws), Err(_)) => Ok(Self::Aws(aws)),
            (Err(_), Ok(azure)) => Ok(Self::Azure(azure)),
            (Ok(aws), Ok(azure)) => Err(ParseError::ambiguous(aws, azure)),
            (Err(aws), Err(azure)) => Err(ParseError::unknown(aws, azure)),
        }
    }
}

#[derive(Error, Debug)]
pub enum ParseError {
    #[error("Ambiguous region, use either {0}")]
    AmbiguousLocation(String),
    #[error("Unknown region: {0}")]
    UnknownLocation(String),
}

impl ParseError {
    fn ambiguous(aws: aws::AwsRegion, azure: azure::AzureRegion) -> Self {
        Self::AmbiguousLocation(format!("{:#} or {:#}", aws, azure))
    }

    fn unknown(aws: InvalidLocation, azure: InvalidLocation) -> Self {
        Self::UnknownLocation(format!("{} or {}", aws, azure))
    }
}

#[derive(Debug, Error)]
#[error(r#"Invalid {vendor} region "{region}""#)]
pub struct InvalidLocation {
    vendor: String,
    region: String,
}

impl InvalidLocation {
    pub fn new(vendor: &str, region: &str) -> Self {
        let vendor = vendor.to_string();
        let region = region.to_string();
        Self { vendor, region }
    }
}
