//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::fmt;
use std::str;

use schemars::JsonSchema;
use serde::{Deserialize, Serialize};
use serde_with::{DeserializeFromStr, SerializeDisplay};
use thiserror::Error;

pub mod aws;
pub mod azure;

pub trait CloudLocation {
    const VENDOR: &'static str;
    const VENDOR_PREFIX: &'static str;
    fn as_str(&self) -> &'static str;
}

#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq, JsonSchema, Serialize, Deserialize)]
pub enum Location {
    Aws(aws::AwsRegion),
    Azure(azure::AzureRegion),
    // Gcp(v1::GcpRegion),
}

impl Location {
    pub fn is_aws(&self) -> bool {
        matches!(self, Location::Aws(_))
    }

    pub fn is_azure(&self) -> bool {
        matches!(self, Location::Azure(_))
    }
}

impl fmt::Display for Location {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Aws(region) => region.fmt(f),
            Self::Azure(region) => region.fmt(f),
        }
    }
}

impl From<aws::AwsRegion> for Location {
    fn from(region: aws::AwsRegion) -> Self {
        Self::Aws(region)
    }
}

impl From<azure::AzureRegion> for Location {
    fn from(region: azure::AzureRegion) -> Self {
        Self::Azure(region)
    }
}

impl str::FromStr for Location {
    type Err = String;

    fn from_str(text: &str) -> Result<Self, Self::Err> {
        let aws = text.parse::<aws::AwsRegion>();
        let azure = text.parse::<azure::AzureRegion>();
        // let gcp = text.parse::<v1::GcpRegion>();

        match (aws, azure) {
            (Ok(aws), Err(_)) => Ok(Self::Aws(aws)),
            (Err(_), Ok(azure)) => Ok(Self::Azure(azure)),
            (Ok(aws), Ok(azure)) => Err(format!(
                "Ambiguous region, use either {:#} or {:#}",
                aws, azure
            )),
            (Err(aws), Err(azure)) => {
                let error = format!("{} or {}", aws, azure);
                Err(error)
            }
        }
    }
}

#[derive(Debug, Error)]
#[error(r#"Invalid {vendor} region "{region}""#)]
pub struct InvalidLocation {
    vendor: String,
    region: String,
}

impl InvalidLocation {
    pub fn new(vendor: &str, region: &str) -> Self {
        let vendor = vendor.to_string();
        let region = region.to_string();
        Self { vendor, region }
    }
}
