//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::time::Duration;

use serde_json as json;

use super::*;
use crate::ext::ResourceExt as _;

impl super::Kubectl {
    pub fn delete_params(&self) -> api::DeleteParams {
        api::DeleteParams {
            grace_period_seconds: Some(0),
            ..api::DeleteParams::default()
        }
    }

    pub fn list_params(&self) -> api::ListParams {
        api::ListParams::default()
    }

    pub fn post_params(&self) -> api::PostParams {
        api::PostParams::default()
    }

    pub fn post_with_manager(&self, manager: &str) -> api::PostParams {
        let mut pp = self.post_params();
        pp.field_manager = Some(manager.to_string());
        pp
    }

    pub fn patch_params(&self) -> api::PatchParams {
        api::PatchParams::default()
    }

    pub fn patch_with_manager(&self, manager: &str) -> api::PatchParams {
        api::PatchParams::apply(manager)
    }

    pub fn log_params(&self) -> api::LogParams {
        api::LogParams::default()
    }

    pub fn log_params_previous(&self) -> api::LogParams {
        api::LogParams {
            previous: true,
            ..api::LogParams::default()
        }
    }

    pub async fn patch<K>(&self, object: K, api: kube::Api<K>) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let patch = api::Patch::Apply(&object);
        let pp = self.patch_with_manager(&self.manager);
        api.patch(&name, &pp, &patch).await
    }

    pub async fn force_patch<K>(&self, object: K, api: kube::Api<K>) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let patch = api::Patch::Apply(&object);
        let pp = self.patch_with_manager(&self.manager).force();
        api.patch(&name, &pp, &patch).await
    }

    pub fn api<T>(&self) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client();
        api::Api::<T>::all(client)
    }

    pub fn default_namespaced_api<T>(&self) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client();
        api::Api::<T>::default_namespaced(client)
    }

    pub fn namespaced_api<T>(&self, namespace: &str) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client();
        api::Api::<T>::namespaced(client, namespace)
    }

    pub fn configmaps<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> api::Api<corev1::ConfigMap> {
        if let Some(namespace) = namespace.into() {
            self.namespaced_api::<corev1::ConfigMap>(namespace)
        } else {
            self.default_namespaced_api::<corev1::ConfigMap>()
        }
    }

    pub fn daemonsets<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> api::Api<appsv1::DaemonSet> {
        if let Some(namespace) = namespace.into() {
            self.namespaced_api::<appsv1::DaemonSet>(namespace)
        } else {
            self.default_namespaced_api::<appsv1::DaemonSet>()
        }
    }

    pub fn deployments<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> api::Api<appsv1::Deployment> {
        if let Some(namespace) = namespace.into() {
            self.namespaced_api::<appsv1::Deployment>(namespace)
        } else {
            self.default_namespaced_api::<appsv1::Deployment>()
        }
    }

    pub fn nodes(&self) -> api::Api<corev1::Node> {
        self.api::<corev1::Node>()
    }

    pub fn crds(&self) -> api::Api<apiextensionsv1::CustomResourceDefinition> {
        self.api::<apiextensionsv1::CustomResourceDefinition>()
    }

    pub fn namespaces(&self) -> api::Api<corev1::Namespace> {
        self.api::<corev1::Namespace>()
    }

    pub fn secrets<'a>(&self, namespace: impl Into<Option<&'a str>>) -> api::Api<corev1::Secret> {
        if let Some(namespace) = namespace.into() {
            self.namespaced_api::<corev1::Secret>(namespace)
        } else {
            self.default_namespaced_api::<corev1::Secret>()
        }
    }

    pub fn pods<'a>(&self, namespace: impl Into<Option<&'a str>>) -> api::Api<corev1::Pod> {
        if let Some(namespace) = namespace.into() {
            self.namespaced_api::<corev1::Pod>(namespace)
        } else {
            self.default_namespaced_api::<corev1::Pod>()
        }
    }

    pub async fn get_deployment<'a>(
        &self,
        name: &str,
        namespace: impl Into<Option<&'a str>>,
    ) -> kube::Result<appsv1::Deployment> {
        self.deployments(namespace).get(name).await
    }

    pub async fn get_namespace(&self, name: &str) -> kube::Result<corev1::Namespace> {
        self.namespaces().get(name).await
    }

    pub async fn get_secret(&self, name: &str) -> kube::Result<corev1::Secret> {
        self.secrets(None).get(name).await
    }

    pub async fn put_secret(
        &self,
        name: &str,
        r#type: &str,
        data: BTreeMap<String, ByteString>,
    ) -> kube::Result<corev1::Secret> {
        let secret: corev1::Secret = json::from_value(json::json!(
            {
                "metadata": {
                    "name": name
                },
                "data": data,
                "type": r#type
            }
        ))
        .expect("Failed to create corev1::Secret");

        self.ensure_default_namespaced_k_is_installed(secret).await
    }

    pub async fn get_config_map(&self, name: &str) -> kube::Result<corev1::ConfigMap> {
        self.configmaps(None).get(name).await
    }

    pub async fn put_config_map(
        &self,
        name: &str,
        data: BTreeMap<String, String>,
    ) -> kube::Result<corev1::ConfigMap> {
        const NOT_FOUND: http::StatusCode = http::StatusCode::NOT_FOUND;

        let pp = self.post_params();
        let configmaps = self.configmaps(None);
        let mut cm = corev1::ConfigMap::new(name).data(data);

        match configmaps.get(name).await {
            Ok(existing) => {
                if let Some(version) = existing.resource_version() {
                    cm = cm.with_resource_version(version);
                }
                configmaps.replace(name, &pp, &cm).await
            }
            Err(kube::Error::Api(e)) if e.code == NOT_FOUND.as_u16() => {
                configmaps.create(&pp, &cm).await
            }
            Err(e) => Err(e),
        }
    }

    pub async fn update_config_map(
        &self,
        name: &str,
        mut update_fn: impl FnMut(BTreeMap<String, String>) -> BTreeMap<String, String>,
    ) -> kube::Result<corev1::ConfigMap> {
        const NOT_FOUND: http::StatusCode = http::StatusCode::NOT_FOUND;
        const CONFLICT: http::StatusCode = http::StatusCode::CONFLICT;
        const INTERNAL_SERVER_ERROR: http::StatusCode = http::StatusCode::INTERNAL_SERVER_ERROR;
        const RETRY_COUNT: u8 = 16;

        let pp = self.post_params();
        let configmaps = self.configmaps(None);

        let mut last_err = None;
        for retry in 0..RETRY_COUNT {
            match configmaps.get(name).await {
                Ok(mut existing) => {
                    let data = std::mem::take(&mut existing.data).unwrap_or_default();
                    let data = update_fn(data);
                    let cm = if let Some(resource_version) = existing.resource_version() {
                        corev1::ConfigMap::new(name)
                            .data(data)
                            .with_resource_version(resource_version)
                    } else {
                        continue;
                    };

                    match configmaps.replace(name, &pp, &cm).await {
                        // if we got `CONFLICT` then either uid or resourceVersion do not match
                        // sleep with a backoff and retry
                        Err(kube::Error::Api(e)) if e.code == CONFLICT.as_u16() => {
                            last_err = Some(e);
                        }
                        result => return result,
                    }
                }
                Err(kube::Error::Api(e)) if e.code == NOT_FOUND.as_u16() => {
                    let data = update_fn(BTreeMap::new());
                    let cm = corev1::ConfigMap::new(name).data(data);
                    return configmaps.create(&pp, &cm).await;
                }
                Err(e) => return Err(e),
            }
            tokio::time::sleep(Duration::from_millis((1 << retry) * 10)).await
        }

        let e = last_err.unwrap_or_else(|| kube::error::ErrorResponse {
            code: INTERNAL_SERVER_ERROR.as_u16(),
            status: INTERNAL_SERVER_ERROR.as_str().to_string(),
            message: String::new(),
            reason: String::new(),
        });
        Err(kube::Error::Api(e))
    }

    pub async fn list_daemonsets<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> kube::Result<Vec<appsv1::DaemonSet>> {
        let lp = self.list_params();
        self.daemonsets(namespace)
            .list(&lp)
            .await
            .map(|object| object.items)
    }

    pub async fn list_deployments<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> kube::Result<Vec<appsv1::Deployment>> {
        let lp = self.list_params();
        self.deployments(namespace)
            .list(&lp)
            .await
            .map(|object| object.items)
    }

    pub async fn list_pods<'a>(
        &self,
        namespace: impl Into<Option<&'a str>>,
    ) -> kube::Result<Vec<corev1::Pod>> {
        let lp = self.list_params();
        self.pods(namespace)
            .list(&lp)
            .await
            .map(|object| object.items)
    }

    pub async fn list_crds(&self) -> kube::Result<Vec<apiextensionsv1::CustomResourceDefinition>> {
        let lp = self.list_params();
        self.crds().list(&lp).await.map(|object| object.items)
    }

    pub(super) async fn ensure_global_k_is_installed_impl<K>(
        &self,
        mut object: K,
        force: bool,
    ) -> kube::Result<K>
    where
        K: kube::ResourceExt + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.api::<K>();
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        if force {
            self.force_patch(object, objects).await
        } else {
            self.patch(object, objects).await
        }
    }

    pub(crate) async fn ensure_namespaced_k_is_installed_impl<K>(
        &self,
        mut object: K,
        namespace: &str,
        force: bool,
    ) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.namespaced_api::<K>(namespace);
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        if force {
            self.force_patch(object, objects).await
        } else {
            self.patch(object, objects).await
        }
    }

    pub(crate) async fn ensure_default_namespaced_k_is_installed_impl<K>(
        &self,
        mut object: K,
        force: bool,
    ) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.default_namespaced_api::<K>();
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        if force {
            self.force_patch(object, objects).await
        } else {
            self.patch(object, objects).await
        }
    }
}
