//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::collections::BTreeMap;
use std::fmt;

use k8s_openapi::apimachinery;
use kube::api;
use kube::config;
use kube::{Resource, ResourceExt};
use serde_yaml as yaml;

use super::*;

mod helper;

#[derive(Clone)]
pub struct Kubectl {
    client: kube::Client,
}

impl Kubectl {
    pub async fn try_default() -> kube::Result<Self> {
        kube::Client::try_default()
            .await
            .map(|client| Self { client })
    }

    pub fn client(&self) -> kube::Client {
        self.client.clone()
    }

    pub async fn apiserver_version(&self) -> kube::Result<apimachinery::pkg::version::Info> {
        self.client.apiserver_version().await
    }

    pub async fn load_text_from_secret(&self, name: &str, value: &str) -> kube::Result<String> {
        let text = self
            .get_secret(name)
            .await?
            .data
            .unwrap_or_default()
            .get(value)
            .map(|text| String::from_utf8_lossy(&text.0))
            .unwrap_or_default()
            .to_string();
        Ok(text)
    }

    pub async fn load_kubeconfig_from_secret(&self, secret: &str) -> kube::Result<kube::Config> {
        let text = self.load_text_from_secret(secret, "value").await?;
        load_config_from_text(&text)
            .await
            .map_err(from_kubeconfig_error)
    }
}

impl fmt::Debug for Kubectl {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Kubectl")
            .field("client", &"<kube::Client>")
            .finish()
    }
}

fn kubeconfig(text: &str) -> Result<config::Kubeconfig, config::KubeconfigError> {
    let value: yaml::Value = yaml::from_str(text).map_err(config::KubeconfigError::Parse)?;
    let kubeconfig = yaml::from_value(value).map_err(config::KubeconfigError::InvalidStructure)?;

    Ok(kubeconfig)
}

async fn load_config_from_text(text: &str) -> Result<config::Config, config::KubeconfigError> {
    let kubeconfig = kubeconfig(text)?;
    let options = kube::config::KubeConfigOptions::default();
    config::Config::from_custom_kubeconfig(kubeconfig, &options).await
}

fn from_kubeconfig_error(err: config::KubeconfigError) -> kube::Error {
    // This is a workaround until the real error code will be possible to create
    // let infer = config::InferConfigError {
    //     in_cluster: config::InClusterError::MissingEnvironmentVariables, // Dummy error
    //     kubeconfig: err,
    // };
    // kube::Error::InferConfig(infer)
    log::warn!(
        "Replacing {} with Error::LinesCodecMaxLineLengthExceeded",
        err
    );

    kube::Error::LinesCodecMaxLineLengthExceeded
}
