//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use std::collections::BTreeMap;
use std::fmt;

use k8s_openapi::apimachinery;
use k8s_openapi::ByteString;
use kube::api;
use kube::config;
use kube::{Resource, ResourceExt};
use serde::{de, ser};

use super::*;

mod capi;
mod helper;

#[derive(Clone)]
pub struct Kubectl {
    client: kube::Client,
}

impl Kubectl {
    pub async fn try_default() -> kube::Result<Self> {
        kube::Client::try_default()
            .await
            .map(|client| Self { client })
    }

    pub fn from_config(config: kube::Config) -> kube::Result<Self> {
        kube::Client::try_from(config).map(|client| Self { client })
    }

    pub fn client(&self) -> kube::Client {
        self.client.clone()
    }

    pub async fn apiserver_version(&self) -> kube::Result<apimachinery::pkg::version::Info> {
        self.client.apiserver_version().await
    }

    pub async fn load_text_from_secret(&self, name: &str, value: &str) -> kube::Result<String> {
        let text = self
            .get_secret(name)
            .await?
            .data
            .unwrap_or_default()
            .get(value)
            .map(|text| String::from_utf8_lossy(&text.0))
            .unwrap_or_default()
            .to_string();
        Ok(text)
    }

    pub async fn load_config_from_secret(&self, secret: &str) -> kube::Result<kube::Config> {
        let text = self.load_text_from_secret(secret, "value").await?;
        load_config_from_text(&text)
            .await
            .map_err(from_kubeconfig_error)
    }

    pub async fn ensure_global_k_is_installed<K>(&self, mut object: K) -> kube::Result<K>
    where
        K: kube::ResourceExt + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.api::<K>();
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        let pp = self.post_params();

        let object = if object.resource_version().is_some() {
            objects.replace(&name, &pp, &object).await?
        } else {
            objects.create(&pp, &object).await?
        };

        Ok(object)
    }

    pub async fn ensure_namespaced_k_is_installed<K>(
        &self,
        mut object: K,
        namespace: &str,
    ) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.namespaced_api::<K>(namespace);
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        let pp = self.post_params();

        let object = if object.resource_version().is_some() {
            objects.replace(&name, &pp, &object).await?
        } else {
            objects.create(&pp, &object).await?
        };

        Ok(object)
    }

    pub async fn ensure_default_namespaced_k_is_installed<K>(
        &self,
        mut object: K,
    ) -> kube::Result<K>
    where
        K: kube::Resource + ser::Serialize + de::DeserializeOwned + Clone + fmt::Debug,
        <K as kube::Resource>::DynamicType: Default,
    {
        let name = object.name();
        let objects = self.default_namespaced_api::<K>();
        if let Ok(existing) = objects.get(&name).await {
            object.meta_mut().resource_version = existing.resource_version();
        }

        let pp = self.post_params();

        let object = if object.resource_version().is_some() {
            objects.replace(&name, &pp, &object).await?
        } else {
            objects.create(&pp, &object).await?
        };

        Ok(object)
    }
}

impl fmt::Debug for Kubectl {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Kubectl")
            .field("client", &"<kube::Client>")
            .finish()
    }
}

async fn load_config_from_text(text: &str) -> Result<config::Config, config::KubeconfigError> {
    let kubeconfig = config::Kubeconfig::from_yaml(text)?;
    let options = kube::config::KubeConfigOptions::default();
    config::Config::from_custom_kubeconfig(kubeconfig, &options).await
}

fn from_kubeconfig_error(err: config::KubeconfigError) -> kube::Error {
    // This is a workaround until the real error code will be possible to create
    // let infer = config::InferConfigError {
    //     in_cluster: config::InClusterError::MissingEnvironmentVariables, // Dummy error
    //     kubeconfig: err,
    // };
    // kube::Error::InferConfig(infer)
    log::warn!(
        "Replacing {} with Error::LinesCodecMaxLineLengthExceeded",
        err
    );

    kube::Error::LinesCodecMaxLineLengthExceeded
}
