//
// Copyright (c) 2021 RepliXio Ltd. All rights reserved.
// Use is subject to license terms.
//

use serde_json as json;

use super::*;

impl super::Kubectl {
    pub fn delete_params(&self) -> api::DeleteParams {
        api::DeleteParams {
            grace_period_seconds: Some(0),
            ..api::DeleteParams::default()
        }
    }

    pub fn list_params(&self) -> api::ListParams {
        api::ListParams::default()
    }

    pub fn post_params(&self) -> api::PostParams {
        api::PostParams::default()
    }

    pub fn api<T>(&self) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client.clone();
        api::Api::<T>::all(client)
    }

    pub fn default_namespaced_api<T>(&self) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client.clone();
        api::Api::<T>::default_namespaced(client)
    }

    pub fn namespaced_api<T>(&self, namespace: &str) -> api::Api<T>
    where
        T: kube::Resource,
        <T as kube::Resource>::DynamicType: Default,
    {
        let client = self.client.clone();
        api::Api::<T>::namespaced(client, namespace)
    }

    pub async fn get_secret(&self, name: &str) -> kube::Result<corev1::Secret> {
        self.default_namespaced_api::<corev1::Secret>()
            .get(name)
            .await
    }

    pub async fn get_config_map(&self, name: &str) -> kube::Result<corev1::ConfigMap> {
        self.default_namespaced_api::<corev1::ConfigMap>()
            .get(name)
            .await
    }

    pub async fn put_config_map(
        &self,
        name: &str,
        data: BTreeMap<String, String>,
    ) -> kube::Result<corev1::ConfigMap> {
        let configmaps = self.default_namespaced_api::<corev1::ConfigMap>();
        let mut cm: corev1::ConfigMap = json::from_value(json::json!(
            {
                "metadata": {
                    "name": name
                },
                "data": data
            }
        ))
        .expect("Failed to create ConfigMap");

        let pp = self.post_params();

        let cm = if let Ok(existing) = configmaps.get(name).await {
            cm.meta_mut().resource_version = existing.resource_version();
            configmaps.replace(name, &pp, &cm).await?
        } else {
            configmaps.create(&pp, &cm).await?
        };

        Ok(cm)
    }
}
