struct Statecharts {
    machine: Machine,
}

struct Machine {
    current_state: State,
}

enum Event {
    SUBMIT,
    RESOLVE,
    REJECT,
    IDLE,
    RETRY,
}

#[derive(PartialEq, Debug)]
enum State {
    Idle,
    Loading,
    Success,
    Failure,
}

impl Machine {
    fn new(initial_state: State) -> Self {
        Self {
            current_state: initial_state,
        }
    }
    fn transition(&mut self, e: Event) {
        self.current_state = match self.current_state {
            State::Idle => State::Loading,
            State::Loading => match e {
                Event::RESOLVE => State::Success,
                Event::REJECT => State::Failure,
                _ => State::Loading,
            },
            State::Success => match e {
                Event::IDLE => State::Idle,
                _ => State::Success,
            },
            State::Failure => match e {
                Event::RETRY => State::Loading,
                _ => State::Failure,
            },
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn idle_to_loading() {
        let mut machine = Machine::new(State::Idle);
        machine.transition(Event::SUBMIT);
        assert_eq!(machine.current_state, State::Loading);
    }
    #[test]
    fn loading_resolve() {
        let mut machine = Machine::new(State::Loading);
        machine.transition(Event::RESOLVE);
        assert_eq!(machine.current_state, State::Success);
    }
    #[test]
    fn loading_reject() {
        let mut machine = Machine::new(State::Loading);
        machine.transition(Event::REJECT);
        assert_eq!(machine.current_state, State::Failure);
    }
    #[test]
    fn success_to_idle() {
        let mut machine = Machine::new(State::Success);
        machine.transition(Event::IDLE);
        assert_eq!(machine.current_state, State::Idle);
    }
    #[test]
    fn failure_to_retry() {
        let mut machine = Machine::new(State::Failure);
        machine.transition(Event::RETRY);
        assert_eq!(machine.current_state, State::Loading);
    }
}
