//     starship-gitlab - GitLab Custom Command for starship.rs
//
//         The MIT License (MIT)
//
//      Copyright (c) KoresFramework (https://gitlab.com/Kores/)
//      Copyright (c) contributors
//
//      Permission is hereby granted, free of charge, to any person obtaining a copy
//      of this software and associated documentation files (the "Software"), to deal
//      in the Software without restriction, including without limitation the rights
//      to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//      copies of the Software, and to permit persons to whom the Software is
//      furnished to do so, subject to the following conditions:
//
//      The above copyright notice and this permission notice shall be included in
//      all copies or substantial portions of the Software.
//
//      THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//      IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//      FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//      AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//      LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//      OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//      THE SOFTWARE.
mod git_api;
mod gitlab_api;
mod gitlab_cache;

use crate::git_api::GitApi;
use crate::gitlab_api::{GitlabClient, Pipeline};
use crate::gitlab_cache::GitlabCache;
use clap::{AppSettings, Clap};
use git2::Repository;
use itertools::Itertools;
use simplelog::{Config, LevelFilter};
use termion::color;

#[derive(Clap)]
#[clap(version = "1.0", author = "Jonathan H. R. Lopes <jhrldev@gmail.com>")]
#[clap(setting = AppSettings::ColoredHelp)]
struct Opts {
    #[clap(
        short,
        long,
        default_value = "gitlab.com",
        about = "The Gitlab host, used to detect if project exists on Gitlab and to query Pipeline information"
    )]
    gitlab_host: String,
    #[clap(
        short,
        long,
        default_value = "master",
        about = "This branch will be queried alongside the current branch everytime"
    )]
    main: String,
    #[clap(long, about = "Clears the starship-gitlab global cache")]
    clear: bool,
    /*#[clap(short, long, default_value = "3")]
    commits: usize,*/
    #[clap(
        short,
        long,
        default_value = "3",
        about = "Amount of tags to query alongside current branch and main branch"
    )]
    tags: usize,
    #[clap(
        short,
        long,
        about = "Query Pipelines of latest created tags alongside current and main branches"
    )]
    last_tags: bool,
    #[clap(
        short,
        long,
        about = "Verbose mode, configures the logger to INFO Level and prints cache invalidations"
    )]
    verbose: bool,
    #[clap(flatten, about = "Pipeline status format configuration")]
    format: Format,
}

#[derive(Clap)]
struct Format {
    #[clap(long, default_value = "🏭")]
    prefix: String,
    #[clap(long, default_value = "📖")]
    created: String,
    #[clap(long, default_value = "💻")]
    waiting_for_resource: String,
    #[clap(long, default_value = "🔨")]
    preparing: String,
    #[clap(long, default_value = "◌")]
    pending: String,
    #[clap(long, default_value = "➜")]
    running: String,
    #[clap(long, default_value = "✔")]
    success: String,
    #[clap(long, default_value = "⨯")]
    failure: String,
    #[clap(long, default_value = "◼")]
    canceled: String,
    #[clap(long, default_value = "»")]
    skipped: String,
    #[clap(long, default_value = "🖐")]
    manual: String,
    #[clap(long, default_value = "⏳")]
    scheduled: String,
}

#[derive(Debug)]
enum Error {
    RepoLoadError(String),
    RemoteReadError(String),
    UrlParseError(String),
    ReadTagError,
    FindRefError(String),
    FindRefOidError,
    NoCommitsError,
    CouldNotClearKeyError(String, String),
    CouldNotClearDbError(String),
    CouldNotSetVersionError(String),
    CouldNotOpenDbError(String),
    CouldNotReadVersionError(String),
    CouldNotCacheProjectError(String),
    CouldReadCacheProjectError(String),
    ProjectNotFoundError,
    VersionNotFoundError,
    FindCommitError(String),
    RetrieveTagError(String),
    MissingHeadError(String),
    ShortHandNameError,
    NotGitLabUrl(String),
    NoRemoteUrls,
    MultiErrors(Vec<Error>),
    FailedToBuildEndpoint(String),
    FailedToQueryProject(String),
    FailedToQueryPipelines(String),
    FailedToBuildGitlabClient(String),
    ExtractHostError(String),
    CouldNotFindCacheDirError,
}

#[tokio::main]
async fn main() -> Result<(), Error> {
    let opts: Opts = Opts::parse();

    if opts.verbose {
        simplelog::SimpleLogger::init(LevelFilter::Info, Config::default()).unwrap();
    }

    if opts.clear {
        return GitlabCache::new().and_then(|mut f| f.clear());
    }

    let repo = load_repo()?;
    let git = GitApi::new(repo);
    let head = git.current_head()?;
    let head_name = head.name();

    let gitlab = git.find_gitlab(&opts.gitlab_host)?;

    let host = gitlab
        .clone()
        .host
        .ok_or(Error::ExtractHostError(format!("{:?}", gitlab)))?;

    let full_name = gitlab.fullname;
    let time = git.last_commit_time()?;
    let mut client = GitlabClient::new(host).await?;
    let project = client.read_project_from_cache_or_network(full_name).await?;
    let pipelines = client
        .read_pipeline_from_cache_or_network(&git, &project, time, &head_name)
        .await?;

    let mut messages: Vec<String> = vec![];

    messages.extend(get_single_pipeline_status_message(
        &opts, &head_name, pipelines,
    ));

    if head_name != opts.main {
        let main_pipelines = client
            .read_pipeline_from_cache_or_network(&git, &project, time, &opts.main)
            .await?;

        messages.extend(get_single_pipeline_status_message(
            &opts,
            &opts.main,
            main_pipelines,
        ));
    }

    if opts.last_tags {
        if let Ok(tag) = git.last_tags(opts.tags) {
            /*let mut commit_pipelines: Vec<Pipeline> = vec![];
            for c in commits {
                let pipelines = client
                    .read_pipeline(project.id, c.commit.id().to_string().as_str())
                    .await?;

                for pipe in pipelines {
                    commit_pipelines.push(pipe);
                }
            }*/

            for t in tag {
                let pipelines = client
                    .read_pipeline_from_cache_or_network(&git, &project, time, &t.tag)
                    .await?;
                let pipe: Vec<Pipeline> = pipelines.into_iter().take(1).collect();

                messages.push(get_pipeline_status_message(&opts, &t.tag, &pipe));
            }
        }
    }

    if !messages.is_empty() {
        print!("{} {}", opts.format.prefix, messages.join(" "))
    }

    Ok(())
}

fn get_single_pipeline_status_message(
    opts: &Opts,
    ref_name: &String,
    pipelines: Vec<Pipeline>,
) -> Vec<String> {
    let mut messages: Vec<String> = vec![];
    for p in pipelines {
        let pipe = vec![p];
        let msg = get_pipeline_status_message(&opts, ref_name, &pipe);
        messages.push(msg);
        break;
    }
    messages
}

fn get_pipeline_status_message(
    opts: &Opts,
    ref_name: &String,
    pipelines: &Vec<Pipeline>,
) -> String {
    pipelines
        .iter()
        .map(|p| match p.status.as_ref() {
            "created" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::LightCyan),
                ref_name,
                &opts.format.created
            )),
            "waiting_for_resource" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Yellow),
                ref_name,
                &opts.format.waiting_for_resource
            )),
            "preparing" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Blue),
                ref_name,
                &opts.format.preparing
            )),
            "pending" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::LightBlue),
                ref_name,
                &opts.format.pending
            )),
            "running" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::LightYellow),
                ref_name,
                &opts.format.running
            )),
            "success" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Cyan),
                ref_name,
                &opts.format.success
            )),
            "failed" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Red),
                ref_name,
                &opts.format.failure
            )),
            "canceled" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Cyan),
                ref_name,
                &opts.format.failure
            )),
            "skipped" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::Reset),
                ref_name,
                &opts.format.failure
            )),
            "manual" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::LightBlue),
                ref_name,
                &opts.format.failure
            )),
            "scheduled" => Some(format!(
                "{}[{} {}]",
                color::Fg(color::LightYellow),
                ref_name,
                &opts.format.failure
            )),
            _ => None,
        })
        .filter(|p| p.is_some())
        .map(|p| p.unwrap())
        .join(" ")
}

fn load_repo() -> Result<Repository, Error> {
    match Repository::open(".") {
        Ok(repo) => Ok(repo),
        Err(e) => Err(Error::RepoLoadError(e.to_string())),
    }
}
