# @generated
# Copyright 2016 The Bazel Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

_REDIRECTS_FILETYPE = [".redirects"]
_TAR_FILETYPE = [".tar"]

def _gen_redirects_impl(ctx):
  redirect_str = ""
  redirects_map = ctx.attr.redirects
  for path in redirects_map:
    redirect_str += "%s\t%s\n" % (path, redirects_map[path])
  ctx.file_action(output=ctx.outputs.redirects, content=redirect_str)

def _site_tar_impl(ctx):
  ctx.action(
      inputs = [ctx.file.src] + (
          [ctx.file.redirects_file] if ctx.file.redirects_file else []),
      executable = ctx.executable.jekyll_tree,
      arguments = [
              ctx.outputs.out.path,
              ctx.file.src.path
          ] + ([ctx.file.redirects_file.path]
               if ctx.file.redirects_file
               else []),
      outputs = [ctx.outputs.out],
      mnemonic = "SiteTar",
      use_default_shell_env = True,
      progress_message = "Generating site tarball.")


_gen_redirects = rule(
    implementation = _gen_redirects_impl,
    attrs = {
        "redirects": attr.string_dict(mandatory=True, allow_empty=False),
    },
    outputs = {
        "redirects": "%{name}.redirects",
    },
)
"""Writes a tab-delimited file containing mapping of page path to redirect URL.

Helper rule for `_site_tar`. Takes a string_dict containing a mapping of page
path to redirect URL as input and writes the map to a text file. Each line
represents a single redirect page, consists of the page path, a tab character,
and the redirect URL.

Args:
  redirects: String dict containing the redirect mapping.

Outputs:
  redirects: Text file containing the mapping.
"""

_site_tar = rule(
    implementation = _site_tar_impl,
    attrs = {
        "src": attr.label(mandatory=True,
                          allow_files=_TAR_FILETYPE,
                          single_file=True),
        "redirects_file": attr.label(allow_files=_REDIRECTS_FILETYPE,
                                     single_file=True),
        "jekyll_tree": attr.label(default=Label("//:build-jekyll-tree"),
                                  cfg="host",
                                  executable=True),
    },
    outputs = {
        "out": "%{name}.tar",
    },
)
"""Generates redirects in the Jekyll tree archive.

Args:
  src: Label of the Jekyll tree archive.
  redirects_file: File containing the mapping of page path to redirect URL as
    generated by `_gen_redirects`

Outputs:
  out: Tar archive containing the Jekyll tree with the generated redirect pages.
"""

def site_tar(name, src, redirects={}):
  """Modifies the Jekyll tree, generating the specified redirect pages.

  Args:
    name: A unique name for this rule.
    src: The label of the Jekyll tree archive.
    redirects: Dict mapping page path to redirect URL.
  """
  _gen_redirects(
      name = "%s_redirects" % name,
      redirects = redirects,
  )

  _site_tar(
      name = name,
      src = src,
      redirects_file = "%s_redirects" % name,
  )
