/*
 * Copyright 2019 The Starlark in Rust Authors.
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//! Instruction arguments for working with stack.

use crate::{eval::bc::stack_ptr::BcStackPtr, values::Value};

/// Instruction parameter for values popped from stack or pushed to stack.
pub(crate) trait BcStackValues<'v>: Copy {
    /// How many values.
    const N: u32;

    /// Pop a value off the stack.
    fn pop(stack: &BcStackPtr<'v, '_>) -> Self;

    /// Push on the stack.
    fn push(stack: &mut BcStackPtr<'v, '_>, values: Self);
}

impl<'v> BcStackValues<'v> for () {
    const N: u32 = 0;

    fn pop(_interp: &BcStackPtr<'v, '_>) {}

    fn push(_interp: &mut BcStackPtr<'v, '_>, (): ()) {}
}

impl<'v> BcStackValues<'v> for Value<'v> {
    const N: u32 = 1;

    fn pop(stack: &BcStackPtr<'v, '_>) -> Self {
        stack.pop()
    }

    fn push(stack: &mut BcStackPtr<'v, '_>, values: Self) {
        stack.push(values);
    }
}

impl<'v, const N: usize> BcStackValues<'v> for [Value<'v>; N] {
    const N: u32 = N as u32;

    fn pop(stack: &BcStackPtr<'v, '_>) -> Self {
        stack.pop_array()
    }

    fn push(stack: &mut BcStackPtr<'v, '_>, values: Self) {
        stack.push_array(values);
    }
}
