// stargazer - A Gemini Server
// Copyright (C) 2021 Ben Aaron Goldberg <ben@benaaron.dev>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

use colored::Colorize;
use crossbeam_channel::{self as chan, Sender};
use log::{Level, LevelFilter, Log, Metadata, Record, SetLoggerError};
use std::{
    io::{self, Write},
    thread,
};
use time::{format_description::well_known::Rfc3339, OffsetDateTime};

enum Channel {
    StdOut,
    StdErr,
}

pub struct Logger {
    _shutdown: Sender<()>,
    message_queue: Sender<Option<(String, Channel)>>,
}

pub fn init(log_level: LevelFilter) -> Result<(), SetLoggerError> {
    let (_shutdown, s): (Sender<()>, _) = chan::bounded(0);
    let (message_queue, m): (Sender<Option<(String, Channel)>>, _) =
        chan::unbounded();

    thread::Builder::new()
        .name("stargazer-logger".into())
        .spawn(move || {
            let mut stderr = io::stderr();
            let mut stdout = io::stdout();
            while !s.try_recv().unwrap_err().is_disconnected() {
                while let Ok(op) = m.recv() {
                    match op {
                        Some((mut msg, channel)) => {
                            if !msg.ends_with('\n') {
                                msg.push('\n');
                            }
                            match channel {
                                Channel::StdOut => stdout
                                    .write_all(msg.as_bytes())
                                    .expect("Error writing to stdout"),
                                Channel::StdErr => stderr
                                    .write_all(msg.as_bytes())
                                    .expect("Error writing to stderr"),
                            }
                        }
                        None => {
                            stdout.flush().expect("Error flushing stdout");
                            stderr.flush().expect("Error flushing stderr");
                        }
                    }
                }
            }
        })
        .expect("Error starting logger thread");

    let logger = Logger {
        _shutdown,
        message_queue,
    };
    log::set_boxed_logger(Box::new(logger))
        .map(|()| log::set_max_level(log_level))
}

impl Log for Logger {
    fn enabled(&self, metadata: &Metadata) -> bool {
        metadata.target().starts_with("stargazer")
    }

    fn log(&self, record: &Record) {
        if self.enabled(record.metadata()) {
            let level = match record.level() {
                Level::Error => Level::Error.to_string().red(),
                Level::Debug => Level::Debug.to_string().yellow(),
                level => level.to_string().normal(),
            };
            let channel = match record.level() {
                Level::Info => Channel::StdOut,
                _ => Channel::StdErr,
            };
            let now = OffsetDateTime::now_local()
                .unwrap_or_else(|_| OffsetDateTime::now_utc());
            let msg = if record.level() == Level::Info {
                format!(
                    "{}\t{}",
                    now.format(&Rfc3339).unwrap_or_else(|_| {
                        "1970-01-01T00:00:00.00+00:00".to_owned()
                    }),
                    record.args()
                )
            } else {
                format!(
                    "{} {} - {}",
                    now.format(&Rfc3339).unwrap_or_else(|_| {
                        "1970-01-01T00:00:00.00+00:00".to_owned()
                    }),
                    level,
                    record.args()
                )
            };
            self.message_queue
                .send(Some((msg, channel)))
                .expect("Logger thread isn't running");
        }
    }

    fn flush(&self) {
        // Wait for the message queue empty
        while !self.message_queue.is_empty() {}
        // Tell the logger to flush
        self.message_queue.send(None).unwrap();
        // This should make sure that the first flush finished before the function returns
        while !self.message_queue.is_empty() {}
        self.message_queue.send(None).unwrap();
        while !self.message_queue.is_empty() {}
    }
}
