// stargazer - A Gemini Server
// Copyright (C) 2021 Ben Aaron Goldberg <ben@benaaron.dev>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

use anyhow::Error;
use std::{
    convert::From,
    error::Error as StdError,
    fmt::{Display, Formatter},
    io::Error as IOError,
    result::Result as Res,
};

pub type Result<T = ()> = Res<T, GemError>;

#[derive(Debug)]
pub enum GemError {
    NotFound,
    ServerError(Error),
    IOError(Error),
    TimeOut,
    BadReq(Error),
    NoProxy(Error),
    Redirect(String),
    CGIError(Error),
    ClientCertRequired,
    ClientCertNoAuth,
}

impl GemError {
    pub fn unwrap_err(self) -> Error {
        match self {
            GemError::BadReq(e) | GemError::ServerError(e) => e,
            _ => panic!("Unwrapped on GemError with no error"),
        }
    }

    pub fn status(&self) -> u8 {
        match self {
            GemError::NotFound => 51,
            GemError::ServerError(_) => 40,
            GemError::IOError(_) => 40,
            GemError::TimeOut => 59,
            GemError::BadReq(_) => 59,
            GemError::NoProxy(_) => 53,
            GemError::Redirect(_) => 31,
            GemError::CGIError(_) => 42,
            GemError::ClientCertRequired => 60,
            GemError::ClientCertNoAuth => 61,
        }
    }

    pub fn meta(&self) -> &str {
        match self {
            GemError::NotFound => "Not found",
            GemError::ServerError(_) => "Internal server error",
            GemError::IOError(_) => "I/O error",
            GemError::TimeOut => "Request timeout",
            GemError::BadReq(_) => "Bad request",
            GemError::NoProxy(_) => "Proxy request refused",
            GemError::Redirect(url) => url,
            GemError::CGIError(_) => "CGI error",
            GemError::ClientCertRequired => "Please provide your cert to verify yourself",
            GemError::ClientCertNoAuth => "Client cert provided is not allowed to access this page",
        }
    }

    pub fn log(&self) {
        match self {
            GemError::NotFound => log::debug!("{}", self),
            GemError::ServerError(_) => log::error!("{}", self),
            GemError::IOError(_) => log::debug!("{}", self),
            GemError::TimeOut => log::debug!("{}", self),
            GemError::BadReq(_) => log::debug!("{}", self),
            GemError::NoProxy(_) => log::debug!("{}", self),
            GemError::Redirect(_) => log::debug!("{}", self),
            GemError::CGIError(_) => log::warn!("{}", self),
            GemError::ClientCertRequired => log::debug!("{}", self),
            GemError::ClientCertNoAuth => log::debug!("{}", self),
        }
    }
}

impl<T: std::error::Error + Send + Sync + 'static> From<T> for GemError {
    fn from(other: T) -> Self {
        Self::ServerError(other.into())
    }
}

impl Display for GemError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self {
            GemError::NotFound => write!(f, "{}", self.meta()),
            GemError::ServerError(e) => write!(f, "{}: {:#}", self.meta(), e),
            GemError::IOError(e) => write!(f, "{}: {:#}", self.meta(), e),
            GemError::TimeOut => write!(f, "{}", self.meta()),
            GemError::BadReq(e) => write!(f, "{}: {:#}", self.meta(), e),
            GemError::NoProxy(e) => write!(f, "{}: {:#}", self.meta(), e),
            GemError::Redirect(_) => writeln!(f, "Redirect to: {}", self.meta()),
            GemError::CGIError(e) => write!(f, "{}: {:#}", self.meta(), e),
            GemError::ClientCertRequired => write!(f, "{}", self.meta()),
            GemError::ClientCertNoAuth => write!(f, "{}", self.meta()),
        }
    }
}

pub trait Context<T, E> {
    /// Wrap the error value with additional context.
    fn context<C>(self, context: C) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static;

    /// Wrap the error value with additional context that is evaluated lazily
    /// only once an error does occur.
    fn with_context<C, F>(self, f: F) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static,
        F: FnOnce() -> C;
}

impl<T, E> Context<T, E> for Res<T, E>
where
    E: StdError + Send + Sync + 'static,
{
    fn context<C>(self, c: C) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static,
    {
        anyhow::Context::context(self, c).map_err(GemError::ServerError)
    }

    fn with_context<C, F>(self, c: F) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static,
        F: FnOnce() -> C,
    {
        anyhow::Context::with_context(self, c)
            .map_err(GemError::ServerError)
    }
}

impl<T> Context<T, GemError> for Option<T> {
    fn context<C>(self, c: C) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static,
    {
        anyhow::Context::context(self, c).map_err(GemError::ServerError)
    }

    fn with_context<C, F>(self, c: F) -> Res<T, GemError>
    where
        C: Display + Send + Sync + 'static,
        F: FnOnce() -> C,
    {
        anyhow::Context::with_context(self, c)
            .map_err(GemError::ServerError)
    }
}

pub trait ErrorConv<T> {
    fn into_io_error(self) -> Res<T, GemError>;
    fn into_bad_req(self) -> Res<T, GemError>;
    fn into_server_error(self) -> Res<T, GemError>;
}

impl<T> ErrorConv<T> for Res<T, GemError> {
    fn into_io_error(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::IOError(e.unwrap_err()))
    }
    fn into_bad_req(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::BadReq(e.unwrap_err()))
    }
    fn into_server_error(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::ServerError(e.unwrap_err()))
    }
}

impl<T> ErrorConv<T> for Res<T, IOError> {
    fn into_io_error(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::IOError(e.into()))
    }
    fn into_bad_req(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::BadReq(e.into()))
    }
    fn into_server_error(self) -> Res<T, GemError> {
        self.map_err(|e| GemError::ServerError(e.into()))
    }
}

impl<T> ErrorConv<T> for Res<T, Error> {
    fn into_io_error(self) -> Res<T, GemError> {
        self.map_err(GemError::IOError)
    }
    fn into_bad_req(self) -> Res<T, GemError> {
        self.map_err(GemError::BadReq)
    }
    fn into_server_error(self) -> Res<T, GemError> {
        self.map_err(GemError::ServerError)
    }
}

pub fn bad_req(s: &'static str) -> GemError {
    GemError::BadReq(anyhow::anyhow!(s))
}

pub fn error(s: &'static str) -> GemError {
    GemError::ServerError(anyhow::anyhow!(s))
}
