// stargazer - A Gemini Server
// Copyright (C) 2021 Ben Aaron Goldberg <ben@benaaron.dev>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

use crate::error::Result;
use async_net::TcpStream;
use futures_rustls::server::TlsStream;
use futures_lite::*;
use percent_encoding::{percent_encode, NON_ALPHANUMERIC};
use std::cmp::Ordering;
use std::ffi::OsString;
use std::path::Path;

pub async fn gen(
    dir_path: &Path,
    stream: &mut TlsStream<TcpStream>,
) -> Result<usize> {
    let mut body_size = 6usize;
    stream.write_all(b"=> ..\n").await?;

    let mut entries = async_fs::read_dir(dir_path).await?;
    let mut file_list = Vec::with_capacity(32);

    while let Some(res) = entries.next().await {
        let entry = res?;
        let file_type = entry.file_type().await?;
        let is_dir = entry.file_type().await?.is_dir();
        let name = entry.file_name();
        let name_str = name.to_string_lossy().to_string();
        let name_bytes = os_str_to_bytes(name);
        let encoded: String =
            percent_encode(&name_bytes, NON_ALPHANUMERIC).collect();
        let mut line_buf = String::with_capacity(64);
        line_buf.push_str("=>");
        line_buf.push_str(&encoded);
        if file_type.is_dir() {
            line_buf.push('/');
        }
        line_buf.push(' ');
        line_buf.push_str(&name_str);
        line_buf.push('\n');
        file_list.push((line_buf, is_dir));
    }
    file_list.sort_unstable_by(|left, right| {
        if left.1 && !right.1 {
            Ordering::Less
        } else if !left.1 && right.1 {
            Ordering::Greater
        } else {
            left.0.cmp(&right.0)
        }
    });
    for (file_line, _) in file_list {
        stream.write_all(file_line.as_bytes()).await?;
        body_size += file_line.as_bytes().len();
    }

    Ok(body_size)
}

#[cfg(unix)]
fn os_str_to_bytes(s: OsString) -> Vec<u8> {
    use std::os::unix::ffi::OsStringExt;
    s.into_vec()
}

#[cfg(not(unix))]
fn os_str_to_bytes(s: OsString) -> Vec<u8> {
    s.to_string_lossy().as_bytes().to_vec()
}
