// stargazer - A Gemini Server
// Copyright (C) 2021 Ben Aaron Goldberg <ben@benaaron.dev>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

use std::path::{Path, PathBuf};

const HELP: &str = concat!("stargazer ", env!("CARGO_PKG_VERSION"), "
Sashanoraa <sasha@noraa.gay>

A fast and easy to use Gemini server.

Project home page: https://sr.ht/~zethra/stargazer/

USAGE:
    stargazer [FLAGS] [CONFIG_PATH]

ARGS:
    CONFIG_PATH         Path to config file. Defaults to /etc/stargazer.ini if no path is provided.

FLAGS:
        --check-config    Check config and exit.
    -D, --debug           Print debug information while server is running.
    -d, --dev             Run server is dev mode. Servers out of current directory.
    -h, --help            Prints help information.
    -V, --version         Prints version information.
");

#[derive(Debug)]
pub struct Args {
    /// Print debug information while server is running
    pub debug: bool,
    /// Path to config file
    pub config_path: PathBuf,
    /// Run server is dev mode. Servers out of current directory
    pub dev: bool,
    /// Check config and exit
    pub check_config: bool,
}

impl Args {
    pub fn from_args() -> Args {
        let mut config_path = None;
        let mut dev = false;
        let mut debug = false;
        let mut check_config = false;
        for arg in std::env::args_os().skip(1) {
            match arg.to_str() {
                Some(s) if s.starts_with('-') => match s {
                    "-D" | "--debug" => debug = true,
                    "-d" | "--dev" => dev = true,
                    "--check-config" => check_config = true,
                    "-V" | "--version" => {
                        println!("stargazer {}", env!("CARGO_PKG_VERSION"));
                        std::process::exit(0);
                    }
                    "-h" | "--help" => {
                        println!("{}", HELP);
                        std::process::exit(0);
                    }
                    _ => {
                        eprintln!("ERROR: stargazer does not recognize the flag {}\n", s);
                        print_error_help();
                    }
                },
                _ => match config_path {
                    Some(_) => {
                        eprintln!("ERROR: stargazer only takes one positional argument CONFIG_PATH\n");
                        print_error_help();
                    }
                    None => config_path = Some(Path::new(&arg).to_owned()),
                },
            }
        }
        Args {
            debug,
            config_path: config_path
                .unwrap_or(Path::new("/etc/stargazer.ini").to_owned()),
            dev,
            check_config,
        }
    }
}

fn print_error_help() {
    println!("{}", HELP);
    std::process::exit(1);
}
