use bytes::Bytes;
use exoquant::{Color, ColorMap, Colorf, SimpleColorSpace};
use flate2::bufread::DeflateDecoder;
use flate2::write::DeflateEncoder;
use flate2::Compression;
use image::{GenericImageView, ImageBuffer, Rgb, RgbImage, Rgba};
#[cfg(feature = "qr")]
use qrcode::{EcLevel, QrCode};
use serde::{Deserialize, Serialize};
use std::error::Error;
use std::fmt;
use std::fs;
use std::io::prelude::*;

const METADATA_BYTES: usize = 32;

#[derive(Debug, Clone)]
pub enum StarBoxError {
    ImageSize,
    MetadataSize,
    MalformedPalette,
    MalformedSegments,
    LoadError,
    UnnecessarySegmentalization,
    SegmentRasterization,
}

impl fmt::Display for StarBoxError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            StarBoxError::ImageSize => write!(f, "Input image is too large."),
            StarBoxError::MetadataSize => {
                write!(f, "Title/Author cannot exceed {} bytes.", METADATA_BYTES)
            }
            StarBoxError::MalformedPalette => write!(f, "JASC-PAL file is malformed."),
            StarBoxError::MalformedSegments => write!(f, "Recieved unexpected segment count."),
            StarBoxError::LoadError => write!(f, "Failed to load StarBox file."),
            StarBoxError::UnnecessarySegmentalization => write!(
                f,
                "Desired segment size is less than or equal to original canvas size."
            ),
            StarBoxError::SegmentRasterization => write!(f, "Cannot rasterize a StarBox segment."),
        }
    }
}

impl Error for StarBoxError {}

#[derive(Debug, Clone, Copy, Serialize, Deserialize, PartialEq)]
pub enum Size {
    Tiny = 8,
    Small = 16,
    Medium = 32,
    Large = 64,
    Huge = 128,
}

#[derive(Debug, Clone, Copy, Serialize, Deserialize, PartialEq)]
pub enum Tilable {
    No,
    X,
    Y,
    XY,
}

#[derive(Debug, Clone, Copy, Serialize, Deserialize, PartialEq)]
pub enum Segment {
    Complete,
    Partial(usize, usize),
}

impl Size {
    pub fn from_dimensions(dim: (u32, u32)) -> Result<Size, Box<dyn Error>> {
        let largest_size = dim.0.max(dim.1);
        if largest_size <= 8 {
            Ok(Size::Tiny)
        } else if largest_size <= 16 {
            Ok(Size::Small)
        } else if largest_size <= 32 {
            Ok(Size::Medium)
        } else if largest_size <= 64 {
            Ok(Size::Large)
        } else if largest_size <= 128 {
            Ok(Size::Huge)
        } else {
            Err(Box::new(StarBoxError::ImageSize))
        }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize, PartialEq)]
pub struct Canvas {
    title: String,
    author: String,
    palette: String,
    segmental: Segment,
    tileable: Tilable,
    size: Size,
    bytes: Bytes,
}

impl Canvas {
    pub fn from_image(file: &str, pal_name: &str) -> Result<Self, Box<dyn Error>> {
        // Load palette
        let palette = load_jasc_palette(pal_name)?;
        // Load image
        let raw_img = image::open(file)?.into_rgba8();
        // Verify size
        let size = Size::from_dimensions(raw_img.dimensions())?;

        // Place into appropriately sized box
        let img = ImageBuffer::from_fn(size as u32, size as u32, |x, y| {
            if raw_img.in_bounds(x, y) {
                let p = raw_img.get_pixel(x, y);
                if p[3] > 0 {
                    p.to_owned()
                } else {
                    Rgba([0, 0, 0, 0])
                }
            } else {
                Rgba([0, 0, 0, 0])
            }
        });

        // Convert to indexed color bytes
        let bytes: Bytes = img
            .pixels()
            .map(|Rgba([r, g, b, _])| {
                palette.find_nearest(Colorf {
                    r: *r as f64 / 255.0,
                    g: *g as f64 / 255.0,
                    b: *b as f64 / 255.0,
                    a: 1.0,
                }) as u8
            })
            .collect();

        Ok(Canvas {
            palette: pal_name.to_string(),
            size,
            bytes,
            ..Default::default()
        })
    }

    pub fn author(&self) -> String {
        self.author.to_owned()
    }

    pub fn title(&self) -> String {
        self.title.to_owned()
    }

    pub fn segmental(&self) -> Segment {
        self.segmental
    }

    pub fn size(&self) -> Size {
        self.size
    }

    pub fn bytes(&self) -> Bytes {
        self.bytes.clone()
    }

    pub fn is_segment(&self) -> bool {
        match self.segmental {
            Segment::Complete => false,
            Segment::Partial(_, _) => true,
        }
    }

    pub fn set_author(&mut self, author: &str) -> Result<(), Box<dyn Error>> {
        let author = author.trim();

        if author.len() > METADATA_BYTES {
            Err(Box::new(StarBoxError::MetadataSize))
        } else {
            self.author = author.to_string();
            Ok(())
        }
    }

    pub fn set_title(&mut self, title: &str) -> Result<(), Box<dyn Error>> {
        let title = title.trim();

        if title.len() > METADATA_BYTES {
            Err(Box::new(StarBoxError::MetadataSize))
        } else {
            self.title = title.to_string();
            Ok(())
        }
    }

    pub fn get_pixel(&self, x: usize, y: usize) -> Result<u8, Box<dyn Error>> {
        if self.is_segment() {
            Err(Box::new(StarBoxError::SegmentRasterization))
        } else {
            Ok(self.bytes[y * self.size as usize + x])
        }
    }

    pub fn to_image(&self) -> Result<RgbImage, Box<dyn Error>> {
        if self.is_segment() {
            return Err(Box::new(StarBoxError::SegmentRasterization));
        }

        // Load palette
        let palette = load_jasc_palette(&self.palette)?;

        Ok(ImageBuffer::from_fn(
            self.size as u32,
            self.size as u32,
            |x, y| {
                let Colorf { r, g, b, .. } =
                    palette.float_color(self.get_pixel(x as usize, y as usize).unwrap() as usize);
                Rgb([(r * 255.0) as u8, (g * 255.0) as u8, (b * 255.0) as u8])
            },
        ))
    }

    pub fn segmentalize(&self, mut size_bytes: usize) -> Result<Vec<Canvas>, Box<dyn Error>> {
        if self.serialized_size()? as usize <= size_bytes {
            return Err(Box::new(StarBoxError::UnnecessarySegmentalization));
        }

        let overflow = {
            let test_segment = Canvas {
                author: self.author.to_owned(),
                title: self.title.to_owned(),
                bytes: Bytes::copy_from_slice(&self.bytes[..size_bytes]),
                segmental: Segment::Partial(0, 0),
                ..Default::default()
            };
            test_segment.serialized_size()?
        } as usize;

        if overflow > size_bytes {
            size_bytes = size_bytes - (overflow - size_bytes);
        }

        let total_segments = (self.bytes.len() as f64 / size_bytes as f64).ceil() as usize;

        Ok(self
            .bytes
            .chunks(size_bytes)
            .enumerate()
            .map(|(idx, chunk)| Canvas {
                title: self.title.to_owned(),
                author: self.author.to_owned(),
                palette: self.palette.to_owned(),
                segmental: Segment::Partial(idx, total_segments),
                tileable: self.tileable,
                size: self.size,
                bytes: Bytes::copy_from_slice(chunk),
            })
            .collect())
    }

    pub fn from_segments(segments: &[Canvas]) -> Result<Self, Box<dyn Error>> {
        if segments.is_empty() {
            return Err(Box::new(StarBoxError::MalformedSegments));
        }

        let expected_length = match segments[0].segmental() {
            Segment::Complete => 1,
            Segment::Partial(_i, total) => total,
        };

        if expected_length != segments.len() {
            return Err(Box::new(StarBoxError::MalformedSegments));
        }

        let first_segment = &segments[0];
        let bytes = segments
            .iter()
            .flat_map(|seg| seg.bytes.to_owned())
            .collect();
        Ok(Canvas {
            title: first_segment.title.to_owned(),
            author: first_segment.author.to_owned(),
            palette: first_segment.palette.to_owned(),
            segmental: Segment::Complete,
            tileable: first_segment.tileable,
            size: first_segment.size,
            bytes,
        })
    }

    pub fn save_image(&self, file: &str) -> Result<(), Box<dyn Error>> {
        let img = self.to_image()?;
        img.save(file)?;
        Ok(())
    }

    #[cfg(feature = "qr")]
    pub fn save_qr(&self, file: &str) -> Result<(), Box<dyn Error>> {
        let qr = QrCode::with_error_correction_level(self.to_bytes()?, EcLevel::L)?;
        qr.render::<Rgb<u8>>().build().save(file)?;
        Ok(())
    }

    pub fn load(file: &str) -> Result<Canvas, Box<dyn Error>> {
        let bytes = &fs::read(file)?;
        Canvas::from_bytes(bytes)
    }

    pub fn save(&self, file: &str) -> Result<(), Box<dyn Error>> {
        fs::write(file, &self.to_bytes()?)?;
        Ok(())
    }

    pub fn to_bytes(&self) -> Result<Vec<u8>, Box<dyn Error>> {
        let bytes = bincode::serialize(&self)?;
        let mut encoder = DeflateEncoder::new(Vec::new(), Compression::best());
        encoder.write_all(&bytes)?;
        Ok(encoder.finish()?)
    }

    pub fn from_bytes(bytes: &[u8]) -> Result<Self, Box<dyn Error>> {
        let mut decoder = DeflateDecoder::new(bytes);
        let mut bytes = Vec::new();
        decoder.read_to_end(&mut bytes)?;
        Ok(bincode::deserialize::<Canvas>(&bytes)?)
    }

    pub fn serialized_size(&self) -> Result<u64, Box<dyn Error>> {
        Ok(self.to_bytes()?.len() as u64)
    }
}

impl Default for Canvas {
    fn default() -> Canvas {
        Canvas {
            title: String::from("Untitled"),
            author: String::from("Unknown"),
            palette: String::from("windows-95-256"),
            segmental: Segment::Complete,
            tileable: Tilable::No,
            size: Size::Tiny,
            bytes: Bytes::new(),
        }
    }
}

fn load_jasc_palette(file: &str) -> Result<ColorMap, Box<dyn Error>> {
    let mut colors: Vec<Color> = Vec::new();

    let palette_file = fs::read_to_string(format!("palettes/{}.pal", file)).unwrap();
    let mut palette_file = palette_file.lines().map(|l| l.trim());

    if palette_file.next() != Some("JASC-PAL") || palette_file.next() != Some("0100") {
        return Err(Box::new(StarBoxError::MalformedPalette));
    }

    for color in palette_file.skip(1) {
        if let [r, g, b] = color
            .split_whitespace()
            .map(|c| c.parse::<u8>().unwrap())
            .take(3)
            .collect::<Vec<u8>>()[..]
        {
            colors.push(Color { r, g, b, a: 255 })
        }
    }

    Ok(ColorMap::new(&colors, &SimpleColorSpace::default()))
}
