#![cfg_attr(not(feature = "std"), no_std)]
#![deny(rust_2018_idioms, unused_qualifications)]
#![warn(unreachable_pub)]
#![allow(non_camel_case_types, unstable_name_collisions, clippy::all)]

/*!
Standback backports a number of methods, structs, and macros that have been stabilized in the Rust
standard library since 1.36.0. This allows crate authors to depend on Standback rather than forcing
downstream users to upgrade their compiler (or not use the new feature at all).

Due to a variety of restrictions in the Rust, it is not possible to implement everything that has
been stabilized.

# Usage

For most cases, importing the prelude should suffice.

```rust,no_run
use standback::prelude::*;
```

If you are using anything that would normally have to be imported, just use the `standback` crate
instead of `core`, `alloc`, or `std`.

```rust,no_run
use standback::mem::take;
```

It is _highly_ recommended to use `#![allow(unstable_name_collisions)]`, as that's the whole point
of this crate. Just be extra-careful to not do it for anything that _can't_ be backported.

# `#![no_std]` support

By default, there standard library is used where necessary. If support for `#![no_std]` is required,
use `default-features = false`.

Items that require an allocator are gated under the `alloc` feature, which is enabled by default via
the `std` feature.

# Minimum supported Rust version

By default, this crate has a minimum supported Rust version of 1.36. If you do not require a MSRV
this low, you can raise it by using `default-features = false` (be sure to re-enable `std` or
`alloc` if you need it) and enabling a `msrv-1.XX` feature flag, substituting the appropriate
version; the standback crate is not guaranteed to work (let alone produce a reasonable result) if no
MSRV is declared when `default-features = false`. All versions up to the most recent stable release
of a compiler are supported.

Note that items stabilized prior to the declared MSRV _will not_ be re-exported.

# Inherent and trait methods, associated constants

The following methods and constants are available via the prelude. For brevity, `i*` is `i8`, `i16`,
`i32`, `i64`, `i128`, and `isize`; `u*` is `u8`, `u16`, `u32`, `u64`, `u128`, and `usize`.

## 1.52

```text
char::MAX
char::REPLACEMENT_CHARACTER
char::UNICODE_VERSION
char::decode_utf16
char::from_digit
char::from_u32_unchecked
char::from_u32
slice::partition_point
str::rsplit_once
str::split_once
```

## 1.51

```text
Arc::decrement_strong_count
Arc::increment_strong_count
Peekable::next_if_eq
Peekable::next_if
Seek::stream_position
slice::fill_with
slice::split_inclusive_mut
slice::split_inclusive
slice::strip_prefix
slice::strip_suffix
str::split_inclusive
task::Wake
i*::unsigned_abs
Poll::map_ok
Poll::map_err
```

## 1.50

```text
bool::then
btree_map::Entry::or_insert_with_key
hash_map::Entry::or_insert_with_key
{f32, f64}::clamp
Ord::clamp
RefCell::take
slice::fill
UnsafeCell::get_mut
```

## 1.49

```text
slice::select_nth_unstable
slice::select_nth_unstable_by
slice::select_nth_unstable_by_key
```

## 1.48

```text
slice::as_ptr_range
slice::as_mut_ptr_range
```

## 1.47

```text
Range::is_empty
Result::as_deref
Result::as_deref_mut
Vec::leak
f32::TAU
f64::TAU
```

## 1.46

```text
{i*, u*}::leading_ones
{i*, u*}::trailing_ones
Option::zip
```

## 1.45

```text
i*::saturating_abs
i*::saturating_neg
str::strip_prefix
str::strip_suffix
```

## 1.44

```text
PathBuf::with_capacity
PathBuf::capacity
PathBuf::clear
PathBuf::reserve
PathBuf::reserve_exact
PathBuf::shrink_to_fit
Layout::align_to
Layout::pad_to_align
Layout::array
Layout::extend
{f32, f64}::to_int_unchecked
```

## 1.43

```text
{f32, f64}::RADIX
{f32, f64}::MANTISSA_DIGITS
{f32, f64}::DIGITS
{f32, f64}::EPSILON
{f32, f64}::MIN
{f32, f64}::MIN_POSITIVE
{f32, f64}::MAX
{f32, f64}::MIN_EXP
{f32, f64}::MAX_EXP
{f32, f64}::MIN_10_EXP
{f32, f64}::MAX_10_EXP
{f32, f64}::NAN
{f32, f64}::INFINITY
{f32, f64}::NEG_INFINITY
{i*, u*}::MIN
{i*, u*}::MAX
```

## 1.42

```text
CondVar::wait_while
CondVar::wait_timeout_while
ManuallyDrop::take
```

## 1.41

```text
Result::map_or
Result::map_or_else
```

## 1.40

```text
Option::as_deref
Option::as_deref_mut
{f32, f64}::to_be_bytes
{f32, f64}::to_le_bytes
{f32, f64}::to_ne_bytes
{f32, f64}::from_be_bytes
{f32, f64}::from_le_bytes
{f32, f64}::from_ne_bytes
slice::repeat
```

## 1.39

None :(

## 1.38

```text
<*const T>::cast
<*mut T>::cast
Duration::as_secs_f32
Duration::as_secs_f64
Duration::div_f32
Duration::div_f64
Duration::from_secs_f32
Duration::from_secs_f64
Duration::mul_f32
Duration::mul_f64
{i*, u*}::rem_euclid
{i*, u*}::checked_rem_euclid
{i*, u*}::wrapping_rem_euclid
{i*, u*}::overflowing_rem_euclid
{i*, u*}::div_euclid
{i*, u*}::checked_div_euclid
{i*, u*}::wrapping_div_euclid
{i*, u*}::overflowing_div_euclid
{f32, f64}::rem_euclid
{f32, f64}::div_euclid
```

## 1.37

```text
Cell::from_mut
Cell<[T]>::as_slice_of_cells
DoubleEndedIterator::nth_back
Option::xor
slice::copy_within
```

# Free functions and constants

```text
future::pending // 1.48
future::ready // 1.48
char::UNICODE_VERSION // 1.45
{f32, f64}::consts::LOG10_2 // 1.43
{f32, f64}::consts::LOG2_10 // 1.43
iter::once_with // 1.43
mem::take // 1.40
```

# Macros

```text
matches! // 1.42
todo! // 1.39
```
*/

#[allow(unused_extern_crates)]
#[cfg(feature = "alloc")]
extern crate alloc;

mod traits {
    pub trait Sealed<T: ?Sized> {}
    impl<T: ?Sized> Sealed<T> for T {}

    macro_rules! impl_trait_for_all {
        ($trait:ident => $($type:ty)+) => {$(
            impl $trait for $type {}
        )+};
    }

    #[allow(unreachable_pub)]
    pub trait Integer: Sized {}
    impl_trait_for_all!(Integer => i8 i16 i32 i64 i128 isize u8 u16 u32 u64 u128 usize);

    #[allow(unreachable_pub)]
    pub trait SignedInteger {}
    impl_trait_for_all!(SignedInteger => i8 i16 i32 i64 i128 isize);

    #[allow(unreachable_pub)]
    pub trait UnsignedInteger {}
    impl_trait_for_all!(UnsignedInteger => u8 u16 u32 u64 u128 usize);

    #[allow(unreachable_pub)]
    pub trait Float {}
    impl_trait_for_all!(Float => f32 f64);
}

mod pattern;

#[cfg(shim = "1.37")]
mod v1_37;
#[cfg(shim = "1.38")]
mod v1_38;
#[cfg(shim = "1.40")]
mod v1_40;
#[cfg(shim = "1.41")]
mod v1_41;
#[cfg(shim = "1.42")]
mod v1_42;
#[cfg(shim = "1.43")]
mod v1_43;
#[cfg(shim = "1.44")]
mod v1_44;
#[cfg(shim = "1.45")]
mod v1_45;
#[cfg(shim = "1.46")]
mod v1_46;
#[cfg(shim = "1.47")]
mod v1_47;
#[cfg(shim = "1.48")]
mod v1_48;
#[cfg(shim = "1.49")]
mod v1_49;
#[cfg(shim = "1.50")]
mod v1_50;
#[cfg(shim = "1.51")]
mod v1_51;
#[cfg(shim = "1.52")]
mod v1_52;

#[doc(hidden)]
pub mod prelude {
    #[cfg(shim = "1.39")]
    pub use core::unimplemented as todo;

    #[cfg(shim = "1.42")]
    pub use crate::matches;
    #[cfg(shim = "1.37")]
    pub use crate::v1_37::{
        Cell_v1_37, Cell_v1_37_, DoubleEndedIterator_v1_37, Option_v1_37, Slice_v1_37,
    };
    #[cfg(all(shim = "1.38", feature = "std"))]
    pub use crate::v1_38::EuclidFloat_v1_38;
    #[cfg(shim = "1.38")]
    pub use crate::v1_38::{ConstPtr_v1_38, Duration_v1_38, Euclid_v1_38, MutPtr_v1_38};
    #[cfg(all(shim = "1.40", feature = "alloc"))]
    pub use crate::v1_40::slice_v1_40;
    #[cfg(shim = "1.40")]
    pub use crate::v1_40::{f32_v1_40, f64_v1_40, Option_v1_40, Option_v1_40_};
    #[cfg(shim = "1.41")]
    pub use crate::v1_41::Result_v1_41;
    #[cfg(all(shim = "1.42", feature = "std"))]
    pub use crate::v1_42::Condvar_v1_42;
    #[cfg(shim = "1.42")]
    pub use crate::v1_42::ManuallyDrop_v1_42;
    #[cfg(shim = "1.43")]
    pub use crate::v1_43::{float_v1_43, int_v1_43};
    #[cfg(all(shim = "1.44", feature = "std"))]
    pub use crate::v1_44::PathBuf_v1_44;
    #[cfg(shim = "1.44")]
    pub use crate::v1_44::{float_v1_44, Layout_v1_44};
    #[cfg(shim = "1.45")]
    pub use crate::v1_45::{int_v1_45, str_v1_45};
    #[cfg(shim = "1.46")]
    pub use crate::v1_46::{int_v1_46, Option_v1_46};
    #[cfg(all(shim = "1.47", feature = "alloc"))]
    pub use crate::v1_47::Vec_v1_47;
    #[cfg(shim = "1.47")]
    pub use crate::v1_47::{Range_v1_47, Result_v1_47};
    #[cfg(shim = "1.48")]
    pub use crate::v1_48::Slice_v1_48;
    #[cfg(shim = "1.49")]
    pub use crate::v1_49::Slice_v1_49;
    #[cfg(all(shim = "1.50", feature = "alloc"))]
    pub use crate::v1_50::BTreeMapEntry_v1_50;
    #[cfg(all(shim = "1.50", feature = "std"))]
    pub use crate::v1_50::HashMapEntry_v1_50;
    #[cfg(shim = "1.50")]
    pub use crate::v1_50::{
        Bool_v1_50, Float_v1_50, Ord_v1_50, RefCell_v1_50, Slice_v1_50, UnsafeCell_v1_50,
    };
    #[cfg(all(shim = "1.51", feature = "alloc"))]
    pub use crate::v1_51::Arc_v1_51;
    #[cfg(all(shim = "1.51", feature = "std"))]
    pub use crate::v1_51::Seek_v1_51;
    #[cfg(shim = "1.51")]
    pub use crate::v1_51::{
        str_v1_51, Peekable_v1_51, Poll_v1_51, SignedInteger_v1_51, Slice_v1_51,
    };
    #[cfg(shim = "1.52")]
    pub use crate::v1_52::{char_v1_52, str_v1_52, Slice_v1_52};
}
#[doc(hidden)]
pub mod mem {
    #[cfg(reexport = "1.40")]
    pub use core::mem::take;

    #[cfg(shim = "1.40")]
    pub use crate::v1_40::take;
}
#[doc(hidden)]
pub mod iter {
    #[cfg(reexport = "1.43")]
    pub use core::iter::{once_with, OnceWith};

    #[cfg(shim = "1.43")]
    pub use crate::v1_43::{once_with, OnceWith};
}
#[doc(hidden)]
pub mod task {
    #[cfg(all(reexport = "1.51", feature = "alloc"))]
    pub use alloc::task::Wake;

    #[cfg(all(shim = "1.51", feature = "alloc"))]
    pub use crate::v1_51::Wake;
}
#[doc(hidden)]
pub mod f32 {
    pub mod consts {
        #[cfg(reexport = "1.47")]
        pub use core::f32::consts::TAU;
        #[cfg(reexport = "1.43")]
        pub use core::f32::consts::{LOG10_2, LOG2_10};

        #[cfg(shim = "1.43")]
        pub use crate::v1_43::f32::{LOG10_2, LOG2_10};
        #[cfg(shim = "1.47")]
        pub use crate::v1_47::f32::TAU;
    }
}
#[doc(hidden)]
pub mod f64 {
    pub mod consts {
        #[cfg(reexport = "1.47")]
        pub use core::f64::consts::TAU;
        #[cfg(reexport = "1.43")]
        pub use core::f64::consts::{LOG10_2, LOG2_10};

        #[cfg(shim = "1.43")]
        pub use crate::v1_43::f64::{LOG10_2, LOG2_10};
        #[cfg(shim = "1.47")]
        pub use crate::v1_47::f64::TAU;
    }
}
#[doc(hidden)]
pub mod char {
    #[cfg(shim = "1.38")]
    pub const UNICODE_VERSION: (u8, u8, u8) = (11, 0, 0);
    #[cfg(all(reexport = "1.38", shim = "1.44"))]
    pub const UNICODE_VERSION: (u8, u8, u8) = (12, 1, 0);
    #[cfg(all(reexport = "1.44", shim = "1.45"))]
    pub const UNICODE_VERSION: (u8, u8, u8) = (13, 0, 0);
    #[cfg(reexport = "1.45")]
    pub use core::char::UNICODE_VERSION;
}
#[doc(hidden)]
pub mod future {
    #[cfg(reexport = "1.48")]
    pub use core::future::{pending, ready, Pending, Ready};

    #[cfg(shim = "1.48")]
    pub use crate::v1_48::future::{pending, ready, Pending, Ready};
}
