pub mod err;

use tokio::io::{AsyncRead, AsyncReadExt, AsyncWrite, AsyncWriteExt};

use bytes::BytesMut;

use serde::de::DeserializeOwned;
use serde::Serialize;

use crc32fast::Hasher;

pub use err::Error;


/// Send a serialized buffer over a tokio `AsyncWrite` connection.
pub async fn send<C, D>(conn: &mut C, data: D) -> Result<(), Error>
where
  C: AsyncWrite + Unpin,
  D: Serialize
{
  //
  // Serialize buffer
  //
  let encoded: Vec<u8> = match bincode::serialize(&data) {
    Ok(buf) => buf,
    Err(e) => {
      return Err(Error::SerDe(e.to_string()));
    }
  };

  let mut crc32 = Hasher::new();
  crc32.update(&encoded[..]);
  let crc32 = crc32.finalize();

  //
  // Determine length of buffer
  //
  const MAXLEN: usize = std::u32::MAX as usize;
  if encoded.len() > MAXLEN {
    // Yikes!
    return Err(Error::TooLong);
  }
  let len = encoded.len() as u32;

  //
  // Write the length, big endian
  //
  let hdr = len.to_be_bytes();
  conn.write_all(&hdr).await?;

  //
  // Write the message buffer
  //
  conn.write_all(&encoded).await?;

  //
  // Write crc32
  //
  let crc = crc32.to_be_bytes();
  conn.write_all(&crc).await?;

  Ok(())
}


/// Receive, and deserialize, a buffer from a tokio `AsyncRead` connection.
pub async fn recv<C, D>(conn: &mut C) -> Result<D, Error>
where
  C: AsyncRead + Unpin,
  D: DeserializeOwned
{
  let len = read_u32(conn).await?;

  Ok(read_buf_to(conn, len as usize).await?)
}


async fn read_u32<T>(stream: &mut T) -> Result<u32, Error>
where
  T: AsyncRead + Unpin
{
  let mut buf = [0u8; 4];

  match stream.read_exact(&mut buf).await {
    Ok(n) => {
      if n != 4 {
        return Err(Error::ConnectionLost);
      }
    }
    Err(ref e) if e.kind() == tokio::io::ErrorKind::UnexpectedEof => {
      return Err(Error::ConnectionLost);
    }
    Err(e) => {
      return Err(e.into());
    }
  }

  Ok(u32::from_be_bytes(buf))
}


async fn read_buf_to<S, T>(stream: &mut S, len: usize) -> Result<T, Error>
where
  S: AsyncRead + Unpin,
  T: DeserializeOwned
{
  //
  // Preallocate buffer for receiving serialized data.
  //
  let mut buf = BytesMut::with_capacity(len);

  //
  // Read the serialized buffer
  //
  loop {
    let n = stream.read_buf(&mut buf).await?;
    if n == 0 {
      return Err(Error::ConnectionLost);
    }
    if buf.len() == len {
      break;
    }
  }

  //
  // Get the CRC32 value
  //
  let recv_crc = read_u32(stream).await?;

  //
  // Calculate the CRC32 of the received buffer
  //
  let mut crc32 = Hasher::new();
  crc32.update(&buf[..]);
  let calc_crc = crc32.finalize();

  //
  // Make sure the CRC values match
  //
  if recv_crc != calc_crc {
    return Err(Error::Checksum);
  }

  //
  // Deserialized buffer
  //
  let data = match bincode::deserialize(&buf[..]) {
    Ok(buf) => buf,
    Err(e) => {
      return Err(Error::SerDe(e.to_string()));
    }
  };

  Ok(data)
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
