use super::String;

use core::{convert::TryFrom, fmt};
use serde::{
    de::{Error, Visitor},
    Deserialize, Deserializer, Serialize, Serializer,
};

impl<const L: usize> Serialize for String<L> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_str(self)
    }
}

impl<'de, const L: usize> Deserialize<'de> for String<L> {
    fn deserialize<D>(deserializer: D) -> Result<String<L>, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct V;
        impl<'de> Visitor<'de> for V {
            type Value = &'de str;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("an ordinary string slice")
            }
        }

        let s = deserializer.deserialize_str(V)?;

        Self::try_from(s)
            .map_err(|e| D::Error::custom(format!("Expected a String with len {}, got {}", L, e)))
    }
}
