/// module containing nodes
pub mod nodes {
    #[derive(Debug, Clone)]
    /// structure for a specific node inside of a stack
    pub struct Node<T>
    where
        T: Clone,
    {
        next: Box<Option<Node<T>>>,
        prev: Box<Option<Node<T>>>,
        val: T,
    }

    impl<T: std::clone::Clone> Node<T> {
        /// create a new node
        ///
        /// # Arguments
        ///
        /// * `next`: the next node
        /// * `prev`: the previews node
        /// * `val`: the value of the node
        ///
        /// # Returns
        ///
        /// this function returns a new Node built with the specified parameters
        pub fn new(next: Option<Node<T>>, prev: Option<Node<T>>, val: T) -> Node<T> {
            Node {
                next: Box::new(next),
                prev: Box::new(prev),
                val,
            }
        }

        /// get the previous Node
        ///
        /// # Returns
        ///
        /// the previous Node
        pub fn get_prev(&self) -> Option<Node<T>> {
            *self.prev.clone()
        }

        /// set the previous Node
        ///
        /// # Arguments
        ///
        /// * `new_prev`: the new previous Node
        pub fn set_prev(&mut self, new_prev: Option<Node<T>>) {
            self.prev = Box::new(new_prev);
        }

        /// get the next Node
        ///
        /// # Returns
        ///
        /// the next Node
        pub fn get_next(&self) -> Option<Node<T>> {
            *self.next.clone()
        }

        /// set the next Node
        ///
        /// # Arguments
        ///
        /// * `new_next`: the new next Node
        pub fn set_next(&mut self, new_next: Option<Node<T>>) {
            self.next = Box::new(new_next);
        }

        /// get the current value
        ///
        /// # Returns
        ///
        /// the current value
        pub fn get_val(&self) -> T {
            self.val.clone()
        }

        /// set the value
        ///
        /// # Arguments
        ///
        /// * `new_val`: the new value
        pub fn set_val(&mut self, new_val: T) {
            self.val = new_val;
        }
    }
}

/// module containing stacks
pub mod stacks {
    use crate::stack::nodes::Node;

    #[derive(Debug, Clone)]
    /// structure representing a stack
    ///
    /// A stack is handled as a collection of nodes.
    /// we specify the top node who himself specifies the previous node and so on.
    pub struct Stack<T: std::clone::Clone> {
        top: Option<Node<T>>,
        len: usize,
    }

    impl<T: std::clone::Clone> Stack<T> {
        /// create a new stack
        ///
        /// # Arguments
        ///
        /// * `val`: the value to use
        ///
        /// # Returns
        ///
        /// this function returns a new stack with one node that has the value of `val`
        pub fn new() -> Stack<T> {
            Stack { top: None, len: 0 }
        }

        /// push a new element to the stack
        ///
        /// # Arguments
        ///
        /// `val`: the value for the new element to have
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        ///
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(4);
        /// assert_eq!(stack.pop(), Some(4));
        /// ```
        pub fn push(&mut self, val: T) {
            let new_top: Node<T> = Node::new(None, self.top.clone(), val);
            self.top = Some(new_top);
            self.len += 1;
        }

        /// pop the top element of the stack.
        ///
        /// # Returns
        ///
        /// an `Option<T>` for the value that the element had
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        ///
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(4);
        /// stack.push(5);
        /// assert_eq!(stack.pop(), Some(5));
        /// assert_eq!(stack.pop(), Some(4));
        /// ```
        pub fn pop(&mut self) -> Option<T> {
            if self.len == 0 {
                return None;
            }
            match self.top.clone() {
                Some(top) => {
                    let retval = top.get_val().clone();
                    let new_top = match top.clone().get_prev() {
                        Some(mut n) => {
                            n.set_next(None);
                            Some(n)
                        }
                        None => None,
                    };
                    self.top = new_top;
                    self.len -= 1;
                    Some(retval)
                }
                None => None,
            }
        }

        /// return the length of the stack
        ///
        /// # Returns
        ///
        /// the length of the stack as a `usize`
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        ///
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(5);
        /// assert_eq!(stack.len(), 1);
        /// stack.pop();
        /// assert_eq!(stack.len(), 0);
        /// ```
        pub fn len(&self) -> usize {
            self.len
        }
    }

    #[derive(Debug, Clone)]
    /// a struct representing a queue which is a specific type of stack.
    /// You push to the top and pop from the bottom.
    pub struct Queue<T: Clone> {
        top: Option<Node<T>>,
        bottom: Option<Node<T>>,
        len: usize,
    }

    impl<T: Clone> Queue<T> {
        /// create a new instance of a Queue
        /// # Returns
        /// a new empty instance of a Queue
        pub fn new() -> Queue<T> {
            Queue {
                top: None,
                bottom: None,
                len: 0,
            }
        }

        /// push a new element to the Queue
        ///
        /// # Arguments
        /// 
        /// * `val`: the value to push
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Queue;
        ///
        /// let mut queue: Queue<i32> = Queue::new();
        /// queue.push(4);
        /// queue.push(5);
        /// assert_eq!(queue.pop(), Some(4));
        /// assert_eq!(queue.pop(), Some(5));
        /// ```
        pub fn push(&mut self, val: T) {
            let new_top: Node<T> = Node::new(None, self.top.clone(), val);
            self.top = Some(new_top.clone());
            if self.len == 0 {
                self.bottom = Some(new_top.clone());
            } else if self.len == 1 {
                self.bottom = Some(Node::new(self.top.clone(), None, self.bottom.clone().unwrap().get_val()))
            }
            self.len += 1;
        }

        /// pop the element at the bottom and return it's value
        ///
        /// # Returns
        /// the value of the element at the bottom
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Queue;
        ///
        /// let mut queue: Queue<i32> = Queue::new();
        /// queue.push(4);
        /// queue.push(5);
        /// assert_eq!(queue.pop(), Some(4));
        /// assert_eq!(queue.pop(), Some(5));
        /// ```
        pub fn pop(&mut self) -> Option<T> {
            if self.len == 0 {
                return None;
            }
            match self.bottom.clone() {
                Some(bottom) => {
                    let val = bottom.get_val().clone();
                    let new_bottom = match bottom.clone().get_next() {
                        Some(mut n) => {
                            n.set_prev(None);
                            Some(n)
                        }
                        None => None,
                    };
                    self.bottom = new_bottom;
                    self.len -= 1;
                    Some(val)
                }
                None => None,
            }
        }

        /// get the length of the current Queue
        ///
        /// # Returns
        /// the length of the current Queue as a `usize`
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Queue;
        ///
        /// let mut queue: Queue<i32> = Queue::new();
        /// assert_eq!(queue.len(), 0);
        /// queue.push(4);
        /// assert_eq!(queue.len(), 1);
        /// ```
        pub fn len(&self) -> usize {
            self.len
        }
    }   
}
