//! `stacking` is a module containing tools for building stacks in rust
//!
//! # Stack
//!
//! `stacking` comes with a [`Stack`] builtin.
//! It only supports `pop`, `push` and `len` as actions
//!
//! ## Examples
//!
//! ```rust
//! use stacking::stacks::Stack;
//!
//! let mut stack: Stack<i32> = Stack::new();
//! assert_eq!(stack.len(), 0);
//! stack.push(4);
//! assert_eq!(stack.len(), 1);
//! assert_eq!(stack.pop(), Some(4));
//! ```
//!
//! This simple example creates a stack, appends `4` to it and pops it off again.
//!
//! # Nodes
//! `stacking` also contains a [`Node`].
//! A [`Node`] can be used to build custom stacks or other data structures with specific orders.
//!
//! ## Examples
//!
//! ```rust
//! use stacking::nodes::Node;
//!
//! let mut n1: Node<i32> = Node::new(None, None, 15);
//! let mut n2: Node<i32> = Node::new(None, Some(n1.clone()), 14);
//! assert_eq!(n1.get_val(), 15);
//! assert_eq!(n2.get_val(), 14);
//! ```
//!
//! This will create two nodes referencing each other to create an order in which after `n2` comes `n1`.
//!
//! [`Node`]: crate::nodes::Node
//! [`Stack`]: crate::stacks::Stack

pub mod stack;
pub use stack::*;

#[cfg(test)]
mod tests {
    use crate::stacks::{Stack, Queue};
    #[test]
    fn stack_push_pop() {
        let mut stack: Stack<i32> = Stack::new();
        stack.push(4);
        assert_eq!(stack.pop(), Some(4));
    }

    #[test]
    fn empty_stack_pop() {
        let mut stack: Stack<i32> = Stack::new();
        assert_eq!(stack.pop(), None);
    }

    #[test]
    fn stack_len() {
        let mut stack: Stack<i32> = Stack::new();
        assert_eq!(stack.len(), 0);
        stack.push(4);
        assert_eq!(stack.len(), 1);
        let _var = stack.pop();
        assert_eq!(stack.len(), 0);
    }

    #[test]
    fn queue_push_pop() {
        let mut queue: Queue<i32> = Queue::new();
        queue.push(4);
        queue.push(5);
        assert_eq!(queue.pop(), Some(4));
        assert_eq!(queue.pop(), Some(5));
    }

    #[test]
    fn empty_queue_pop() {
        let mut queue: Queue<i32> = Queue::new();
        assert_eq!(queue.pop(), None);
    }

    #[test]
    fn queue_len() {
        let mut queue: Queue<i32> = Queue::new();
        assert_eq!(queue.len(), 0);
        queue.push(4);
        assert_eq!(queue.len(), 1);
        let _var = queue.pop();
        assert_eq!(queue.len(), 0);
    }
}
