/// module containing nodes
pub mod nodes {
    #[derive(Debug, Clone)]
    /// structure for a specific node inside of a stack
    pub struct Node<T>
    where
        T: Clone,
    {
        next: Box<Option<Node<T>>>,
        prev: Box<Option<Node<T>>>,
        val: T,
    }

    impl<T: std::clone::Clone> Node<T> {
        /// create a new node
        ///
        /// # Arguments
        ///
        /// * `next`: the next node
        /// * `prev`: the previews node
        /// * `val`: the value of the node
        ///
        /// # Returns
        ///
        /// this function returns a new Node built with the specified parameters
        pub fn new(next: Option<Node<T>>, prev: Option<Node<T>>, val: T) -> Node<T> {
            Node {
                next: Box::new(next),
                prev: Box::new(prev),
                val,
            }
        }

        /// get the previous Node
        ///
        /// # Returns
        ///
        /// the previous Node
        pub fn get_prev(&self) -> Option<Node<T>> {
            *self.prev.clone()
        }

        /// set the previous Node
        ///
        /// # Arguments
        ///
        /// * `new_prev`: the new previous Node
        pub fn set_prev(&mut self, new_prev: Option<Node<T>>) {
            self.prev = Box::new(new_prev);
        }

        /// get the next Node
        ///
        /// # Returns
        ///
        /// the next Node
        pub fn get_next(&self) -> Option<Node<T>> {
            *self.next.clone()
        }

        /// set the next Node
        ///
        /// # Arguments
        ///
        /// * `new_next`: the new next Node
        pub fn set_next(&mut self, new_next: Option<Node<T>>) {
            self.next = Box::new(new_next);
        }

        /// get the current value
        ///
        /// # Returns
        ///
        /// the current value
        pub fn get_val(&self) -> T {
            self.val.clone()
        }

        /// set the value
        ///
        /// # Arguments
        ///
        /// * `new_val`: the new value
        pub fn set_val(&mut self, new_val: T) {
            self.val = new_val;
        }
    }
}

/// module containing stacks
pub mod stacks {
    use crate::stack::nodes::Node;

    #[derive(Debug, Clone)]
    /// structure representing a stack
    ///
    /// A stack is handled as a collection of nodes.
    /// we specify the top node who himself specifies the previous node and so on.
    pub struct Stack<T: std::clone::Clone> {
        top: Option<Node<T>>,
        len: usize,
    }

    impl<T: std::clone::Clone> Stack<T> {
        /// create a new stack
        ///
        /// # Arguments
        ///
        /// * `val`: the value to use
        ///
        /// # Returns
        ///
        /// this function returns a new stack with one node that has the value of `val`
        pub fn new() -> Stack<T> {
            Stack { top: None, len: 0 }
        }

        /// push a new element to the stack
        ///
        /// # Arguments
        ///
        /// `val`: the value for the new element to have
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        ///
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(4);
        /// assert_eq!(stack.pop(), Some(4));
        /// ```
        pub fn push(&mut self, val: T) {
            let new_top: Node<T> = Node::new(None, self.top.clone(), val);
            self.top = Some(new_top);
            self.len += 1;
        }

        /// pop the top element of the stack.
        ///
        /// # Returns
        ///
        /// an `Option<T>` for the value that the element had
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        ///
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(4);
        /// stack.push(5);
        /// assert_eq!(stack.pop(), Some(5));
        /// assert_eq!(stack.pop(), Some(4));
        /// ```
        pub fn pop(&mut self) -> Option<T> {
            if self.len == 0 {
                return None;
            }
            match self.top.clone() {
                Some(top) => {
                    let retval = top.get_val().clone();
                    let new_top = match top.clone().get_prev() {
                        Some(mut n) => {
                            n.set_next(None);
                            Some(n)
                        }
                        None => None,
                    };
                    self.top = new_top;
                    self.len -= 1;
                    Some(retval)
                }
                None => None,
            }
        }

        /// return the length of the stack
        ///
        /// # Returns
        ///
        /// the length of the stack as a `usize`
        ///
        /// # Examples
        ///
        /// ```rust
        /// use stacking::stacks::Stack;
        /// 
        /// let mut stack: Stack<i32> = Stack::new();
        /// stack.push(5);
        /// assert_eq!(stack.len(), 1);
        /// stack.pop();
        /// assert_eq!(stack.len(), 0);
        /// ```
        pub fn len(&self) -> usize {
            self.len
        }
    }
}
